/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.contamination;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.util.Locatable;
import htsjdk.variant.variantcontext.VariantContext;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.math3.util.FastMath;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.BaseUtils;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import org.broadinstitute.hellbender.utils.pileup.ReadPileup;
import org.broadinstitute.hellbender.utils.tsv.DataLine;
import org.broadinstitute.hellbender.utils.tsv.TableColumnCollection;
import org.broadinstitute.hellbender.utils.tsv.TableReader;
import org.broadinstitute.hellbender.utils.tsv.TableWriter;

public class PileupSummary
implements Locatable {
    private final String contig;
    private final int position;
    private final int refCount;
    private final int altCount;
    private final int otherAltsCount;
    private final int totalCount;
    private final double alleleFrequency;

    public PileupSummary(String contig, int position, int refCount, int altCount, int otherAltsCount, double alleleFrequency) {
        this.contig = contig;
        this.position = position;
        this.altCount = altCount;
        this.refCount = refCount;
        this.otherAltsCount = otherAltsCount;
        this.totalCount = refCount + altCount + otherAltsCount;
        this.alleleFrequency = alleleFrequency;
    }

    public PileupSummary(VariantContext vc, ReadPileup pileup) {
        this.contig = vc.getContig();
        this.position = vc.getStart();
        this.alleleFrequency = vc.getAttributeAsDouble("AF", 0.0);
        byte altBase = vc.getAlternateAllele(0).getBases()[0];
        byte refBase = vc.getReference().getBases()[0];
        int[] baseCounts = pileup.getBaseCounts();
        this.altCount = baseCounts[BaseUtils.simpleBaseToBaseIndex(altBase)];
        this.refCount = baseCounts[BaseUtils.simpleBaseToBaseIndex(refBase)];
        this.totalCount = (int)MathUtils.sum(baseCounts);
        this.otherAltsCount = this.totalCount - this.altCount - this.refCount;
    }

    public String getContig() {
        return this.contig;
    }

    public int getStart() {
        return this.position;
    }

    public int getEnd() {
        return this.position;
    }

    public int getAltCount() {
        return this.altCount;
    }

    public int getRefCount() {
        return this.refCount;
    }

    public int getOtherAltCount() {
        return this.otherAltsCount;
    }

    public int getTotalCount() {
        return this.totalCount;
    }

    public double getAlleleFrequency() {
        return this.alleleFrequency;
    }

    public double getRefFrequency() {
        return 1.0 - this.alleleFrequency;
    }

    public double getAltFraction() {
        return this.totalCount == 0 ? 0.0 : (double)this.altCount / (double)this.totalCount;
    }

    public double getMinorAlleleFraction() {
        double altFraction = this.getAltFraction();
        return FastMath.min((double)altFraction, (double)(1.0 - altFraction));
    }

    public static void writeToFile(String sample, List<PileupSummary> records, File outputTable) {
        try (PileupSummaryTableWriter writer = new PileupSummaryTableWriter(IOUtils.fileToPath(outputTable));){
            writer.writeMetadata("SAMPLE", sample);
            writer.writeAllRecords(records);
        }
        catch (IOException e) {
            throw new UserException(String.format("Encountered an IO exception while writing to %s.", outputTable));
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void writeToFile(List<File> inputFiles, File output) {
        boolean headerWritten = false;
        String sample = "";
        try (PileupSummaryTableWriter writer = new PileupSummaryTableWriter(output.toPath());){
            for (File inputFile : inputFiles) {
                try {
                    PileupSummaryTableReader reader = new PileupSummaryTableReader(inputFile.toPath());
                    Throwable throwable = null;
                    try {
                        String thisSample;
                        if (!headerWritten) {
                            sample = reader.getMetadata().get("SAMPLE");
                            writer.writeMetadata("SAMPLE", sample);
                            headerWritten = true;
                        }
                        if (!(thisSample = reader.getMetadata().get("SAMPLE")).equals(sample)) {
                            throw new UserException.BadInput(String.format("Combining PileupSummaryTables from different samples is not supported. Got samples %s and %s", sample, thisSample));
                        }
                        List pileupSummaries = reader.toList();
                        writer.writeAllRecords(pileupSummaries);
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (reader == null) continue;
                        if (throwable != null) {
                            try {
                                reader.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                            continue;
                        }
                        reader.close();
                    }
                }
                catch (IOException e) {
                    throw new UserException(String.format("Encountered an IO exception while reading from %s.", inputFile));
                    return;
                }
            }
        }
        catch (IOException e) {
            throw new UserException(String.format("Encountered an IO exception while writing to %s.", output));
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ImmutablePair<String, List<PileupSummary>> readFromFile(File tableFile) {
        try (PileupSummaryTableReader reader = new PileupSummaryTableReader(IOUtils.fileToPath(tableFile));){
            List pileupSummaries = reader.toList();
            ImmutablePair immutablePair = ImmutablePair.of((Object)reader.getMetadata().get("SAMPLE"), pileupSummaries);
            return immutablePair;
        }
        catch (IOException e) {
            throw new UserException(String.format("Encountered an IO exception while reading from %s.", tableFile));
        }
    }

    private static enum PileupSummaryTableColumn {
        CONTIG("contig"),
        POSITION("position"),
        REF_COUNT("ref_count"),
        ALT_COUNT("alt_count"),
        OTHER_ALT_COUNT("other_alt_count"),
        ALT_ALLELE_FREQUENCY("allele_frequency");

        private final String columnName;
        public static final TableColumnCollection COLUMNS;

        private PileupSummaryTableColumn(String columnName) {
            this.columnName = Utils.nonNull(columnName);
        }

        public String toString() {
            return this.columnName;
        }

        static {
            COLUMNS = new TableColumnCollection((Object[])PileupSummaryTableColumn.values());
        }
    }

    private static class PileupSummaryTableReader
    extends TableReader<PileupSummary> {
        public PileupSummaryTableReader(Path path) throws IOException {
            super(path);
        }

        @Override
        protected PileupSummary createRecord(DataLine dataLine) {
            String contig = dataLine.get(PileupSummaryTableColumn.CONTIG);
            int position = dataLine.getInt(PileupSummaryTableColumn.POSITION);
            int refCount = dataLine.getInt(PileupSummaryTableColumn.REF_COUNT);
            int altCount = dataLine.getInt(PileupSummaryTableColumn.ALT_COUNT);
            int otherAltCount = dataLine.getInt(PileupSummaryTableColumn.OTHER_ALT_COUNT);
            double alleleFrequency = dataLine.getDouble(PileupSummaryTableColumn.ALT_ALLELE_FREQUENCY);
            return new PileupSummary(contig, position, refCount, altCount, otherAltCount, alleleFrequency);
        }
    }

    private static class PileupSummaryTableWriter
    extends TableWriter<PileupSummary> {
        private PileupSummaryTableWriter(Path output) throws IOException {
            super(output, PileupSummaryTableColumn.COLUMNS);
        }

        @Override
        protected void composeLine(PileupSummary record, DataLine dataLine) {
            dataLine.set(PileupSummaryTableColumn.CONTIG.toString(), record.getContig()).set(PileupSummaryTableColumn.POSITION.toString(), record.getStart()).set(PileupSummaryTableColumn.REF_COUNT.toString(), record.getRefCount()).set(PileupSummaryTableColumn.ALT_COUNT.toString(), record.getAltCount()).set(PileupSummaryTableColumn.OTHER_ALT_COUNT.toString(), record.getOtherAltCount()).set(PileupSummaryTableColumn.ALT_ALLELE_FREQUENCY.toString(), record.getAlleleFrequency());
        }
    }

    public static class PileupSummaryComparator
    implements Comparator<PileupSummary> {
        final SAMSequenceDictionary sequenceDictionary;
        final List<String> contigsInOrder;

        public PileupSummaryComparator(SAMSequenceDictionary sequenceDictionary) {
            this.sequenceDictionary = sequenceDictionary;
            this.contigsInOrder = sequenceDictionary.getSequences().stream().map(ssr -> ssr.getSequenceName()).collect(Collectors.toList());
        }

        @Override
        public int compare(PileupSummary ps1, PileupSummary ps2) {
            int contigIndex2;
            int contigIndex1 = this.contigsInOrder.indexOf(ps1.getContig());
            if (contigIndex1 != (contigIndex2 = this.contigsInOrder.indexOf(ps2.getContig()))) {
                return Integer.compare(contigIndex1, contigIndex2);
            }
            return Integer.compare(ps1.getStart(), ps2.getStart());
        }
    }
}

