/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.genotyper;

import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.GenotypeLikelihoods;
import java.util.Comparator;
import java.util.PriorityQueue;
import org.broadinstitute.hellbender.tools.walkers.genotyper.GenotypeAlleleCounts;
import org.broadinstitute.hellbender.tools.walkers.genotyper.GenotypeLikelihoodCalculators;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.genotyper.LikelihoodMatrix;

public class GenotypeLikelihoodCalculator {
    private final int[][] alleleFirstGenotypeOffsetByPloidy;
    private final GenotypeAlleleCounts[] genotypeAlleleCounts;
    final int genotypeCount;
    final int alleleCount;
    final int ploidy;
    private final PriorityQueue<Integer> alleleHeap;
    final double[][] readLikelihoodsByGenotypeIndex;
    private final int[] genotypeAllelesAndCounts;
    private int maximumDistinctAllelesInGenotype;
    private GenotypeAlleleCounts lastOverheadCounts;
    double[] readAlleleLikelihoodByAlleleCount = null;
    private int readCapacity = -1;
    private double[] readGenotypeLikelihoodComponents;

    public GenotypeLikelihoodCalculator(int ploidy, int alleleCount, int[][] alleleFirstGenotypeOffsetByPloidy, GenotypeAlleleCounts[][] genotypeTableByPloidy) {
        this.maximumDistinctAllelesInGenotype = Math.min(ploidy, alleleCount);
        this.alleleFirstGenotypeOffsetByPloidy = alleleFirstGenotypeOffsetByPloidy;
        this.genotypeAlleleCounts = genotypeTableByPloidy[ploidy];
        this.genotypeCount = this.alleleFirstGenotypeOffsetByPloidy[ploidy][alleleCount];
        this.alleleCount = alleleCount;
        this.ploidy = ploidy;
        this.alleleHeap = new PriorityQueue(ploidy, Comparator.naturalOrder().reversed());
        this.readLikelihoodsByGenotypeIndex = new double[this.genotypeCount][];
        this.genotypeAllelesAndCounts = new int[this.maximumDistinctAllelesInGenotype * 2];
    }

    public void ensureReadCapacity(int requestedCapacity) {
        Utils.validateArg(requestedCapacity >= 0, "capacity may not be negative");
        if (this.readCapacity == -1) {
            int minimumCapacity = Math.max(requestedCapacity, 10);
            this.readAlleleLikelihoodByAlleleCount = new double[minimumCapacity * this.alleleCount * (this.ploidy + 1)];
            for (int i = 0; i < this.genotypeCount; ++i) {
                this.readLikelihoodsByGenotypeIndex[i] = new double[minimumCapacity];
            }
            this.readGenotypeLikelihoodComponents = new double[this.ploidy * minimumCapacity];
            this.readCapacity = minimumCapacity;
        } else if (this.readCapacity < requestedCapacity) {
            int doubleCapacity = requestedCapacity << 1;
            this.readAlleleLikelihoodByAlleleCount = new double[doubleCapacity * this.alleleCount * (this.ploidy + 1)];
            for (int i = 0; i < this.genotypeCount; ++i) {
                this.readLikelihoodsByGenotypeIndex[i] = new double[doubleCapacity];
            }
            this.readGenotypeLikelihoodComponents = new double[this.maximumDistinctAllelesInGenotype * doubleCapacity];
            this.readCapacity = doubleCapacity;
        }
    }

    public int allelesToIndex(int ... alleleIndices) {
        if (this.ploidy == 0) {
            return 0;
        }
        this.alleleHeap.clear();
        for (int i = 0; i < alleleIndices.length; ++i) {
            this.alleleHeap.add(alleleIndices[i]);
        }
        return this.alleleHeapToIndex();
    }

    public int genotypeCount() {
        return this.genotypeCount;
    }

    public GenotypeAlleleCounts genotypeAlleleCountsAt(int index) {
        Utils.validateArg(index >= 0 && index < this.genotypeCount, () -> "invalid likelihood index: " + index + " >= " + this.genotypeCount + " (genotype count for nalleles = " + this.alleleCount + " and ploidy " + this.ploidy);
        if (index < 1000) {
            return this.genotypeAlleleCounts[index];
        }
        if (this.lastOverheadCounts == null || this.lastOverheadCounts.index() > index) {
            GenotypeAlleleCounts result = this.genotypeAlleleCounts[999].copy();
            result.increase(index - 1000 + 1);
            this.lastOverheadCounts = result;
            return result.copy();
        }
        this.lastOverheadCounts.increase(index - this.lastOverheadCounts.index());
        return this.lastOverheadCounts.copy();
    }

    public <EVIDENCE, A extends Allele> GenotypeLikelihoods genotypeLikelihoods(LikelihoodMatrix<EVIDENCE, A> likelihoods) {
        double[] readLikelihoodsByGenotypeIndex = this.getReadRawReadLikelihoodsByGenotypeIndex(likelihoods);
        return GenotypeLikelihoods.fromLog10Likelihoods((double[])readLikelihoodsByGenotypeIndex);
    }

    <EVIDENCE, A extends Allele> double[] getReadRawReadLikelihoodsByGenotypeIndex(LikelihoodMatrix<EVIDENCE, A> likelihoods) {
        Utils.nonNull(likelihoods);
        Utils.validateArg(likelihoods.numberOfAlleles() == this.alleleCount, "mismatch between allele list and alleleCount");
        int readCount = likelihoods.evidenceCount();
        this.ensureReadCapacity(readCount);
        double[] readLikelihoodComponentsByAlleleCount = this.readLikelihoodComponentsByAlleleCount(likelihoods);
        double[][] genotypeLikelihoodByRead = this.genotypeLikelihoodByRead(readLikelihoodComponentsByAlleleCount, readCount);
        return this.genotypeLikelihoods(genotypeLikelihoodByRead, readCount);
    }

    double[] genotypeLikelihoods(double[][] readLikelihoodsByGenotypeIndex, int readCount) {
        double[] result = new double[this.genotypeCount];
        double denominator = (double)readCount * MathUtils.log10(this.ploidy);
        for (int g = 0; g < this.genotypeCount; ++g) {
            result[g] = MathUtils.sum(readLikelihoodsByGenotypeIndex[g], 0, readCount) - denominator;
        }
        return result;
    }

    protected double[][] genotypeLikelihoodByRead(double[] readLikelihoodComponentsByAlleleCount, int readCount) {
        GenotypeAlleleCounts alleleCounts = this.genotypeAlleleCounts[0];
        for (int genotypeIndex = 0; genotypeIndex < this.genotypeCount; ++genotypeIndex) {
            double[] readLikelihoods = this.readLikelihoodsByGenotypeIndex[genotypeIndex];
            int componentCount = alleleCounts.distinctAlleleCount();
            switch (componentCount) {
                case 1: {
                    this.singleComponentGenotypeLikelihoodByRead(alleleCounts, readLikelihoods, readLikelihoodComponentsByAlleleCount, readCount);
                    break;
                }
                case 2: {
                    this.twoComponentGenotypeLikelihoodByRead(alleleCounts, readLikelihoods, readLikelihoodComponentsByAlleleCount, readCount);
                    break;
                }
                default: {
                    this.manyComponentGenotypeLikelihoodByRead(alleleCounts, readLikelihoods, readLikelihoodComponentsByAlleleCount, readCount);
                }
            }
            if (genotypeIndex >= this.genotypeCount - 1) continue;
            alleleCounts = this.nextGenotypeAlleleCounts(alleleCounts);
        }
        return this.readLikelihoodsByGenotypeIndex;
    }

    private GenotypeAlleleCounts nextGenotypeAlleleCounts(GenotypeAlleleCounts alleleCounts) {
        GenotypeAlleleCounts result;
        int index = alleleCounts.index();
        int cmp = index - 1000 + 1;
        if (cmp < 0) {
            result = this.genotypeAlleleCounts[index + 1];
        } else if (cmp == 0) {
            result = this.genotypeAlleleCounts[index].copy();
            result.increase();
        } else {
            alleleCounts.increase();
            result = alleleCounts;
        }
        return result;
    }

    private void manyComponentGenotypeLikelihoodByRead(GenotypeAlleleCounts genotypeAlleleCounts, double[] likelihoodByRead, double[] readLikelihoodComponentsByAlleleCount, int readCount) {
        genotypeAlleleCounts.copyAlleleCounts(this.genotypeAllelesAndCounts, 0);
        int componentCount = genotypeAlleleCounts.distinctAlleleCount();
        int alleleDataSize = (this.ploidy + 1) * readCount;
        int cc = 0;
        for (int c = 0; c < componentCount; ++c) {
            int alleleIndex = this.genotypeAllelesAndCounts[cc++];
            int alleleCount = this.genotypeAllelesAndCounts[cc++];
            int alleleDataOffset = alleleDataSize * alleleIndex + alleleCount * readCount;
            int r = 0;
            int readDataOffset = c;
            while (r < readCount) {
                this.readGenotypeLikelihoodComponents[readDataOffset] = readLikelihoodComponentsByAlleleCount[alleleDataOffset++];
                ++r;
                readDataOffset += this.maximumDistinctAllelesInGenotype;
            }
        }
        int r = 0;
        int readDataOffset = 0;
        while (r < readCount) {
            likelihoodByRead[r] = MathUtils.approximateLog10SumLog10(this.readGenotypeLikelihoodComponents, readDataOffset, readDataOffset + componentCount);
            ++r;
            readDataOffset += this.maximumDistinctAllelesInGenotype;
        }
    }

    void twoComponentGenotypeLikelihoodByRead(GenotypeAlleleCounts genotypeAlleleCounts, double[] likelihoodByRead, double[] readLikelihoodComponentsByAlleleCount, int readCount) {
        int allele0 = genotypeAlleleCounts.alleleIndexAt(0);
        int freq0 = genotypeAlleleCounts.alleleCountAt(0);
        int allele1 = genotypeAlleleCounts.alleleIndexAt(1);
        int freq1 = this.ploidy - freq0;
        int allele0LnLkOffset = readCount * ((this.ploidy + 1) * allele0 + freq0);
        int allele1LnLkOffset = readCount * ((this.ploidy + 1) * allele1 + freq1);
        for (int r = 0; r < readCount; ++r) {
            double lnLk0 = readLikelihoodComponentsByAlleleCount[allele0LnLkOffset++];
            double lnLk1 = readLikelihoodComponentsByAlleleCount[allele1LnLkOffset++];
            likelihoodByRead[r] = MathUtils.approximateLog10SumLog10(lnLk0, lnLk1);
        }
    }

    void singleComponentGenotypeLikelihoodByRead(GenotypeAlleleCounts genotypeAlleleCounts, double[] likelihoodByRead, double[] readLikelihoodComponentsByAlleleCount, int readCount) {
        int allele = genotypeAlleleCounts.alleleIndexAt(0);
        int offset = (allele * (this.ploidy + 1) + this.ploidy) * readCount;
        for (int r = 0; r < readCount; ++r) {
            likelihoodByRead[r] = readLikelihoodComponentsByAlleleCount[offset++];
        }
    }

    private <EVIDENCE, A extends Allele> double[] readLikelihoodComponentsByAlleleCount(LikelihoodMatrix<EVIDENCE, A> likelihoods) {
        int readCount = likelihoods.evidenceCount();
        int alleleDataSize = readCount * (this.ploidy + 1);
        int a = 0;
        int frequency1Offset = readCount;
        while (a < this.alleleCount) {
            likelihoods.copyAlleleLikelihoods(a, this.readAlleleLikelihoodByAlleleCount, frequency1Offset);
            int destinationOffset = frequency1Offset + readCount;
            for (int frequency = 2; frequency <= this.ploidy; ++frequency) {
                double log10frequency = MathUtils.log10(frequency);
                int sourceOffset = frequency1Offset;
                for (int r = 0; r < readCount; ++r) {
                    this.readAlleleLikelihoodByAlleleCount[destinationOffset++] = this.readAlleleLikelihoodByAlleleCount[sourceOffset++] + log10frequency;
                }
            }
            ++a;
            frequency1Offset += alleleDataSize;
        }
        return this.readAlleleLikelihoodByAlleleCount;
    }

    public int ploidy() {
        return this.ploidy;
    }

    public int alleleCount() {
        return this.alleleCount;
    }

    public int alleleCountsToIndex(int ... alleleCountArray) {
        Utils.nonNull(alleleCountArray, "the allele counts cannot be null");
        Utils.validateArg((alleleCountArray.length & 1) == 0, "the allele counts array cannot have odd length");
        this.alleleHeap.clear();
        for (int i = 0; i < alleleCountArray.length; i += 2) {
            int index = alleleCountArray[i];
            int count = alleleCountArray[i + 1];
            Utils.validateArg(count >= 0, "no allele count can be less than 0");
            for (int j = 0; j < count; ++j) {
                this.alleleHeap.add(index);
            }
        }
        return this.alleleHeapToIndex();
    }

    private int alleleHeapToIndex() {
        Utils.validateArg(this.alleleHeap.size() == this.ploidy, "the sum of allele counts must be equal to the ploidy of the calculator");
        Utils.validateArg(this.alleleHeap.peek() < this.alleleCount, () -> "invalid allele " + this.alleleHeap.peek() + " more than the maximum " + (this.alleleCount - 1));
        int result = 0;
        for (int p = this.ploidy; p > 0; --p) {
            int allele = (Integer)this.alleleHeap.remove();
            Utils.validateArg(allele >= 0, () -> "invalid allele " + allele + " must be equal or greater than 0 ");
            result += this.alleleFirstGenotypeOffsetByPloidy[p][allele];
        }
        return result;
    }

    public int[] genotypeIndexMap(int[] oldToNewAlleleIndexMap, GenotypeLikelihoodCalculators calculators) {
        Utils.nonNull(oldToNewAlleleIndexMap);
        int resultAlleleCount = oldToNewAlleleIndexMap.length;
        Utils.validateArg(resultAlleleCount <= this.alleleCount, () -> "this calculator does not have enough capacity for handling " + resultAlleleCount + " alleles ");
        int resultLength = resultAlleleCount == this.alleleCount ? this.genotypeCount : calculators.genotypeCount(this.ploidy, resultAlleleCount);
        int[] result = new int[resultLength];
        int[] sortedAlleleCounts = new int[Math.max(this.ploidy, this.alleleCount) << 1];
        this.alleleHeap.clear();
        GenotypeAlleleCounts alleleCounts = this.genotypeAlleleCounts[0];
        for (int i = 0; i < resultLength; ++i) {
            this.genotypeIndexMapPerGenotypeIndex(i, alleleCounts, oldToNewAlleleIndexMap, result, sortedAlleleCounts);
            if (i >= resultLength - 1) continue;
            alleleCounts = this.nextGenotypeAlleleCounts(alleleCounts);
        }
        return result;
    }

    private void genotypeIndexMapPerGenotypeIndex(int newGenotypeIndex, GenotypeAlleleCounts alleleCounts, int[] oldToNewAlleleIndexMap, int[] destination, int[] sortedAlleleCountsBuffer) {
        int genotypeIndex;
        int distinctAlleleCount = alleleCounts.distinctAlleleCount();
        alleleCounts.copyAlleleCounts(sortedAlleleCountsBuffer, 0);
        int jj = 0;
        for (int j = 0; j < distinctAlleleCount; ++j) {
            int oldIndex = sortedAlleleCountsBuffer[jj++];
            int repeats = sortedAlleleCountsBuffer[jj++];
            int newIndex = oldToNewAlleleIndexMap[oldIndex];
            if (newIndex < 0 || newIndex >= this.alleleCount) {
                throw new IllegalArgumentException("found invalid new allele index (" + newIndex + ") for old index (" + oldIndex + ")");
            }
            for (int k = 0; k < repeats; ++k) {
                this.alleleHeap.add(newIndex);
            }
        }
        destination[newGenotypeIndex] = genotypeIndex = this.alleleHeapToIndex();
    }
}

