/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.genotyper;

import java.util.Arrays;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.tools.walkers.genotyper.GenotypeAlleleCounts;
import org.broadinstitute.hellbender.tools.walkers.genotyper.GenotypeLikelihoodCalculator;
import org.broadinstitute.hellbender.tools.walkers.genotyper.GenotypeLikelihoodCalculatorDRAGEN;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Utils;

public final class GenotypeLikelihoodCalculators {
    private static final Logger logger = LogManager.getLogger(GenotypeLikelihoodCalculators.class);
    private int maximumPloidy = 2;
    public static final int MAXIMUM_STRONG_REF_GENOTYPE_PER_PLOIDY = 1000;
    static final int GENOTYPE_COUNT_OVERFLOW = -1;
    private int maximumAllele = 1;
    private int[][] alleleFirstGenotypeOffsetByPloidy = GenotypeLikelihoodCalculators.buildAlleleFirstGenotypeOffsetTable(this.maximumPloidy, this.maximumAllele);
    private GenotypeAlleleCounts[][] genotypeTableByPloidy = GenotypeLikelihoodCalculators.buildGenotypeAlleleCountsTable(this.maximumPloidy, this.maximumAllele, this.alleleFirstGenotypeOffsetByPloidy);

    private static int[][] buildAlleleFirstGenotypeOffsetTable(int maximumPloidy, int maximumAllele) {
        GenotypeLikelihoodCalculators.checkPloidyAndMaximumAllele(maximumPloidy, maximumAllele);
        int rowCount = maximumPloidy + 1;
        int colCount = maximumAllele + 1;
        int[][] result = new int[rowCount][colCount];
        Arrays.fill(result[0], 1, colCount, 1);
        for (int ploidy = 1; ploidy < rowCount; ++ploidy) {
            for (int allele = 1; allele < colCount; ++allele) {
                result[ploidy][allele] = result[ploidy][allele - 1] + result[ploidy - 1][allele];
                if (result[ploidy][allele] >= result[ploidy][allele - 1]) continue;
                result[ploidy][allele] = -1;
            }
        }
        return result;
    }

    private static GenotypeAlleleCounts[][] buildGenotypeAlleleCountsTable(int maximumPloidy, int maximumAllele, int[][] offsetTable) {
        GenotypeLikelihoodCalculators.checkPloidyAndMaximumAllele(maximumPloidy, maximumAllele);
        GenotypeLikelihoodCalculators.checkOffsetTableCapacity(offsetTable, maximumPloidy, maximumAllele);
        int rowCount = maximumPloidy + 1;
        GenotypeAlleleCounts[][] result = new GenotypeAlleleCounts[rowCount][];
        for (int ploidy = 0; ploidy <= maximumPloidy; ++ploidy) {
            result[ploidy] = GenotypeLikelihoodCalculators.buildGenotypeAlleleCountsArray(ploidy, maximumAllele, offsetTable);
        }
        return result;
    }

    private static GenotypeAlleleCounts[] buildGenotypeAlleleCountsArray(int ploidy, int alleleCount, int[][] genotypeOffsetTable) {
        Utils.validateArg(ploidy >= 0, () -> "the requested ploidy cannot be negative: " + ploidy);
        Utils.validateArg(alleleCount >= 0, () -> "the requested maximum allele cannot be negative: " + alleleCount);
        int length = genotypeOffsetTable[ploidy][alleleCount];
        int strongRefLength = length == -1 ? 1000 : Math.min(length, 1000);
        GenotypeAlleleCounts[] result = new GenotypeAlleleCounts[strongRefLength];
        result[0] = GenotypeAlleleCounts.first(ploidy);
        for (int genotypeIndex = 1; genotypeIndex < strongRefLength; ++genotypeIndex) {
            result[genotypeIndex] = result[genotypeIndex - 1].next();
        }
        return result;
    }

    public synchronized GenotypeLikelihoodCalculator getInstance(int ploidy, int alleleCount) {
        this.calculateGenotypeCountsUsingTablesAndValidate(ploidy, alleleCount);
        return new GenotypeLikelihoodCalculator(ploidy, alleleCount, this.alleleFirstGenotypeOffsetByPloidy, this.genotypeTableByPloidy);
    }

    private synchronized void calculateGenotypeCountsUsingTablesAndValidate(int ploidy, int alleleCount) {
        GenotypeLikelihoodCalculators.checkPloidyAndMaximumAllele(ploidy, alleleCount);
        if (this.calculateGenotypeCountUsingTables(ploidy, alleleCount) == -1) {
            double largeGenotypeCount = Math.pow(10.0, MathUtils.log10BinomialCoefficient(ploidy + alleleCount - 1, alleleCount - 1));
            throw new IllegalArgumentException(String.format("the number of genotypes is too large for ploidy %d and allele %d: approx. %.0f", ploidy, alleleCount, largeGenotypeCount));
        }
    }

    public synchronized GenotypeLikelihoodCalculatorDRAGEN getInstanceDRAGEN(int ploidy, int alleleCount) {
        Utils.validate(ploidy == 2, "DRAGEN genotyping mode currently only supports diploid samples");
        this.calculateGenotypeCountsUsingTablesAndValidate(ploidy, alleleCount);
        return new GenotypeLikelihoodCalculatorDRAGEN(ploidy, alleleCount, this.alleleFirstGenotypeOffsetByPloidy, this.genotypeTableByPloidy);
    }

    private synchronized void ensureCapacity(int requestedMaximumAllele, int requestedMaximumPloidy) {
        boolean needsToExpandPloidyCapacity;
        boolean needsToExpandAlleleCapacity = requestedMaximumAllele > this.maximumAllele;
        boolean bl = needsToExpandPloidyCapacity = requestedMaximumPloidy > this.maximumPloidy;
        if (!needsToExpandAlleleCapacity && !needsToExpandPloidyCapacity) {
            return;
        }
        int newMaximumPloidy = Math.max(this.maximumPloidy, requestedMaximumPloidy);
        int newMaximumAllele = Math.max(this.maximumAllele, requestedMaximumAllele);
        logger.debug("Expanding capacity ploidy:" + this.maximumPloidy + "->" + newMaximumPloidy + " allele:" + this.maximumAllele + "->" + newMaximumAllele);
        this.alleleFirstGenotypeOffsetByPloidy = GenotypeLikelihoodCalculators.buildAlleleFirstGenotypeOffsetTable(newMaximumPloidy, newMaximumAllele);
        this.genotypeTableByPloidy = GenotypeLikelihoodCalculators.buildGenotypeAlleleCountsTable(newMaximumPloidy, newMaximumAllele, this.alleleFirstGenotypeOffsetByPloidy);
        if (needsToExpandAlleleCapacity) {
            this.maximumAllele = requestedMaximumAllele;
        }
        if (needsToExpandPloidyCapacity) {
            this.maximumPloidy = requestedMaximumPloidy;
        }
    }

    private static void checkPloidyAndMaximumAllele(int ploidy, int maximumAllele) {
        Utils.validateArg(ploidy >= 0, () -> "the ploidy provided cannot be negative: " + ploidy);
        Utils.validateArg(maximumAllele >= 0, () -> "the maximum allele index provided cannot be negative: " + maximumAllele);
    }

    private static void checkOffsetTableCapacity(int[][] offsetTable, int maximumPloidy, int maximumAllele) {
        Utils.nonNull(offsetTable, "the allele first genotype offset table provided cannot be null");
        Utils.validateArg(offsetTable.length > maximumPloidy, () -> "the allele first genotype offset table provided does not have enough capacity for requested maximum ploidy: " + maximumPloidy);
        Utils.validateArg(offsetTable[0].length >= maximumAllele, () -> "the allele first genotype offset table provided does not have enough capacity for requested maximum allele index: " + maximumAllele);
    }

    public int genotypeCount(int ploidy, int alleleCount) {
        int result = this.calculateGenotypeCountUsingTables(ploidy, alleleCount);
        if (result == -1) {
            double largeGenotypeCount = Math.pow(10.0, MathUtils.log10BinomialCoefficient(ploidy + alleleCount - 1, alleleCount - 1));
            throw new IllegalArgumentException(String.format("the number of genotypes is too large for ploidy %d and allele %d: approx. %.0f", ploidy, alleleCount, largeGenotypeCount));
        }
        return result;
    }

    public static int computeMaxAcceptableAlleleCount(int ploidy, int maxGenotypeCount) {
        int upper;
        GenotypeLikelihoodCalculators.checkPloidyAndMaximumAllele(ploidy, ploidy);
        if (ploidy == 1) {
            return maxGenotypeCount;
        }
        double log10MaxGenotypeCount = Math.log10(maxGenotypeCount);
        double x = Math.pow(10.0, (MathUtils.log10Factorial(ploidy) + log10MaxGenotypeCount) / (double)ploidy);
        int lower = (int)Math.floor(x) - ploidy - 1;
        for (int a = upper = (int)Math.ceil(x); a >= lower; --a) {
            double log10GTCnt = MathUtils.log10BinomialCoefficient(ploidy + a - 1, a - 1);
            if (!(log10MaxGenotypeCount >= log10GTCnt)) continue;
            return a;
        }
        throw new GATKException("Code should never reach here.");
    }

    private synchronized int calculateGenotypeCountUsingTables(int ploidy, int alleleCount) {
        GenotypeLikelihoodCalculators.checkPloidyAndMaximumAllele(ploidy, alleleCount);
        this.ensureCapacity(alleleCount, ploidy);
        return this.alleleFirstGenotypeOffsetByPloidy[ploidy][alleleCount];
    }
}

