/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.readorientation;

import com.google.common.primitives.Ints;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.metrics.Header;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.metrics.StringHeader;
import htsjdk.samtools.util.Histogram;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.ReferenceContext;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.tools.walkers.readorientation.AltSiteRecord;
import org.broadinstitute.hellbender.tools.walkers.readorientation.CollectF1R2CountsArgumentCollection;
import org.broadinstitute.hellbender.tools.walkers.readorientation.DepthOneHistograms;
import org.broadinstitute.hellbender.tools.walkers.readorientation.F1R2FilterConstants;
import org.broadinstitute.hellbender.tools.walkers.readorientation.F1R2FilterUtils;
import org.broadinstitute.hellbender.tools.walkers.readorientation.ReadOrientation;
import org.broadinstitute.hellbender.utils.BaseUtils;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Nucleotide;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import org.broadinstitute.hellbender.utils.pileup.ReadPileup;
import org.broadinstitute.hellbender.utils.read.ReadUtils;

public class F1R2CountsCollector {
    public static final String ALT_TABLE_EXTENSION = ".alt_table";
    public static final String ALT_HIST_EXTENSION = ".alt_histogram";
    public static final String REF_HIST_EXTENSION = ".ref_histogram";
    private static final Logger logger = LogManager.getLogger(F1R2CountsCollector.class);
    private final Set<String> samples;
    private final SAMFileHeader header;
    private final CollectF1R2CountsArgumentCollection CF1R2Args;
    private final Map<String, Map<String, Histogram<Integer>>> refSiteHistograms;
    private Map<String, DepthOneHistograms> depthOneAltHistograms;
    private Map<String, AltSiteRecord.AltSiteRecordTableWriter> altTableWriters;
    private final File outputTarGzFile;
    private final File tmpDir = IOUtils.createTempDir("untarred");

    public F1R2CountsCollector(CollectF1R2CountsArgumentCollection CF1R2Args, SAMFileHeader header, File outputTarGzFile, Collection<String> samples) {
        this.CF1R2Args = CF1R2Args;
        this.samples = samples.size() == 1 ? Collections.singleton(samples.iterator().next()) : new HashSet<String>(samples);
        this.header = header;
        this.outputTarGzFile = outputTarGzFile;
        this.refSiteHistograms = new HashMap<String, Map<String, Histogram<Integer>>>(samples.size());
        this.depthOneAltHistograms = new HashMap<String, DepthOneHistograms>(samples.size());
        this.altTableWriters = new HashMap<String, AltSiteRecord.AltSiteRecordTableWriter>(samples.size());
        for (String sample : samples) {
            HashMap refSitesHistogramsForThisSample = new HashMap(F1R2FilterConstants.ALL_KMERS.size());
            F1R2FilterConstants.ALL_KMERS.forEach(context -> {
                Histogram<Integer> emptyRefHistogram = F1R2FilterUtils.createRefHistogram(context, CF1R2Args.maxDepth);
                refSitesHistogramsForThisSample.put(context, emptyRefHistogram);
            });
            this.refSiteHistograms.put(sample, refSitesHistogramsForThisSample);
            this.depthOneAltHistograms.put(sample, new DepthOneHistograms(CF1R2Args.maxDepth));
            File altTableFile = new File(this.tmpDir, IOUtils.urlEncode(sample) + ALT_TABLE_EXTENSION);
            try {
                this.altTableWriters.put(sample, new AltSiteRecord.AltSiteRecordTableWriter(IOUtils.fileToPath(altTableFile), sample));
            }
            catch (IOException e) {
                throw new UserException(String.format("Encountered an IO exception creating a writer for %s", altTableFile), e);
            }
        }
    }

    public void process(ReadPileup pileup, ReferenceContext referenceContext) {
        int position = referenceContext.getInterval().getStart();
        String refContext = referenceContext.getKmerAround(position, 1);
        if (refContext == null) {
            return;
        }
        Nucleotide refBase = F1R2FilterUtils.getMiddleBase(refContext);
        if (refContext.contains("N") || refContext.length() != 3) {
            return;
        }
        if (refContext == null) {
            logger.warn(String.format("Skipped a site with null reference at interval %s, k-mer = %s", referenceContext.getInterval().toString(), refContext));
            return;
        }
        String onlySample = this.samples.size() == 1 ? this.samples.iterator().next() : null;
        Map<String, ReadPileup> splitPileup = this.samples.size() == 1 ? Collections.singletonMap(onlySample, pileup.makeFilteredPileup(pe -> Objects.equals(ReadUtils.getSampleName(pe.getRead(), this.header), onlySample) && pe.getQual() > this.CF1R2Args.minBaseQuality)) : pileup.splitBySample(this.header, null);
        for (Map.Entry<String, ReadPileup> entry : splitPileup.entrySet()) {
            boolean referenceSite;
            String sample = entry.getKey();
            ReadPileup samplePileup = this.samples.size() == 1 ? entry.getValue() : entry.getValue().makeFilteredPileup(pe -> pe.getQual() > this.CF1R2Args.minBaseQuality);
            int[] baseCounts = samplePileup.getBaseCounts();
            int depth = (int)MathUtils.sum(baseCounts);
            if (!this.isPileupGood(samplePileup)) {
                return;
            }
            int[] baseCountsCopy = Arrays.copyOf(baseCounts, baseCounts.length);
            baseCountsCopy[refBase.ordinal()] = -1;
            int altBaseIndex = MathUtils.maxElementIndex(baseCountsCopy);
            boolean bl = referenceSite = baseCounts[altBaseIndex] == 0;
            if (referenceSite) {
                this.refSiteHistograms.get(sample).get(refContext).increment((Comparable)Integer.valueOf(Math.min(depth, this.CF1R2Args.maxDepth)));
                return;
            }
            Nucleotide altBase = Nucleotide.decode(BaseUtils.baseIndexToSimpleBase(altBaseIndex));
            int refCount = baseCounts[refBase.ordinal()];
            int altCount = baseCounts[altBaseIndex];
            Utils.validate(altCount > 0, "We must have a nonzero alt read but got " + altCount);
            int refF1R2 = samplePileup.getNumberOfElements(pe -> Nucleotide.decode(pe.getBase()) == refBase && ReadUtils.isF1R2(pe.getRead()));
            int altF1R2 = samplePileup.getNumberOfElements(pe -> Nucleotide.decode(pe.getBase()) == altBase && ReadUtils.isF1R2(pe.getRead()));
            if (altCount == 1) {
                ReadOrientation type = altF1R2 == 1 ? ReadOrientation.F1R2 : ReadOrientation.F2R1;
                this.depthOneAltHistograms.get(sample).increment(refContext, altBase, type, depth);
                return;
            }
            try {
                this.altTableWriters.get(sample).writeRecord(new AltSiteRecord(refContext, refCount, altCount, refF1R2, altF1R2, altBase));
            }
            catch (IOException e) {
                throw new UserException("Encountered an IO Exception writing to the alt data table", e);
            }
        }
    }

    public void writeHistograms() {
        for (String sample : this.samples) {
            MetricsFile refMetricsFile = new MetricsFile();
            refMetricsFile.addHeader((Header)new StringHeader(sample));
            this.refSiteHistograms.get(sample).values().forEach(arg_0 -> ((MetricsFile)refMetricsFile).addHistogram(arg_0));
            refMetricsFile.write(new File(this.tmpDir, IOUtils.urlEncode(sample) + REF_HIST_EXTENSION));
            MetricsFile altMetricsFile = new MetricsFile();
            altMetricsFile.addHeader((Header)new StringHeader(sample));
            this.depthOneAltHistograms.get(sample).getHistograms().forEach(arg_0 -> ((MetricsFile)altMetricsFile).addHistogram(arg_0));
            altMetricsFile.write(new File(this.tmpDir, IOUtils.urlEncode(sample) + ALT_HIST_EXTENSION));
        }
    }

    public void closeAndArchiveFiles() {
        if (this.altTableWriters != null) {
            for (AltSiteRecord.AltSiteRecordTableWriter writer : this.altTableWriters.values()) {
                if (writer == null) continue;
                try {
                    writer.close();
                }
                catch (IOException e) {
                    throw new UserException("Encountered an IO exception while closing the alt table writer", e);
                }
            }
        }
        try {
            IOUtils.writeTarGz(this.outputTarGzFile.getAbsolutePath(), this.tmpDir.listFiles());
        }
        catch (IOException ex) {
            throw new UserException.CouldNotCreateOutputFile("Could not create tar.gz file", (Exception)ex);
        }
    }

    private boolean isPileupGood(ReadPileup pileup) {
        int[] baseCounts = pileup.getBaseCounts();
        int depth = (int)MathUtils.sum(baseCounts);
        List mappingQualities = Ints.asList((int[])pileup.getMappingQuals());
        int indelThreshold = depth / 100;
        boolean isIndel = pileup.getNumberOfElements(pe -> pe.isDeletion() || pe.isAfterInsertion() || pe.isBeforeDeletionStart()) > indelThreshold;
        isIndel = isIndel || depth == 0 && pileup.size() > 0;
        return depth > 0 && !isIndel && MathUtils.median(mappingQualities) >= (double)this.CF1R2Args.minMedianMapQual;
    }

    public static List<File> getRefHistogramsFromExtractedTar(File extractedTarDir) {
        return Arrays.stream(extractedTarDir.listFiles()).filter(file -> file.getAbsolutePath().endsWith(REF_HIST_EXTENSION)).collect(Collectors.toList());
    }

    public static List<File> getAltHistogramsFromExtractedTar(File extractedTarDir) {
        return Arrays.stream(extractedTarDir.listFiles()).filter(file -> file.getAbsolutePath().endsWith(ALT_HIST_EXTENSION)).collect(Collectors.toList());
    }

    public static List<File> getAltTablesFromExtractedTar(File extractedTarDir) {
        return Arrays.stream(extractedTarDir.listFiles()).filter(file -> file.getAbsolutePath().endsWith(ALT_TABLE_EXTENSION)).collect(Collectors.toList());
    }
}

