/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.rnaseq;

import htsjdk.samtools.AlignmentBlock;
import htsjdk.samtools.Cigar;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMTag;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.TextCigarCodec;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Locatable;
import htsjdk.samtools.util.OverlapDetector;
import htsjdk.tribble.annotation.Strand;
import htsjdk.tribble.gff.Gff3BaseData;
import htsjdk.tribble.gff.Gff3Feature;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.BetaFeature;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.cmdline.programgroups.CoverageAnalysisProgramGroup;
import org.broadinstitute.hellbender.engine.FeatureContext;
import org.broadinstitute.hellbender.engine.FeatureInput;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.engine.ReadWalker;
import org.broadinstitute.hellbender.engine.ReferenceContext;
import org.broadinstitute.hellbender.engine.filters.AlignmentAgreesWithHeaderReadFilter;
import org.broadinstitute.hellbender.engine.filters.MappingQualityReadFilter;
import org.broadinstitute.hellbender.engine.filters.ReadFilter;
import org.broadinstitute.hellbender.engine.filters.ReadFilterLibrary;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.IntervalUtils;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.tsv.DataLine;
import org.broadinstitute.hellbender.utils.tsv.TableColumnCollection;
import org.broadinstitute.hellbender.utils.tsv.TableReader;
import org.broadinstitute.hellbender.utils.tsv.TableWriter;

@CommandLineProgramProperties(summary="This tool evaluates gene expression from RNA-seq reads aligned to genome.  Features to evaluate expression over are defined in an input annotation file in gff3 fomat (https://github.com/The-Sequence-Ontology/Specifications/blob/master/gff3.md).  Output is a tsv listing sense and antisense expression for all grouping features. For unstranded features, fragments transcribed on the forward strand are counted as sense, and fragments transcribed on the reverse strand are counted as antisense.", oneLineSummary="Evaluate gene expression from RNA-seq reads aligned to genome.", programGroup=CoverageAnalysisProgramGroup.class)
@DocumentedFeature
@BetaFeature
public final class GeneExpressionEvaluation
extends ReadWalker {
    @Argument(doc="Output file for gene expression.", fullName="output", shortName="O")
    private File outputCountsFile = null;
    @Argument(doc="Gff3 file containing feature annotations", shortName="G", fullName="gff-file")
    private FeatureInput<Gff3Feature> gffFile;
    @Argument(doc="Feature types to group by", fullName="grouping-type")
    private Set<String> groupingType = new HashSet<String>(Collections.singleton("gene"));
    @Argument(doc="Feature overlap types", fullName="overlap-type")
    private Set<String> overlapType = new HashSet<String>(Collections.singleton("exon"));
    @Argument(doc="Whether to label features by ID or Name", fullName="feature-label-key")
    private FeatureLabelType featureLabelKey = FeatureLabelType.NAME;
    @Argument(doc="Which strands (forward or reverse) each read is expected to be on", fullName="read-strands")
    private ReadStrands readStrands = ReadStrands.FORWARD_REVERSE;
    @Argument(doc="How to distribute weight of alignments which overlap multiple features", fullName="multi-overlap-method")
    private MultiOverlapMethod multiOverlapMethod = MultiOverlapMethod.PROPORTIONAL;
    @Argument(doc="How to distribute weight of reads with multiple alignments", fullName="multi-map-method")
    private MultiMapMethod multiMapMethod = MultiMapMethod.IGNORE;
    @Argument(doc="Whether the rna is unspliced.  If spliced, alignments must be from an aligner run in a splice-aware mode.  If unspliced, alignments must be from an aligner run in a non-splicing mode.")
    private boolean unspliced = false;
    private final Map<Gff3BaseData, Coverage> featureCounts = new LinkedHashMap<Gff3BaseData, Coverage>();
    private final OverlapDetector<Pair<Gff3BaseData, Interval>> featureOverlapDetector = new OverlapDetector(0, 0);
    private String sampleName = null;
    final MappingQualityReadFilter mappingQualityFilter = new MappingQualityReadFilter();

    @Override
    public List<ReadFilter> getDefaultReadFilters() {
        ArrayList<ReadFilter> readFilters = new ArrayList<ReadFilter>();
        readFilters.add(ReadFilterLibrary.VALID_ALIGNMENT_START);
        readFilters.add(ReadFilterLibrary.VALID_ALIGNMENT_END);
        readFilters.add(new AlignmentAgreesWithHeaderReadFilter(this.getHeaderForReads()));
        readFilters.add(ReadFilterLibrary.HAS_MATCHING_BASES_AND_QUALS);
        readFilters.add(ReadFilterLibrary.READLENGTH_EQUALS_CIGARLENGTH);
        readFilters.add(ReadFilterLibrary.SEQ_IS_STORED);
        readFilters.add(ReadFilterLibrary.MAPPED);
        readFilters.add(ReadFilterLibrary.NON_ZERO_REFERENCE_LENGTH_ALIGNMENT);
        readFilters.add(ReadFilterLibrary.NOT_DUPLICATE);
        readFilters.add(this.mappingQualityFilter);
        return readFilters;
    }

    @Override
    public void onTraversalStart() {
        this.validateOutputFile(this.outputCountsFile);
        if (this.multiMapMethod == MultiMapMethod.EQUAL) {
            this.mappingQualityFilter.minMappingQualityScore = 0;
        }
        SAMSequenceDictionary dict = this.getBestAvailableSequenceDictionary();
        SAMFileHeader header = this.getHeaderForReads();
        for (SAMReadGroupRecord readGroupRecord : header.getReadGroups()) {
            if (this.sampleName == null) {
                this.sampleName = readGroupRecord.getSample();
                continue;
            }
            if (this.sampleName.equals(readGroupRecord.getSample())) continue;
            throw new GATKException("Cannot run GeneExpressionEvaluation on multi-sample bam.");
        }
        if (dict == null) {
            throw new GATKException("sequence dictionary must be specified (sequence-dictionary).");
        }
        this.logger.info("collecting list of features");
        List<SimpleInterval> allIntervals = this.hasUserSuppliedIntervals() ? this.getTraversalIntervals() : IntervalUtils.getAllIntervalsForReference(dict);
        for (SimpleInterval interval : allIntervals) {
            List<Gff3Feature> contigFeatures = this.features.getFeatures(this.gffFile, interval);
            this.logger.info("collecting features in " + interval.getContig() + ":" + interval.getStart() + "-" + interval.getEnd());
            for (Gff3Feature feature : contigFeatures) {
                if (!this.groupingType.contains(feature.getType())) continue;
                List<Interval> overlappingFeatures = feature.getDescendents().stream().filter(f -> this.overlapType.contains(f.getType())).map(f -> new Interval(f.getContig(), f.getStart(), f.getEnd())).collect(Collectors.toList());
                Gff3BaseData shrunkGroupingBaseData = this.shrinkBaseData(feature.getBaseData());
                this.addGroupingFeature(shrunkGroupingBaseData, overlappingFeatures);
            }
        }
        this.logger.info("Collecting read counts...");
    }

    private void validateOutputFile(File file) {
        if (file.exists() ? !Files.isWritable(file.toPath()) : !Files.isWritable(file.getAbsoluteFile().getParentFile().toPath())) {
            throw new UserException.CouldNotCreateOutputFile(file, " is not writable");
        }
    }

    private Gff3BaseData shrinkBaseData(Gff3BaseData baseData) {
        Map<String, List> shrunkAttributes = baseData.getAttributes().entrySet().stream().filter(e -> ((String)e.getKey()).equals(this.featureLabelKey.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        return new Gff3BaseData(baseData.getContig(), baseData.getSource(), baseData.getType(), baseData.getStart(), baseData.getEnd(), Double.valueOf(baseData.getScore()), baseData.getStrand(), baseData.getPhase(), shrunkAttributes);
    }

    private void addGroupingFeature(Gff3BaseData groupingBaseData, List<Interval> overlappingFeatures) {
        String geneLabel = this.featureLabelKey.getValue(groupingBaseData);
        if (geneLabel == null) {
            throw new UserException("no geneid field " + (Object)((Object)this.featureLabelKey) + " found in feature at " + groupingBaseData.getContig() + ":" + groupingBaseData.getStart() + "-" + groupingBaseData.getEnd());
        }
        this.featureCounts.put(groupingBaseData, new Coverage(0.0, 0.0));
        for (Interval overlappingFeature : overlappingFeatures) {
            this.featureOverlapDetector.addLhs((Object)Pair.of((Object)groupingBaseData, (Object)overlappingFeature), (Locatable)overlappingFeature);
        }
    }

    static boolean inGoodPair(GATKRead read, int minimumMappingQuality, ReadStrands readStrands) {
        boolean ret;
        boolean bl = ret = !read.mateIsUnmapped() && read.isProperlyPaired() && read.getContig().equals(read.getMateContig());
        if (ret) {
            boolean readsOnSameStrand = read.isReverseStrand() == read.mateIsReverseStrand();
            boolean bl2 = ret = readStrands.expectReadsOnSameStrand == readsOnSameStrand;
        }
        if (ret) {
            if (!read.hasAttribute(SAMTag.MQ.toString())) {
                throw new GATKException("Mate quality must be included.  Consider running FixMateInformation.");
            }
            ret = read.getAttributeAsInteger(SAMTag.MQ.toString()) >= minimumMappingQuality;
        }
        return ret;
    }

    static List<Interval> getAlignmentIntervals(GATKRead read, boolean unspliced, int minimumMappingQuality, ReadStrands readStrands) {
        if (!unspliced) {
            ArrayList<Interval> alignmentIntervals = new ArrayList<Interval>();
            List readAlignmentBlocks = SAMUtils.getAlignmentBlocks((Cigar)read.getCigar(), (int)read.getStart(), (String)"read cigar");
            for (AlignmentBlock block : readAlignmentBlocks) {
                alignmentIntervals.add(new Interval(read.getContig(), block.getReferenceStart(), block.getReferenceStart() + block.getLength() - 1));
            }
            if (GeneExpressionEvaluation.inGoodPair(read, minimumMappingQuality, readStrands)) {
                String mateCigarString = read.getAttributeAsString(SAMTag.MC.toString());
                if (mateCigarString == null) {
                    throw new GATKException("Mate cigar must be present if using spliced reads");
                }
                List mateAlignmentBlocks = SAMUtils.getAlignmentBlocks((Cigar)TextCigarCodec.decode((String)mateCigarString), (int)read.getMateStart(), (String)"mate cigar");
                for (AlignmentBlock block : mateAlignmentBlocks) {
                    Interval alignmentBlockInterval = new Interval(read.getMateContig(), block.getReferenceStart(), block.getReferenceStart() + block.getLength() - 1);
                    alignmentIntervals.add(alignmentBlockInterval);
                }
            }
            return GeneExpressionEvaluation.getMergedIntervals(alignmentIntervals);
        }
        if (read.isUnmapped()) {
            return Collections.emptyList();
        }
        boolean inGoodPair = GeneExpressionEvaluation.inGoodPair(read, minimumMappingQuality, readStrands);
        int start = inGoodPair ? Math.min(read.getStart(), read.getMateStart()) : read.getStart();
        int end = inGoodPair ? start + Math.abs(read.getFragmentLength()) - 1 : read.getEnd();
        return Collections.singletonList(new Interval(read.getContig(), start, end));
    }

    @Override
    public void apply(GATKRead read, ReferenceContext referenceContext, FeatureContext featureContext) {
        if (read.isFirstOfPair() || !GeneExpressionEvaluation.inGoodPair(read, this.mappingQualityFilter.minMappingQualityScore, this.readStrands)) {
            List<Interval> alignmentIntervals = GeneExpressionEvaluation.getAlignmentIntervals(read, this.unspliced, this.mappingQualityFilter.minMappingQualityScore, this.readStrands);
            Map<Gff3BaseData, Double> initialWeights = this.multiOverlapMethod.getWeights(alignmentIntervals, this.featureOverlapDetector);
            Map<Gff3BaseData, Double> finalWeights = this.multiMapMethod.getWeights(read.hasAttribute(SAMTag.NH.toString()) ? read.getAttributeAsInteger(SAMTag.NH.toString()) : 1, initialWeights);
            for (Map.Entry<Gff3BaseData, Double> weight : finalWeights.entrySet()) {
                Gff3BaseData feature = weight.getKey();
                boolean isSense = this.readStrands.isSense(read, feature);
                if (isSense) {
                    this.featureCounts.get(feature).addSenseCount(weight.getValue());
                    continue;
                }
                this.featureCounts.get(feature).addAntiSenseCount(weight.getValue());
            }
        }
    }

    @Override
    public Object onTraversalSuccess() {
        this.logger.info(String.format("Writing read counts to %s...", this.outputCountsFile.getAbsolutePath()));
        try (FragmentCountWriter writer = new FragmentCountWriter(this.outputCountsFile.toPath(), this.sampleName, this.featureLabelKey);){
            int i = 0;
            for (GATKPath gATKPath : this.readArguments.getReadPathSpecifiers()) {
                writer.writeMetadata("input_bam_" + i, gATKPath.toString());
                ++i;
            }
            writer.writeMetadata("annotation_file", this.gffFile.toString());
            for (Map.Entry entry : this.featureCounts.entrySet()) {
                Gff3BaseData feature = (Gff3BaseData)entry.getKey();
                Coverage coverage = (Coverage)entry.getValue();
                writer.writeRecord(new SingleStrandFeatureCoverage(feature, coverage.sense_count, true));
                if (feature.getStrand() == Strand.NONE) continue;
                writer.writeRecord(new SingleStrandFeatureCoverage(feature, coverage.antisense_count, false));
            }
        }
        catch (IOException ex) {
            throw new UserException(ex.getMessage());
        }
        this.logger.info(String.format("%s complete.", this.getClass().getSimpleName()));
        return null;
    }

    private static List<Interval> getMergedIntervals(List<Interval> intervals) {
        ArrayList<Interval> intervalsCopy = new ArrayList<Interval>(intervals);
        Collections.sort(intervalsCopy);
        IntervalList.IntervalMergerIterator mergeringIterator = new IntervalList.IntervalMergerIterator(intervalsCopy.iterator(), true, true, false);
        ArrayList<Interval> merged = new ArrayList<Interval>();
        while (mergeringIterator.hasNext()) {
            merged.add(mergeringIterator.next());
        }
        return merged;
    }

    static enum ReadStrands {
        FORWARD_FORWARD(true, true),
        FORWARD_REVERSE(true, false),
        REVERSE_FORWARD(false, true),
        REVERSE_REVERSE(false, false);

        final boolean r1TranscriptionStrand;
        final boolean r2TranscriptionStrand;
        final boolean expectReadsOnSameStrand;

        private ReadStrands(boolean r1TranscriptionStrand, boolean r2TranscriptionStrand) {
            this.r1TranscriptionStrand = r1TranscriptionStrand;
            this.r2TranscriptionStrand = r2TranscriptionStrand;
            this.expectReadsOnSameStrand = r1TranscriptionStrand == r2TranscriptionStrand;
        }

        boolean isSense(GATKRead read, Gff3BaseData feature) {
            Strand senseStrand = feature.getStrand() == Strand.NONE ? Strand.POSITIVE : feature.getStrand();
            boolean isTranscriptionStrand = read.isFirstOfPair() ? this.r1TranscriptionStrand : this.r2TranscriptionStrand;
            boolean trancriptionStrandIsReverseStrand = isTranscriptionStrand == read.isReverseStrand();
            Strand transcriptionStrand = trancriptionStrandIsReverseStrand ? Strand.NEGATIVE : Strand.POSITIVE;
            return transcriptionStrand == senseStrand;
        }
    }

    public static class SingleStrandFeatureCoverage {
        public final Gff3BaseData baseData;
        public final double count;
        public final boolean sense;

        SingleStrandFeatureCoverage(Gff3BaseData baseData, double count, boolean sense) {
            this.baseData = baseData;
            this.count = count;
            this.sense = sense;
        }
    }

    public static class Coverage {
        private double sense_count;
        private double antisense_count;

        public Coverage(double sense_count, double antisense_count) {
            this.sense_count = sense_count;
            this.antisense_count = antisense_count;
        }

        public void addSenseCount(double count) {
            this.sense_count += count;
        }

        public void addAntiSenseCount(double count) {
            this.antisense_count += count;
        }

        public double getSenseCount() {
            return this.sense_count;
        }

        public double getAntisenseCount() {
            return this.antisense_count;
        }
    }

    public static class FragmentCountReader
    extends TableReader<SingleStrandFeatureCoverage> {
        public FragmentCountReader(Path file) throws IOException {
            super(file);
        }

        @Override
        protected SingleStrandFeatureCoverage createRecord(DataLine dataLine) {
            return new SingleStrandFeatureCoverage(new Gff3BaseData(dataLine.get("contig"), ".", ".", dataLine.getInt("start"), dataLine.getInt("stop"), Double.valueOf(-1.0), Strand.decode((String)dataLine.get("strand")), -1, Collections.singletonMap("ID", Collections.singletonList(dataLine.get("gene_label")))), dataLine.getDouble(6), dataLine.get("sense_antisense").equals("sense"));
        }
    }

    public static class FragmentCountWriter
    extends TableWriter<SingleStrandFeatureCoverage> {
        final String name;
        final FeatureLabelType gene_label_key;

        public FragmentCountWriter(Path file, String name, FeatureLabelType gene_label_key) throws IOException {
            super(file, new TableColumnCollection("gene_label", "contig", "start", "stop", "strand", "sense_antisense", name + "_counts"));
            this.name = name;
            this.gene_label_key = gene_label_key;
        }

        @Override
        protected void composeLine(SingleStrandFeatureCoverage fragmentCount, DataLine dataLine) {
            String gene_label = this.gene_label_key.getValue(fragmentCount.baseData);
            dataLine.set("contig", fragmentCount.baseData.getContig()).set("start", fragmentCount.baseData.getStart()).set("stop", fragmentCount.baseData.getEnd()).set("strand", fragmentCount.baseData.getStrand().encode()).set("sense_antisense", fragmentCount.sense ? "sense" : "antisense").set(this.name != null ? this.name + "_counts" : "counts", fragmentCount.count, 2).set("gene_label", gene_label == null ? "" : gene_label);
        }
    }

    static enum MultiMapMethod {
        IGNORE{

            @Override
            protected Map<Gff3BaseData, Double> getWeightsForMethod(int nHits, Map<Gff3BaseData, Double> previousWeights) {
                if (nHits == 1) {
                    return previousWeights;
                }
                return Collections.emptyMap();
            }
        }
        ,
        EQUAL{

            @Override
            protected Map<Gff3BaseData, Double> getWeightsForMethod(int nHits, Map<Gff3BaseData, Double> previousWeights) {
                if (nHits == 1) {
                    return previousWeights;
                }
                HashMap<Gff3BaseData, Double> newWeights = new HashMap<Gff3BaseData, Double>(previousWeights.size());
                for (Map.Entry<Gff3BaseData, Double> entry : previousWeights.entrySet()) {
                    newWeights.put(entry.getKey(), entry.getValue() / (double)nHits);
                }
                return newWeights;
            }
        };


        Map<Gff3BaseData, Double> getWeights(int nHits, Map<Gff3BaseData, Double> previousWeights) {
            if (nHits < 1) {
                throw new GATKException("nHits = " + nHits + ", cannot be less than 1");
            }
            return this.getWeightsForMethod(nHits, previousWeights);
        }

        protected abstract Map<Gff3BaseData, Double> getWeightsForMethod(int var1, Map<Gff3BaseData, Double> var2);
    }

    static enum MultiOverlapMethod {
        EQUAL{

            @Override
            Map<Gff3BaseData, Double> getWeights(List<Interval> alignmentIntervals, OverlapDetector<Pair<Gff3BaseData, Interval>> featureOverlapDetector) {
                Set overlappingFeatures = alignmentIntervals.stream().flatMap(i -> featureOverlapDetector.getOverlaps((Locatable)i).stream().map(Pair::getLeft)).collect(Collectors.toSet());
                int nOverlappingFeatures = overlappingFeatures.size();
                LinkedHashMap<Gff3BaseData, Double> weights = new LinkedHashMap<Gff3BaseData, Double>();
                for (Gff3BaseData feature : overlappingFeatures) {
                    weights.put(feature, 1.0 / (double)nOverlappingFeatures);
                }
                return weights;
            }
        }
        ,
        PROPORTIONAL{

            @Override
            Map<Gff3BaseData, Double> getWeights(List<Interval> alignmentIntervals, OverlapDetector<Pair<Gff3BaseData, Interval>> featureOverlapDetector) {
                List mergedAlignmentIntervals = GeneExpressionEvaluation.getMergedIntervals(alignmentIntervals);
                int basesOnReference = mergedAlignmentIntervals.stream().map(Locatable::getLengthOnReference).reduce(0, Integer::sum);
                int totalCoveredBases = 0;
                double summedUnNormalizedWeights = 0.0;
                LinkedHashMap<Gff3BaseData, Double> weights = new LinkedHashMap<Gff3BaseData, Double>();
                for (Interval alignmentInterval : mergedAlignmentIntervals) {
                    Set overlaps = featureOverlapDetector.getOverlaps((Locatable)alignmentInterval);
                    LinkedHashMap<Gff3BaseData, List> overlappingIntervalsByFeature = new LinkedHashMap<Gff3BaseData, List>();
                    ArrayList<Object> allOverlappingIntervals = new ArrayList<Object>();
                    for (Pair overlap : overlaps) {
                        List overlappingIntervals = overlappingIntervalsByFeature.computeIfAbsent((Gff3BaseData)overlap.getLeft(), f -> new ArrayList());
                        overlappingIntervals.add(overlap.getRight());
                        allOverlappingIntervals.add(overlap.getRight());
                    }
                    List allOverlappingIntervalsMerged = GeneExpressionEvaluation.getMergedIntervals(allOverlappingIntervals);
                    totalCoveredBases += allOverlappingIntervalsMerged.stream().map(arg_0 -> ((Interval)alignmentInterval).getIntersectionLength(arg_0)).reduce(0, Integer::sum).intValue();
                    for (Map.Entry overlap : overlappingIntervalsByFeature.entrySet()) {
                        List mergedOverlapIntervals = GeneExpressionEvaluation.getMergedIntervals((List)overlap.getValue());
                        double weight = (double)mergedOverlapIntervals.stream().map(arg_0 -> ((Interval)alignmentInterval).getIntersectionLength(arg_0)).reduce(0, Integer::sum).intValue() / (double)basesOnReference;
                        weights.compute((Gff3BaseData)overlap.getKey(), (k, v) -> v == null ? weight : v + weight);
                        summedUnNormalizedWeights += weight;
                    }
                }
                double normalizationFactor = 1.0 / (summedUnNormalizedWeights += 1.0 - (double)totalCoveredBases / (double)basesOnReference);
                for (Gff3BaseData feature : weights.keySet()) {
                    weights.compute(feature, (k, v) -> v * normalizationFactor);
                }
                return weights;
            }
        };


        abstract Map<Gff3BaseData, Double> getWeights(List<Interval> var1, OverlapDetector<Pair<Gff3BaseData, Interval>> var2);
    }

    static enum FeatureLabelType {
        NAME("Name"){

            @Override
            String getValue(Gff3BaseData baseData) {
                return baseData.getName();
            }
        }
        ,
        ID("ID"){

            @Override
            String getValue(Gff3BaseData baseData) {
                return baseData.getId();
            }
        };

        String key;

        private FeatureLabelType(String key) {
            this.key = key;
        }

        String getKey() {
            return this.key;
        }

        abstract String getValue(Gff3BaseData var1);
    }
}

