/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.tools.walkers.variantutils;

import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLineCount;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.barclay.argparser.Advanced;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.broadinstitute.hellbender.engine.FeatureContext;
import org.broadinstitute.hellbender.engine.ReadsContext;
import org.broadinstitute.hellbender.engine.ReferenceContext;
import org.broadinstitute.hellbender.engine.VariantWalker;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.variant.GATKVariantContextUtils;
import org.broadinstitute.hellbender.utils.variant.VcfUtils;
import picard.cmdline.programgroups.VariantEvaluationProgramGroup;

@CommandLineProgramProperties(summary="Extract specified fields for each variant in a VCF file to a tab-delimited table, which may be easier to work with than a VCF", oneLineSummary="Extract fields from a VCF file to a tab-delimited table", programGroup=VariantEvaluationProgramGroup.class)
@DocumentedFeature
public final class VariantsToTable
extends VariantWalker {
    public static final String SPLIT_MULTI_ALLELIC_LONG_NAME = "split-multi-allelic";
    public static final String SPLIT_MULTI_ALLELIC_SHORT_NAME = "SMA";
    static final Logger logger = LogManager.getLogger(VariantsToTable.class);
    @Argument(fullName="output", shortName="O", doc="File to which the tab-delimited table is written")
    private String out = null;
    @Argument(fullName="fields", shortName="F", doc="The name of a standard VCF field or an INFO field to include in the output table", optional=true)
    protected List<String> fieldsToTake = new ArrayList<String>();
    @Argument(fullName="genotype-fields", shortName="GF", doc="The name of a genotype field to include in the output table", optional=true)
    private List<String> genotypeFieldsToTake = new ArrayList<String>();
    @Argument(shortName="ASF", doc="The name of an allele-specific INFO field to be split if present", optional=true)
    private List<String> asFieldsToTake = new ArrayList<String>();
    @Argument(shortName="ASGF", doc="The name of an allele-specific FORMAT field to be split if present", optional=true)
    private List<String> asGenotypeFieldsToTake = new ArrayList<String>();
    @Advanced
    @Argument(fullName="show-filtered", shortName="raw", doc="Include filtered records in the output", optional=true)
    private boolean showFiltered = false;
    @Argument(fullName="split-multi-allelic", shortName="SMA", doc="Split multi-allelic records into multiple lines", optional=true)
    protected boolean splitMultiAllelic = false;
    @Advanced
    @Argument(fullName="moltenize", shortName="moltenize", doc="Produce molten output", optional=true)
    private boolean moltenizeOutput = false;
    @Advanced
    @Argument(fullName="error-if-missing-data", shortName="EMD", doc="Fail on missing data", optional=true)
    public boolean errorIfMissingData = false;
    private static final String MISSING_DATA = "NA";
    private SortedSet<String> samples;
    private long nRecords = 0L;
    private PrintStream outputStream = null;
    private VCFHeader inputHeader;
    private final Map<String, Function<VariantContext, String>> getters = new LinkedHashMap<String, Function<VariantContext, String>>();

    public VariantsToTable() {
        this.getters.put("CHROM", vc -> vc.getContig());
        this.getters.put("POS", vc -> Integer.toString(vc.getStart()));
        this.getters.put("REF", vc -> vc.getReference().getDisplayString());
        this.getters.put("ALT", vc -> {
            StringBuilder x = new StringBuilder();
            int n = vc.getAlternateAlleles().size();
            if (n == 0) {
                return ".";
            }
            for (int i = 0; i < n; ++i) {
                if (i != 0) {
                    x.append(",");
                }
                x.append(vc.getAlternateAllele(i));
            }
            return x.toString();
        });
        this.getters.put("EVENTLENGTH", vc -> {
            int maxLength = 0;
            for (Allele a : vc.getAlternateAlleles()) {
                int length = a.length() - vc.getReference().length();
                if (Math.abs(length) <= Math.abs(maxLength)) continue;
                maxLength = length;
            }
            return Integer.toString(maxLength);
        });
        this.getters.put("QUAL", vc -> Double.toString(vc.getPhredScaledQual()));
        this.getters.put("TRANSITION", vc -> {
            if (vc.isSNP() && vc.isBiallelic()) {
                return GATKVariantContextUtils.isTransition(vc) ? "1" : "0";
            }
            return "-1";
        });
        this.getters.put("FILTER", vc -> vc.isNotFiltered() ? "PASS" : Utils.join(",", vc.getFilters()));
        this.getters.put("ID", vc -> vc.getID());
        this.getters.put("HET", vc -> Integer.toString(vc.getHetCount()));
        this.getters.put("HOM-REF", vc -> Integer.toString(vc.getHomRefCount()));
        this.getters.put("HOM-VAR", vc -> Integer.toString(vc.getHomVarCount()));
        this.getters.put("NO-CALL", vc -> Integer.toString(vc.getNoCallCount()));
        this.getters.put("TYPE", vc -> vc.getType().toString());
        this.getters.put("VAR", vc -> Integer.toString(vc.getHetCount() + vc.getHomVarCount()));
        this.getters.put("NSAMPLES", vc -> Integer.toString(vc.getNSamples()));
        this.getters.put("NCALLED", vc -> Integer.toString(vc.getNSamples() - vc.getNoCallCount()));
        this.getters.put("MULTI-ALLELIC", vc -> Boolean.toString(vc.getAlternateAlleles().size() > 1));
        this.getters.put("SAMPLE_NAME", vc -> vc.getGenotype(0).getSampleName());
    }

    @Override
    public void onTraversalStart() {
        this.inputHeader = this.getHeaderForVariants();
        this.outputStream = this.createPrintStream();
        if (this.genotypeFieldsToTake.isEmpty() && this.asGenotypeFieldsToTake.isEmpty()) {
            this.samples = Collections.emptySortedSet();
        } else {
            Map<String, VCFHeader> vcfHeaders = Collections.singletonMap(this.getDrivingVariantsFeatureInput().getName(), this.getHeaderForVariants());
            this.samples = VcfUtils.getSortedSampleSet(vcfHeaders, GATKVariantContextUtils.GenotypeMergeType.REQUIRE_UNIQUE);
            if (this.samples.isEmpty()) {
                this.genotypeFieldsToTake.clear();
                this.asGenotypeFieldsToTake.clear();
                logger.warn("There are no samples - the genotype fields will be ignored");
                if (this.fieldsToTake.isEmpty() && this.asFieldsToTake.isEmpty()) {
                    throw new UserException("There are no samples and no fields - no output will be produced");
                }
            }
        }
        if (this.asGenotypeFieldsToTake.isEmpty() && this.asFieldsToTake.isEmpty() && !this.splitMultiAllelic) {
            logger.warn("Allele-specific fields will only be split if splitting multi-allelic variants is specified (`--split-multi-allelic` or `-SMA`");
        }
        if (this.moltenizeOutput) {
            this.outputStream.println("RecordID\tSample\tVariable\tValue");
        } else {
            ArrayList<String> fields = new ArrayList<String>();
            fields.addAll(this.fieldsToTake);
            fields.addAll(this.asFieldsToTake);
            String header = Utils.join("\t", fields) + "\t" + this.createGenotypeHeader();
            this.outputStream.println(header);
        }
    }

    private PrintStream createPrintStream() {
        try {
            return this.out != null ? new PrintStream(this.out) : System.out;
        }
        catch (FileNotFoundException e) {
            throw new UserException.CouldNotCreateOutputFile(this.out, (Exception)e);
        }
    }

    @Override
    public void apply(VariantContext vc, ReadsContext readsContext, ReferenceContext ref, FeatureContext featureContext) {
        if (this.showFiltered || vc.isNotFiltered()) {
            ++this.nRecords;
            List<List<String>> records = this.extractFields(vc);
            if (this.moltenizeOutput) {
                records.forEach(record -> this.emitMoltenizedOutput((List<String>)record));
            } else {
                records.forEach(record -> this.outputStream.println(Utils.join("\t", record)));
            }
        }
    }

    private static boolean isWildCard(String s) {
        return s.endsWith("*");
    }

    private String createGenotypeHeader() {
        boolean firstEntry = true;
        ArrayList<String> allGenotypeFieldsToTake = new ArrayList<String>(this.genotypeFieldsToTake);
        allGenotypeFieldsToTake.addAll(this.asGenotypeFieldsToTake);
        StringBuilder sb = new StringBuilder();
        for (String sample : this.samples) {
            for (String gf : allGenotypeFieldsToTake) {
                if (firstEntry) {
                    firstEntry = false;
                } else {
                    sb.append("\t");
                }
                sb.append(sample.replace(" ", "_"));
                sb.append('.');
                sb.append(gf);
            }
        }
        return sb.toString();
    }

    private void emitMoltenizedOutput(List<String> record) {
        int index = 0;
        for (String field : this.fieldsToTake) {
            this.outputStream.println(String.format("%d\tsite\t%s\t%s", this.nRecords, field, record.get(index++)));
        }
        for (String sample : this.samples) {
            for (String gf : this.genotypeFieldsToTake) {
                this.outputStream.println(String.format("%d\t%s\t%s\t%s", this.nRecords, sample.replace(" ", "_"), gf, record.get(index++)));
            }
        }
    }

    protected List<List<String>> extractFields(VariantContext vc) {
        int numRecordsToProduce = this.splitMultiAllelic ? vc.getAlternateAlleles().size() : 1;
        ArrayList<List<String>> records = new ArrayList<List<String>>(numRecordsToProduce);
        for (int i = 0; i < numRecordsToProduce; ++i) {
            records.add(new ArrayList());
        }
        for (String field : this.fieldsToTake) {
            if (this.splitMultiAllelic && field.equals("ALT")) {
                VariantsToTable.addFieldValue(VariantsToTable.splitAltAlleles(vc), records);
                continue;
            }
            if (this.getters.containsKey(field)) {
                VariantsToTable.addFieldValue(this.getters.get(field).apply(vc), records);
                continue;
            }
            if (vc.hasAttribute(field)) {
                VariantsToTable.addFieldValue(vc.getAttribute(field, null), records);
                continue;
            }
            if (VariantsToTable.isWildCard(field)) {
                TreeSet<String> wildVals = new TreeSet<String>();
                for (Map.Entry elt : vc.getAttributes().entrySet()) {
                    if (!((String)elt.getKey()).startsWith(field.substring(0, field.length() - 1))) continue;
                    wildVals.add(elt.getValue().toString());
                }
                String val = wildVals.isEmpty() ? MISSING_DATA : Utils.join(",", wildVals);
                VariantsToTable.addFieldValue(val, records);
                continue;
            }
            VariantsToTable.handleMissingData(this.errorIfMissingData, field, records, vc);
        }
        for (String field : this.asFieldsToTake) {
            if (vc.hasAttribute(field)) {
                if (this.splitMultiAllelic) {
                    VariantsToTable.addAlleleSpecificFieldValue(Arrays.asList(vc.getAttributeAsString(field, ".").replace("[", "").replace("]", "").split(",")), records, this.inputHeader.getInfoHeaderLine(field).getCountType());
                    continue;
                }
                VariantsToTable.addFieldValue(vc.getAttributeAsString(field, ".").replace("[", "").replace("]", ""), records);
                continue;
            }
            VariantsToTable.handleMissingData(this.errorIfMissingData, field, records, vc);
        }
        if (!this.genotypeFieldsToTake.isEmpty() || !this.asGenotypeFieldsToTake.isEmpty()) {
            this.addGenotypeFieldsToRecords(vc, records, this.errorIfMissingData);
        }
        return records;
    }

    private void addGenotypeFieldsToRecords(VariantContext vc, List<List<String>> records, boolean errorIfMissingData) {
        for (String sample : this.samples) {
            for (String gf : this.genotypeFieldsToTake) {
                if (vc.hasGenotype(sample) && vc.getGenotype(sample).hasAnyAttribute(gf)) {
                    if ("GT".equals(gf)) {
                        VariantsToTable.addFieldValue(vc.getGenotype(sample).getGenotypeString(true), records);
                        continue;
                    }
                    if (vc.getGenotype(sample).getAnyAttribute(gf) != null) {
                        VariantsToTable.addFieldValue(vc.getGenotype(sample).getAnyAttribute(gf), records);
                        continue;
                    }
                    VariantsToTable.handleMissingData(errorIfMissingData, gf, records, vc);
                    continue;
                }
                VariantsToTable.handleMissingData(errorIfMissingData, gf, records, vc);
            }
            for (String field : this.asGenotypeFieldsToTake) {
                if (vc.hasGenotype(sample) && vc.getGenotype(sample).hasAnyAttribute(field)) {
                    if (this.splitMultiAllelic) {
                        if ("AD".equals(field)) {
                            ArrayList<String> altDepths = new ArrayList<String>();
                            int[] allDepths = vc.getGenotype(sample).getAD();
                            for (int i = 1; i < allDepths.length; ++i) {
                                altDepths.add(allDepths[0] + "," + allDepths[i]);
                            }
                            VariantsToTable.addFieldValue(altDepths, records);
                            continue;
                        }
                        VariantsToTable.addAlleleSpecificFieldValue(Utils.split(vc.getGenotype(sample).getExtendedAttribute(field).toString(), ','), records, this.inputHeader.getFormatHeaderLine(field).getCountType());
                        continue;
                    }
                    String value = vc.getGenotype(sample).getAnyAttribute(field).toString();
                    if (field.equals("AD")) {
                        VariantsToTable.addFieldValue(value.replace("[", "").replace("]", "").replaceAll("\\s", ""), records);
                        continue;
                    }
                    VariantsToTable.addFieldValue(value, records);
                    continue;
                }
                VariantsToTable.handleMissingData(errorIfMissingData, field, records, vc);
            }
        }
    }

    private static void handleMissingData(boolean errorIfMissingData, String field, List<List<String>> records, VariantContext vc) {
        if (errorIfMissingData) {
            throw new UserException(String.format("Missing field %s in vc %s at %s", field, vc.getSource(), vc));
        }
        VariantsToTable.addFieldValue(MISSING_DATA, records);
    }

    private static void addFieldValue(Object val, List<List<String>> result) {
        int numResultRecords = result.size();
        if (numResultRecords == 1) {
            result.get(0).add(VariantsToTable.prettyPrintObject(val));
        } else if (val instanceof List && ((List)val).size() == numResultRecords) {
            List list = (List)val;
            for (int i = 0; i < numResultRecords; ++i) {
                result.get(i).add(list.get(i).toString());
            }
        } else {
            String valStr = VariantsToTable.prettyPrintObject(val);
            for (List<String> record : result) {
                record.add(valStr);
            }
        }
    }

    private static void addAlleleSpecificFieldValue(Object val, List<List<String>> result, VCFHeaderLineCount alleleCount) {
        if (val instanceof List && alleleCount.equals((Object)VCFHeaderLineCount.R)) {
            List myList = (List)val;
            VariantsToTable.addFieldValue(new ArrayList(myList.subList(1, myList.size())), result);
        } else {
            VariantsToTable.addFieldValue(val, result);
        }
    }

    private static String prettyPrintObject(Object val) {
        if (val == null) {
            return "";
        }
        if (val instanceof List) {
            return VariantsToTable.prettyPrintObject(((List)val).toArray());
        }
        if (!val.getClass().isArray()) {
            return val.toString();
        }
        int length = Array.getLength(val);
        if (length == 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder(VariantsToTable.prettyPrintObject(Array.get(val, 0)));
        for (int i = 1; i < length; ++i) {
            sb.append(",");
            sb.append(VariantsToTable.prettyPrintObject(Array.get(val, i)));
        }
        return sb.toString();
    }

    private static Object splitAltAlleles(VariantContext vc) {
        int numAltAlleles = vc.getAlternateAlleles().size();
        if (numAltAlleles == 1) {
            return vc.getAlternateAllele(0);
        }
        return vc.getAlternateAlleles();
    }
}

