/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.util.Locatable;
import htsjdk.tribble.Feature;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.barclay.argparser.CommandLineException;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.GenomeLoc;
import org.broadinstitute.hellbender.utils.IntervalUtils;
import org.broadinstitute.hellbender.utils.MRUCachingSAMSequenceDictionary;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.read.GATKRead;

public final class GenomeLocParser {
    private static final Logger logger = LogManager.getLogger(GenomeLocParser.class);
    public static final String UNMAPPED_LOC_NAME = "unmapped";
    private final MRUCachingSAMSequenceDictionary contigInfo;
    private final ValidationLevel validationLevel;

    public GenomeLocParser(ReferenceSequenceFile refFile) {
        this(refFile.getSequenceDictionary());
    }

    public GenomeLocParser(SAMSequenceDictionary seqDict) {
        this(seqDict, ValidationLevel.STANDARD);
    }

    protected GenomeLocParser(SAMSequenceDictionary seqDict, ValidationLevel validationLevel) {
        Utils.nonNull(validationLevel, "validation level cannot be null");
        if (seqDict == null) {
            throw new CommandLineException("Failed to load reference dictionary");
        }
        this.validationLevel = validationLevel;
        this.contigInfo = new MRUCachingSAMSequenceDictionary(seqDict);
        if (logger.isDebugEnabled()) {
            logger.debug(String.format("Prepared reference sequence contig dictionary", new Object[0]));
            for (SAMSequenceRecord contig : seqDict.getSequences()) {
                logger.debug(String.format(" %s (%d bp)", contig.getSequenceName(), contig.getSequenceLength()));
            }
        }
    }

    public final boolean contigIsInDictionary(String contig) {
        return contig != null && this.contigInfo.hasContig(contig);
    }

    public final SAMSequenceRecord getContigInfo(String contig) {
        if (contig == null || !this.contigIsInDictionary(contig)) {
            throw new UserException.MalformedGenomeLoc(String.format("Contig %s given as location, but this contig isn't present in the Fasta sequence dictionary", contig));
        }
        return this.contigInfo.getSequence(contig);
    }

    public final int getContigIndex(String contig) {
        return this.getContigInfo(contig).getSequenceIndex();
    }

    protected int getContigIndexWithoutException(String contig) {
        if (contig == null || !this.contigInfo.hasContig(contig)) {
            return -1;
        }
        return this.contigInfo.getSequenceIndex(contig);
    }

    public final SAMSequenceDictionary getSequenceDictionary() {
        return this.contigInfo.getDictionary();
    }

    public GenomeLoc createGenomeLoc(String contig, int start, int stop) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop);
    }

    public GenomeLoc createGenomeLoc(String contig, int start, int stop, boolean mustBeOnReference) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop, mustBeOnReference);
    }

    public GenomeLoc createGenomeLoc(String contig, int index, int start, int stop) {
        return this.createGenomeLoc(contig, index, start, stop, false);
    }

    public GenomeLoc createGenomeLoc(String contig, int index, int start, int stop, boolean mustBeOnReference) {
        String interned = this.validateGenomeLoc(contig, index, start, stop, mustBeOnReference);
        return new GenomeLoc(interned, index, start, stop);
    }

    public GenomeLoc createGenomeLoc(String contig, int pos) {
        return this.createGenomeLoc(contig, this.getContigIndex(contig), pos, pos);
    }

    protected String validateGenomeLoc(String contig, int contigIndex, int start, int stop, boolean mustBeOnReference) {
        SAMSequenceRecord contigInfo;
        if (this.validationLevel == ValidationLevel.NONE) {
            return contig;
        }
        if (stop < start) {
            this.vglHelper(String.format("The stop position %d is less than start %d in contig %s", stop, start, contig));
        }
        if ((contigInfo = this.contigInfo.getSequence(contig)).getSequenceIndex() != contigIndex) {
            this.vglHelper(String.format("The contig index %d is bad, doesn't equal the contig index %d of the contig from a string %s", contigIndex, contigInfo.getSequenceIndex(), contig));
        }
        if (mustBeOnReference) {
            int contigSize;
            if (start < 1) {
                this.vglHelper(String.format("The start position %d is less than 1", start));
            }
            if (stop < 1) {
                this.vglHelper(String.format("The stop position %d is less than 1", stop));
            }
            if ((contigSize = contigInfo.getSequenceLength()) == 0) {
                logger.warn(String.format("The available sequence dictionary does not contain a sequence length for contig (%s). Skipping validation of the genome loc end coordinate (%d).", contig, stop));
            } else if (start > contigSize || stop > contigSize) {
                this.vglHelper(String.format("The genome loc coordinates %d-%d exceed the contig size (%d)", start, stop, contigSize));
            }
        }
        return contigInfo.getSequenceName();
    }

    public boolean isValidGenomeLoc(String contig, int start, int stop, boolean mustBeOnReference) {
        try {
            this.validateGenomeLoc(contig, this.getContigIndexWithoutException(contig), start, stop, mustBeOnReference);
            return true;
        }
        catch (GATKException | UserException.MalformedGenomeLoc e) {
            return false;
        }
    }

    public boolean isValidGenomeLoc(String contig, int start, int stop) {
        return this.isValidGenomeLoc(contig, start, stop, true);
    }

    private void vglHelper(String msg) {
        throw new UserException.MalformedGenomeLoc("Parameters to GenomeLocParser are incorrect:" + msg);
    }

    public GenomeLoc parseGenomeLoc(String rawStr) {
        String str = rawStr.trim();
        try {
            if (GenomeLocParser.isUnmappedGenomeLocString(str)) {
                return GenomeLoc.UNMAPPED;
            }
            List<SimpleInterval> allResolvedIntervals = IntervalUtils.getResolvedIntervals(str, this.contigInfo.getDictionary());
            SimpleInterval locatable = GenomeLocParser.getUnambiguousInterval(str, allResolvedIntervals);
            String contig = locatable.getContig();
            int start = locatable.getStart();
            int stop = locatable.getEnd();
            if (!this.contigIsInDictionary(contig)) {
                throw new UserException.MalformedGenomeLoc("Contig '" + contig + "' does not match any contig in the GATK sequence dictionary derived from the reference; are you sure you are using the correct reference fasta file?");
            }
            if (stop == Integer.MAX_VALUE) {
                stop = this.getContigInfo(contig).getSequenceLength();
            }
            return this.createGenomeLoc(contig, this.getContigIndex(contig), start, stop, true);
        }
        catch (UserException.MalformedGenomeLoc e) {
            throw e;
        }
        catch (IllegalArgumentException | UserException e) {
            throw new UserException.MalformedGenomeLoc("Failed to parse Genome Location string: " + str + ": " + e.getMessage(), e);
        }
    }

    static SimpleInterval getUnambiguousInterval(String intervalQueryString, List<SimpleInterval> allResolvedIntervals) {
        Utils.nonNull(intervalQueryString);
        Utils.nonNull(allResolvedIntervals);
        if (allResolvedIntervals.isEmpty()) {
            throw new UserException.MalformedGenomeLoc(String.format("Query interval \"%s\" is not valid for this input.", intervalQueryString));
        }
        if (allResolvedIntervals.size() > 1) {
            throw new UserException.MalformedGenomeLoc(String.format("The query interval \"%s\" is ambiguous and can be interpreted as a query against more than one contig: \"%s\". The ambiguity can be resolved by providing the interval in a BED file (using zero-based, half-open coordinates).", intervalQueryString, allResolvedIntervals.stream().map(f -> f.getContig()).collect(Collectors.joining(" or "))));
        }
        return allResolvedIntervals.get(0);
    }

    public static boolean isUnmappedGenomeLocString(String str) {
        return str != null && str.trim().equalsIgnoreCase(UNMAPPED_LOC_NAME);
    }

    public GenomeLoc createGenomeLoc(GATKRead read) {
        if (read.isUnmapped()) {
            return GenomeLoc.UNMAPPED;
        }
        int end = Math.max(read.getEnd(), read.getStart());
        return this.createGenomeLoc(read.getContig(), this.getSequenceDictionary().getSequenceIndex(read.getContig()), read.getStart(), end, false);
    }

    public GenomeLoc createGenomeLoc(Feature feature) {
        return this.createGenomeLoc(feature.getContig(), feature.getStart(), feature.getEnd());
    }

    public GenomeLoc createGenomeLoc(Locatable locatable) {
        Utils.nonNull(locatable, "the input locatable cannot be null");
        return this.createGenomeLoc(locatable.getContig(), locatable.getStart(), locatable.getEnd());
    }

    public GenomeLoc createOverEntireContig(String contigName) {
        SAMSequenceRecord contig = this.contigInfo.getSequence(contigName);
        return this.createGenomeLoc(contigName, contig.getSequenceIndex(), 1, contig.getSequenceLength(), true);
    }

    public GenomeLoc createGenomeLocAtStart(GenomeLoc loc, int maxBasePairs) {
        if (GenomeLoc.isUnmapped(loc)) {
            return null;
        }
        String contigName = loc.getContig();
        SAMSequenceRecord contig = this.contigInfo.getSequence(contigName);
        int contigIndex = contig.getSequenceIndex();
        int start = loc.getStart() - maxBasePairs;
        int stop = loc.getStart() - 1;
        if (start < 1) {
            start = 1;
        }
        if (stop < 1) {
            return null;
        }
        return this.createGenomeLoc(contigName, contigIndex, start, stop, true);
    }

    public GenomeLoc createPaddedGenomeLoc(GenomeLoc loc, int padding) {
        if (GenomeLoc.isUnmapped(loc) || padding == 0) {
            return loc;
        }
        return this.createGenomeLocOnContig(loc.getContig(), loc.getContigIndex(), loc.getStart() - padding, loc.getStop() + padding);
    }

    public GenomeLoc createGenomeLocAtStop(GenomeLoc loc, int maxBasePairs) {
        if (GenomeLoc.isUnmapped(loc)) {
            return null;
        }
        String contigName = loc.getContig();
        SAMSequenceRecord contig = this.contigInfo.getSequence(contigName);
        int contigIndex = contig.getSequenceIndex();
        int contigLength = contig.getSequenceLength();
        int start = loc.getStop() + 1;
        int stop = loc.getStop() + maxBasePairs;
        if (start > contigLength) {
            return null;
        }
        if (stop > contigLength) {
            stop = contigLength;
        }
        return this.createGenomeLoc(contigName, contigIndex, start, stop, true);
    }

    public GenomeLoc createGenomeLocOnContig(String contig, int start, int stop) {
        return this.createGenomeLocOnContig(contig, this.getContigIndex(contig), start, stop);
    }

    public GenomeLoc createGenomeLocOnContig(String contig, int contigIndex, int start, int stop) {
        int contigLength = this.contigInfo.getSequence(contigIndex).getSequenceLength();
        int boundedStart = Math.max(1, start);
        int boundedStop = Math.min(contigLength, stop);
        if (boundedStart > contigLength || boundedStop < 1) {
            return null;
        }
        return this.createGenomeLoc(contig, contigIndex, boundedStart, boundedStop);
    }

    public static enum ValidationLevel {
        STANDARD,
        NONE;

    }
}

