/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.tribble.AbstractFeatureReader;
import htsjdk.tribble.Tribble;
import htsjdk.tribble.index.Index;
import htsjdk.tribble.index.IndexFactory;
import htsjdk.tribble.util.ParsingUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.tools.IndexFeatureFile;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.IOUtils;

public final class IndexUtils {
    private static final Logger logger = LogManager.getLogger(IndexUtils.class);

    private IndexUtils() {
    }

    public static Index loadTribbleIndex(Path featureFile) {
        Utils.nonNull(featureFile);
        String indexFile = Tribble.indexFile((String)featureFile.toString());
        Path indexPath = IOUtils.getPath(indexFile);
        if (!Files.isReadable(indexPath)) {
            return null;
        }
        logger.debug("Loading Tribble index from disk for file " + featureFile);
        try {
            Index index = IndexFactory.loadIndex((String)indexFile);
            IndexUtils.checkIndexVersionAndModificationTime(featureFile, indexPath, index);
            return index;
        }
        catch (RuntimeException e) {
            return null;
        }
    }

    public static Index loadTabixIndex(Path featureFile) {
        Utils.nonNull(featureFile);
        try {
            String path = featureFile.toUri().toString();
            boolean isTabix = new AbstractFeatureReader.ComponentMethods().isTabix(path, null);
            if (!isTabix) {
                return null;
            }
            String indexPath = ParsingUtils.appendToPath((String)path, (String)".tbi");
            logger.debug("Loading tabix index from disk for file " + featureFile);
            Index index = IndexFactory.loadIndex((String)indexPath);
            Path indexFile = IOUtils.getPath(indexPath);
            IndexUtils.checkIndexVersionAndModificationTime(featureFile, indexFile, index);
            return index;
        }
        catch (IOException | RuntimeException e) {
            return null;
        }
    }

    public static void checkIndexVersionAndModificationTime(Path featureFile, Path indexFile, Index index) {
        Utils.nonNull(featureFile, "feature-file");
        Utils.nonNull(indexFile, "indexFile");
        Utils.nonNull(index, "index");
        if (!index.isCurrentVersion()) {
            throw new UserException("Index file " + indexFile + " is out of date (old version). Use " + IndexFeatureFile.class.getSimpleName() + " to make an index.");
        }
        try {
            if (Files.getLastModifiedTime(indexFile, new LinkOption[0]).compareTo(Files.getLastModifiedTime(featureFile, new LinkOption[0])) < 0) {
                logger.warn("Index file " + indexFile + " is out of date (index older than input file). Use " + IndexFeatureFile.class.getSimpleName() + " to make a new index.");
            }
        }
        catch (IOException e) {
            logger.warn("Could not validate index modification time.  Assuming index is up-to-date.");
        }
    }

    public static SAMSequenceDictionary createSequenceDictionaryFromFeatureIndex(Path featureFile) {
        Utils.nonNull(featureFile);
        logger.warn(String.format("Feature file \"%s\" appears to contain no sequence dictionary. Attempting to retrieve a sequence dictionary from the associated index file", featureFile.toAbsolutePath().toUri().toString()));
        Index index = IndexUtils.loadIndex(featureFile);
        return index == null ? null : IndexUtils.getSamSequenceDictionaryFromIndex(index);
    }

    public static boolean isSequenceDictionaryFromIndex(SAMSequenceDictionary sequenceDictionary) {
        Utils.nonNull(sequenceDictionary);
        return sequenceDictionary.getSequences().stream().allMatch(seqRec -> seqRec.getSequenceLength() == 0);
    }

    public static Index loadIndex(Path featureFile) {
        Utils.nonNull(featureFile);
        Index tribbleIndex = IndexUtils.loadTribbleIndex(featureFile);
        if (tribbleIndex != null) {
            return tribbleIndex;
        }
        Index tabixIndex = IndexUtils.loadTabixIndex(featureFile);
        if (tabixIndex != null) {
            return tabixIndex;
        }
        return null;
    }

    private static SAMSequenceDictionary getSamSequenceDictionaryFromIndex(Index index) {
        List seqNames = index.getSequenceNames();
        if (seqNames == null || seqNames.isEmpty()) {
            return null;
        }
        SAMSequenceDictionary dict = new SAMSequenceDictionary();
        seqNames.forEach(seqName -> dict.addSequence(new SAMSequenceRecord(seqName, 0)));
        return dict;
    }
}

