/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils;

import htsjdk.samtools.QueryInterval;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Locatable;
import htsjdk.samtools.util.OverlapDetector;
import htsjdk.tribble.Feature;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.barclay.argparser.CommandLineException;
import org.broadinstitute.hellbender.engine.FeatureDataSource;
import org.broadinstitute.hellbender.engine.FeatureManager;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.GenomeLoc;
import org.broadinstitute.hellbender.utils.GenomeLocParser;
import org.broadinstitute.hellbender.utils.GenomeLocSortedSet;
import org.broadinstitute.hellbender.utils.IntervalMergingRule;
import org.broadinstitute.hellbender.utils.IntervalSetRule;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.fasta.CachingIndexedFastaSequenceFile;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import org.broadinstitute.hellbender.utils.nio.PathLineIterator;
import org.broadinstitute.hellbender.utils.param.ParamUtils;

public final class IntervalUtils {
    private static final Logger log = LogManager.getLogger(IntervalUtils.class);
    public static final List<String> GATK_INTERVAL_FILE_EXTENSIONS = Collections.unmodifiableList(Arrays.asList(".list", ".intervals"));
    public static final Comparator<Locatable> LEXICOGRAPHICAL_ORDER_COMPARATOR = Comparator.comparing(Locatable::getContig, Comparator.nullsLast(String::compareTo)).thenComparingInt(Locatable::getStart).thenComparingInt(Locatable::getEnd);
    private static final Logger logger = LogManager.getLogger(IntervalUtils.class);

    public static final int compareLocatables(Locatable first, Locatable second, SAMSequenceDictionary dictionary) {
        Utils.nonNull(first);
        Utils.nonNull(second);
        Utils.nonNull(dictionary);
        int result = 0;
        if (first != second && (result = IntervalUtils.compareContigs(first, second, dictionary)) == 0 && (result = Integer.compare(first.getStart(), second.getStart())) == 0) {
            result = Integer.compare(first.getEnd(), second.getEnd());
        }
        return result;
    }

    public static boolean isBefore(Locatable first, Locatable second, SAMSequenceDictionary dictionary) {
        Utils.nonNull(first);
        Utils.nonNull(second);
        Utils.nonNull(dictionary);
        int contigComparison = IntervalUtils.compareContigs(first, second, dictionary);
        return contigComparison == -1 || contigComparison == 0 && first.getEnd() < second.getStart();
    }

    public static boolean isAfter(Locatable first, Locatable second, SAMSequenceDictionary dictionary) {
        Utils.nonNull(first);
        Utils.nonNull(second);
        Utils.nonNull(dictionary);
        int contigComparison = IntervalUtils.compareContigs(first, second, dictionary);
        return contigComparison == 1 || contigComparison == 0 && first.getStart() > second.getEnd();
    }

    public static int compareContigs(Locatable first, Locatable second, SAMSequenceDictionary dictionary) {
        Utils.nonNull(first);
        Utils.nonNull(second);
        Utils.nonNull(dictionary);
        int firstRefIndex = dictionary.getSequenceIndex(first.getContig());
        int secondRefIndex = dictionary.getSequenceIndex(second.getContig());
        if (firstRefIndex == -1 || secondRefIndex == -1) {
            throw new IllegalArgumentException("Can't do comparison because Locatables' contigs not found in sequence dictionary");
        }
        return Integer.compare(firstRefIndex, secondRefIndex);
    }

    public static SimpleInterval getSpanningInterval(List<? extends Locatable> locations) {
        Utils.nonNull(locations);
        Utils.containsNoNull(locations, "locations must not contain a null");
        if (locations.isEmpty()) {
            return null;
        }
        List contigs = locations.stream().map(l -> l.getContig()).distinct().collect(Collectors.toList());
        Utils.validateArg(contigs.size() == 1, () -> "found different contigs from inputs:" + contigs);
        int minStart = locations.stream().mapToInt(l -> l.getStart()).min().getAsInt();
        int maxEnd = locations.stream().mapToInt(l -> l.getEnd()).max().getAsInt();
        return new SimpleInterval((String)contigs.get(0), minStart, maxEnd);
    }

    public static List<SimpleInterval> convertGenomeLocsToSimpleIntervals(List<GenomeLoc> genomeLocIntervals) {
        ArrayList<SimpleInterval> convertedIntervals = new ArrayList<SimpleInterval>(genomeLocIntervals.size());
        for (GenomeLoc genomeLoc : genomeLocIntervals) {
            if (genomeLoc.isUnmapped()) {
                throw new UserException("Unmapped intervals cannot be converted to SimpleIntervals");
            }
            convertedIntervals.add(new SimpleInterval(genomeLoc));
        }
        return convertedIntervals;
    }

    public static QueryInterval convertSimpleIntervalToQueryInterval(SimpleInterval interval, SAMSequenceDictionary sequenceDictionary) {
        Utils.nonNull(interval);
        Utils.nonNull(sequenceDictionary);
        int contigIndex = sequenceDictionary.getSequenceIndex(interval.getContig());
        if (contigIndex == -1) {
            throw new UserException("Contig " + interval.getContig() + " not present in reads sequence dictionary");
        }
        return new QueryInterval(contigIndex, interval.getStart(), interval.getEnd());
    }

    public static GenomeLocSortedSet loadIntervals(List<String> intervalStrings, IntervalSetRule intervalSetRule, IntervalMergingRule intervalMergingRule, int padding, GenomeLocParser genomeLocParser) {
        Utils.nonNull(intervalStrings);
        List<GenomeLoc> allIntervals = new ArrayList<GenomeLoc>();
        for (String intervalString : intervalStrings) {
            Utils.nonNull(intervalString);
            List<GenomeLoc> intervals = IntervalUtils.parseIntervalArguments(genomeLocParser, intervalString);
            if (padding > 0) {
                intervals = IntervalUtils.getIntervalsWithFlanks(genomeLocParser, intervals, padding);
            }
            allIntervals = IntervalUtils.mergeListsBySetOperator(intervals, allIntervals, intervalSetRule);
        }
        return IntervalUtils.sortAndMergeIntervals(genomeLocParser, allIntervals, intervalMergingRule);
    }

    public static List<GenomeLoc> loadIntervalsNonMerging(List<String> intervalStrings, int padding, GenomeLocParser genomeLocParser) {
        Utils.nonNull(intervalStrings);
        ArrayList<GenomeLoc> allIntervals = new ArrayList<GenomeLoc>();
        for (String intervalString : intervalStrings) {
            Utils.nonNull(intervalString);
            List<GenomeLoc> intervals = IntervalUtils.parseIntervalArguments(genomeLocParser, intervalString);
            if (padding > 0) {
                intervals = intervals.stream().map(loc -> genomeLocParser.createPaddedGenomeLoc((GenomeLoc)loc, padding)).collect(Collectors.toList());
            }
            allIntervals.addAll(intervals);
        }
        return allIntervals.stream().sorted().collect(Collectors.toList());
    }

    public static List<GenomeLoc> parseIntervalArguments(GenomeLocParser parser, List<String> argList) {
        ArrayList<GenomeLoc> rawIntervals = new ArrayList<GenomeLoc>();
        if (argList != null) {
            for (String argument : argList) {
                rawIntervals.addAll(IntervalUtils.parseIntervalArguments(parser, argument));
            }
        }
        return rawIntervals;
    }

    public static List<GenomeLoc> parseIntervalArguments(GenomeLocParser parser, String arg) {
        Utils.nonNull(parser, "parser is null");
        Utils.nonNull(arg, "arg is null");
        ArrayList<GenomeLoc> rawIntervals = new ArrayList<GenomeLoc>();
        if (arg.indexOf(59) != -1) {
            throw new CommandLineException.BadArgumentValue("-L " + arg, "The legacy -L \"interval1;interval2\" syntax is no longer supported. Please use one -L argument for each interval or an interval file instead.");
        }
        if (FeatureManager.isFeatureFile(IOUtils.getPath(arg))) {
            try {
                rawIntervals.addAll(IntervalUtils.featureFileToIntervals(parser, arg));
            }
            catch (IllegalArgumentException e) {
                throw new UserException.MalformedFile(IOUtils.getPath(arg), "Failure while loading intervals from file.", (Exception)e);
            }
        } else if (IntervalUtils.isGatkIntervalFile(arg)) {
            rawIntervals.addAll(IntervalUtils.gatkIntervalFileToList(parser, arg));
        } else {
            if (new File(arg).exists()) {
                throw new UserException.CouldNotReadInputFile(arg, String.format("The file %s exists, but does not contain Features (ie., is not in a supported Feature file format such as vcf, bcf, bed, or interval_list), and does not have one of the supported interval file extensions (" + GATK_INTERVAL_FILE_EXTENSIONS + "). Please rename your file with the appropriate extension. If %s is NOT supposed to be a file, please move or rename the file at location %s", arg, arg, new File(arg).getAbsolutePath()));
            }
            rawIntervals.add(parser.parseGenomeLoc(arg));
        }
        return rawIntervals;
    }

    public static List<GenomeLoc> featureFileToIntervals(GenomeLocParser parser, String featureFile) {
        try (FeatureDataSource dataSource = new FeatureDataSource(featureFile);){
            ArrayList<GenomeLoc> featureIntervals = new ArrayList<GenomeLoc>();
            for (Feature feature : dataSource) {
                featureIntervals.add(parser.createGenomeLoc(feature));
            }
            ArrayList<GenomeLoc> arrayList = featureIntervals;
            return arrayList;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static List<GenomeLoc> gatkIntervalFileToList(GenomeLocParser glParser, String fileName) {
        Utils.nonNull(glParser, "glParser is null");
        Utils.nonNull(fileName, "file name is null");
        Path inputPath = IOUtils.getPath(fileName);
        ArrayList<GenomeLoc> ret = new ArrayList<GenomeLoc>();
        try (PathLineIterator reader = new PathLineIterator(inputPath);){
            for (String line : reader) {
                String trimmedLine = line.trim();
                if (trimmedLine.isEmpty()) continue;
                ret.add(glParser.parseGenomeLoc(trimmedLine));
            }
            if (ret.isEmpty()) {
                throw new UserException.MalformedFile(inputPath, "It contains no intervals.");
            }
            ArrayList<GenomeLoc> arrayList = ret;
            return arrayList;
        }
        catch (UserException e) {
            throw e;
        }
        catch (Exception e) {
            throw new UserException.MalformedFile(inputPath, "Interval file could not be parsed as a GATK interval file", e);
        }
    }

    public static List<GenomeLoc> mergeListsBySetOperator(List<GenomeLoc> setOne, List<GenomeLoc> setTwo, IntervalSetRule rule) {
        if (setOne == null || setOne.isEmpty() || setTwo == null || setTwo.isEmpty()) {
            return Collections.unmodifiableList(setOne == null || setOne.isEmpty() ? setTwo : setOne);
        }
        LinkedList<GenomeLoc> retList = new LinkedList<GenomeLoc>();
        if (rule == null || rule == IntervalSetRule.UNION) {
            retList.addAll(setOne);
            retList.addAll(setTwo);
            return Collections.unmodifiableList(retList);
        }
        int iOne = 0;
        int iTwo = 0;
        while (iTwo < setTwo.size() && iOne < setOne.size()) {
            if (setTwo.get(iTwo).isBefore(setOne.get(iOne))) {
                ++iTwo;
                continue;
            }
            if (setOne.get(iOne).isBefore(setTwo.get(iTwo))) {
                ++iOne;
                continue;
            }
            retList.add(setOne.get(iOne).intersect(setTwo.get(iTwo)));
            if (setOne.get(iOne).getStop() < setTwo.get(iTwo).getStop()) {
                ++iOne;
                continue;
            }
            ++iTwo;
        }
        if (retList.isEmpty()) {
            throw new UserException.EmptyIntersection("There was an empty intersection");
        }
        return Collections.unmodifiableList(retList);
    }

    public static GenomeLocSortedSet sortAndMergeIntervals(GenomeLocParser parser, List<GenomeLoc> intervals, IntervalMergingRule mergingRule) {
        intervals = new ArrayList<GenomeLoc>(intervals);
        Collections.sort(intervals);
        intervals = IntervalUtils.mergeIntervalLocations(intervals, mergingRule);
        return GenomeLocSortedSet.createSetFromList(parser, intervals);
    }

    public static String equateIntervals(List<GenomeLoc> masterArg, List<GenomeLoc> testArg) {
        LinkedList<GenomeLoc> master = new LinkedList<GenomeLoc>(masterArg);
        LinkedList<GenomeLoc> test = new LinkedList<GenomeLoc>(testArg);
        while (!master.isEmpty()) {
            GenomeLoc masterHead = master.pop();
            GenomeLoc testHead = test.pop();
            if (testHead.overlapsP(masterHead)) {
                IntervalUtils.reverse(masterHead.subtract(testHead)).forEach(master::push);
                continue;
            }
            return "Incompatible locs detected masterHead=" + masterHead + ", testHead=" + testHead;
        }
        if (test.isEmpty()) {
            return null;
        }
        return "Remaining elements found in test: first=" + test.peek();
    }

    private static <T extends Comparable<T>> List<T> reverse(List<T> l) {
        return l.stream().sorted(Collections.reverseOrder()).collect(Collectors.toList());
    }

    public static boolean isGatkIntervalFile(String str) {
        return IntervalUtils.isGatkIntervalFile(str, true);
    }

    public static boolean isGatkIntervalFile(String str, boolean checkExists) {
        Utils.nonNull(str);
        boolean hasIntervalFileExtension = false;
        for (String extension : GATK_INTERVAL_FILE_EXTENSIONS) {
            if (!str.toLowerCase().endsWith(extension)) continue;
            hasIntervalFileExtension = true;
        }
        if (hasIntervalFileExtension) {
            Path path = IOUtils.getPath(str);
            if (!checkExists || Files.exists(path, new LinkOption[0])) {
                return true;
            }
            throw new UserException.CouldNotReadInputFile(path, "The interval file does not exist.");
        }
        return false;
    }

    public static Map<String, Integer> getContigSizes(Path reference) {
        try (CachingIndexedFastaSequenceFile referenceSequenceFile = new CachingIndexedFastaSequenceFile(reference);){
            List<GenomeLoc> locs = GenomeLocSortedSet.createSetFromSequenceDictionary(referenceSequenceFile.getSequenceDictionary()).toList();
            LinkedHashMap<String, Integer> lengths = new LinkedHashMap<String, Integer>();
            for (GenomeLoc loc : locs) {
                lengths.put(loc.getContig(), loc.size());
            }
            LinkedHashMap<String, Integer> linkedHashMap = lengths;
            return linkedHashMap;
        }
    }

    public static void scatterContigIntervals(SAMFileHeader fileHeader, List<GenomeLoc> locs, List<File> scatterParts) {
        long totalBases = 0L;
        for (GenomeLoc loc : locs) {
            totalBases += (long)loc.size();
        }
        long idealBasesPerPart = totalBases / (long)scatterParts.size();
        if (idealBasesPerPart == 0L) {
            throw new UserException.BadInput(String.format("Genome region is too short (%d bases) to split into %d parts", totalBases, scatterParts.size()));
        }
        ArrayList<Integer> contigStartLocs = new ArrayList<Integer>();
        String prevContig = null;
        for (int i = 0; i < locs.size(); ++i) {
            GenomeLoc loc = locs.get(i);
            if (prevContig == null || !loc.getContig().equals(prevContig)) {
                contigStartLocs.add(i);
            }
            prevContig = loc.getContig();
        }
        if (contigStartLocs.size() < scatterParts.size()) {
            throw new UserException.BadInput(String.format("Input genome region has too few contigs (%d) to split into %d parts", contigStartLocs.size(), scatterParts.size()));
        }
        long thisPartBases = 0L;
        int partIdx = 0;
        IntervalList outList = new IntervalList(fileHeader);
        for (int i = 0; i < locs.size(); ++i) {
            GenomeLoc loc = locs.get(i);
            thisPartBases += (long)(loc.getStop() - loc.getStart());
            outList.add(IntervalUtils.toInterval(loc, i));
            boolean partMustStop = false;
            if (partIdx < scatterParts.size() - 1) {
                int nextPart = partIdx + 1;
                int nextPartMustStartBy = (Integer)contigStartLocs.get(nextPart + (contigStartLocs.size() - scatterParts.size()));
                if (i + 1 == nextPartMustStartBy) {
                    partMustStop = true;
                }
            } else if (i == locs.size() - 1) {
                partMustStop = true;
            }
            if (!partMustStop && thisPartBases <= idealBasesPerPart) continue;
            GenomeLoc nextLoc = null;
            if (i + 1 < locs.size()) {
                nextLoc = locs.get(i + 1);
            }
            if (nextLoc != null && nextLoc.getContig().equals(loc.getContig())) continue;
            outList.write(scatterParts.get(partIdx));
            outList = new IntervalList(fileHeader);
            thisPartBases -= idealBasesPerPart;
            ++partIdx;
        }
    }

    public static List<List<GenomeLoc>> splitIntervalsToSubLists(List<GenomeLoc> locs, List<Integer> splits) {
        Utils.nonNull(locs, "locs is null");
        Utils.nonNull(splits, "splits is null");
        int start = 0;
        ArrayList<List<GenomeLoc>> sublists = new ArrayList<List<GenomeLoc>>(splits.size());
        for (Integer stop : splits) {
            ArrayList<GenomeLoc> curList = new ArrayList<GenomeLoc>();
            for (int i = start; i < stop; ++i) {
                curList.add(locs.get(i));
            }
            start = stop;
            sublists.add(curList);
        }
        return sublists;
    }

    public static void scatterFixedIntervals(SAMFileHeader fileHeader, List<List<GenomeLoc>> splits, List<File> scatterParts) {
        Utils.nonNull(fileHeader, "fileHeader is null");
        Utils.nonNull(splits, "splits is null");
        Utils.nonNull(scatterParts, "scatterParts is null");
        Utils.containsNoNull(splits, "null split loc");
        if (splits.size() != scatterParts.size()) {
            throw new CommandLineException.BadArgumentValue("splits", String.format("Split points %d does not equal the number of scatter parts %d.", splits.size(), scatterParts.size()));
        }
        int fileIndex = 0;
        int locIndex = 1;
        for (List<GenomeLoc> split : splits) {
            IntervalList intervalList = new IntervalList(fileHeader);
            for (GenomeLoc loc : split) {
                intervalList.add(IntervalUtils.toInterval(loc, locIndex++));
            }
            intervalList.write(scatterParts.get(fileIndex++));
        }
    }

    public static List<List<GenomeLoc>> splitFixedIntervals(List<GenomeLoc> locs, int numParts) {
        Utils.nonNull(locs, "locs is null");
        if (locs.size() < numParts) {
            throw new CommandLineException.BadArgumentValue("scatterParts", String.format("Cannot scatter %d locs into %d parts.", locs.size(), numParts));
        }
        long locsSize = IntervalUtils.intervalSize(locs);
        ArrayList<Integer> splitPoints = new ArrayList<Integer>();
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize, 0, locs.size(), numParts);
        Collections.sort(splitPoints);
        splitPoints.add(locs.size());
        return IntervalUtils.splitIntervalsToSubLists(locs, splitPoints);
    }

    public static List<List<GenomeLoc>> splitLocusIntervals(List<GenomeLoc> locs, int numParts) {
        Utils.nonNull(locs, "locs is null");
        if (numParts < 0) {
            throw new CommandLineException.BadArgumentValue("scatterParts", String.format("Cannot scatter %d locs into %d parts.", locs.size(), numParts));
        }
        long bp = IntervalUtils.intervalSize(locs);
        long idealSplitSize = Math.max((long)Math.floor((double)bp / (1.0 * (double)numParts)), 1L);
        ArrayList<List<GenomeLoc>> splits = new ArrayList<List<GenomeLoc>>(numParts);
        LinkedList<GenomeLoc> locsLinkedList = new LinkedList<GenomeLoc>(locs);
        while (!locsLinkedList.isEmpty()) {
            if (splits.size() + 1 == numParts) {
                splits.add(new ArrayList<GenomeLoc>(locsLinkedList));
                locsLinkedList.clear();
                continue;
            }
            SplitLocusRecursive one = IntervalUtils.splitLocusIntervals1(locsLinkedList, idealSplitSize);
            splits.add(one.split);
            locsLinkedList = one.remaining;
        }
        return splits;
    }

    private static SplitLocusRecursive splitLocusIntervals1(LinkedList<GenomeLoc> remaining, long idealSplitSize) {
        ArrayList<GenomeLoc> split = new ArrayList<GenomeLoc>();
        long size = 0L;
        while (!remaining.isEmpty()) {
            GenomeLoc head = remaining.pop();
            long newSize = size + (long)head.size();
            if (newSize == idealSplitSize) {
                split.add(head);
                break;
            }
            if (newSize > idealSplitSize) {
                long remainingBp = idealSplitSize - size;
                long cutPoint = (long)head.getStart() + remainingBp;
                GenomeLoc[] parts = head.split((int)cutPoint);
                remaining.push(parts[1]);
                remaining.push(parts[0]);
                continue;
            }
            split.add(head);
            size = newSize;
        }
        return new SplitLocusRecursive(split, remaining);
    }

    public static boolean overlaps(Locatable left, Locatable right) {
        Utils.nonNull(left, "the left locatable is null");
        Utils.nonNull(right, "the right locatable is null");
        if (left.getContig() == null || right.getContig() == null) {
            return false;
        }
        if (!left.getContig().equals(right.getContig())) {
            return false;
        }
        return left.getStart() <= right.getEnd() && right.getStart() <= left.getEnd();
    }

    public static String locatableToString(Locatable interval) {
        Utils.nonNull(interval);
        return String.format("%s:%s-%s", interval.getContig(), interval.getStart(), interval.getEnd());
    }

    public static List<GenomeLoc> flattenSplitIntervals(List<List<GenomeLoc>> splits) {
        Utils.nonNull(splits, "splits is null");
        ArrayList<GenomeLoc> locs = new ArrayList<GenomeLoc>();
        splits.forEach(locs::addAll);
        return locs;
    }

    private static void addFixedSplit(List<Integer> splitPoints, List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int numParts) {
        Utils.nonNull(splitPoints, "splitPoints is null");
        if (numParts < 2) {
            return;
        }
        int halfParts = (numParts + 1) / 2;
        Pair<Integer, Long> splitPoint = IntervalUtils.getFixedSplit(locs, locsSize, startIndex, stopIndex, halfParts, numParts - halfParts);
        int splitIndex = (Integer)splitPoint.getLeft();
        long splitSize = (Long)splitPoint.getRight();
        splitPoints.add(splitIndex);
        IntervalUtils.addFixedSplit(splitPoints, locs, splitSize, startIndex, splitIndex, halfParts);
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize - splitSize, splitIndex, stopIndex, numParts - halfParts);
    }

    private static Pair<Integer, Long> getFixedSplit(List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int minLocs, int maxLocs) {
        int splitIndex = startIndex;
        long splitSize = 0L;
        for (int i = 0; i < minLocs; ++i) {
            splitSize += (long)locs.get(splitIndex).size();
            ++splitIndex;
        }
        long halfSize = locsSize / 2L;
        while (splitIndex < stopIndex - maxLocs && splitSize < halfSize) {
            splitSize += (long)locs.get(splitIndex).size();
            ++splitIndex;
        }
        return new ImmutablePair((Object)splitIndex, (Object)splitSize);
    }

    private static Interval toInterval(GenomeLoc loc, int locIndex) {
        return new Interval(loc.getContig(), loc.getStart(), loc.getStop(), false, "interval_" + locIndex);
    }

    public static List<GenomeLoc> mergeIntervalLocations(List<GenomeLoc> raw, IntervalMergingRule rule) {
        if (raw.size() <= 1) {
            return Collections.unmodifiableList(raw);
        }
        ArrayList<GenomeLoc> merged = new ArrayList<GenomeLoc>();
        Iterator<GenomeLoc> it = raw.iterator();
        GenomeLoc prev = it.next();
        while (it.hasNext()) {
            GenomeLoc curr = it.next();
            if (prev.overlapsP(curr)) {
                prev = prev.merge(curr);
                continue;
            }
            if (prev.contiguousP(curr) && (rule == null || rule == IntervalMergingRule.ALL)) {
                prev = prev.merge(curr);
                continue;
            }
            merged.add(prev);
            prev = curr;
        }
        merged.add(prev);
        return Collections.unmodifiableList(merged);
    }

    public static long intervalSize(List<GenomeLoc> locs) {
        long size = 0L;
        for (GenomeLoc loc : locs) {
            size += (long)loc.size();
        }
        return size;
    }

    public static List<GenomeLoc> getIntervalsWithFlanks(GenomeLocParser parser, List<GenomeLoc> locs, int basePairs) {
        if (locs.isEmpty()) {
            return Collections.emptyList();
        }
        List<GenomeLoc> expanded = locs.stream().map(loc -> parser.createPaddedGenomeLoc((GenomeLoc)loc, basePairs)).collect(Collectors.toList());
        return IntervalUtils.sortAndMergeIntervals(parser, expanded, IntervalMergingRule.ALL).toList();
    }

    public static List<SimpleInterval> getIntervalsWithFlanks(List<SimpleInterval> intervals, int basePairs, SAMSequenceDictionary dictionary) {
        GenomeLocParser parser = new GenomeLocParser(dictionary);
        List<GenomeLoc> intervalsAsGenomeLocs = IntervalUtils.genomeLocsFromLocatables(parser, intervals);
        List<GenomeLoc> paddedGenomeLocs = IntervalUtils.getIntervalsWithFlanks(parser, intervalsAsGenomeLocs, basePairs);
        return IntervalUtils.convertGenomeLocsToSimpleIntervals(paddedGenomeLocs);
    }

    public static List<List<SimpleInterval>> groupIntervalsByContig(List<SimpleInterval> sortedIntervals) {
        ArrayList<List<SimpleInterval>> intervalGroups = new ArrayList<List<SimpleInterval>>();
        ArrayList<SimpleInterval> currentGroup = new ArrayList<SimpleInterval>();
        String currentContig = null;
        for (SimpleInterval currentInterval : sortedIntervals) {
            if (currentContig != null && !currentContig.equals(currentInterval.getContig())) {
                intervalGroups.add(currentGroup);
                currentGroup = new ArrayList();
            }
            currentContig = currentInterval.getContig();
            currentGroup.add(currentInterval);
        }
        if (!currentGroup.isEmpty()) {
            intervalGroups.add(currentGroup);
        }
        return intervalGroups;
    }

    private static LinkedHashMap<String, List<GenomeLoc>> splitByContig(List<GenomeLoc> sorted) {
        LinkedHashMap<String, List<GenomeLoc>> splits = new LinkedHashMap<String, List<GenomeLoc>>();
        GenomeLoc last = null;
        ArrayList<GenomeLoc> contigLocs = null;
        for (GenomeLoc loc : sorted) {
            if (GenomeLoc.isUnmapped(loc)) continue;
            if (last == null || !last.onSameContig(loc)) {
                contigLocs = new ArrayList<GenomeLoc>();
                splits.put(loc.getContig(), contigLocs);
            }
            contigLocs.add(loc);
            last = loc;
        }
        return splits;
    }

    public static List<GenomeLoc> genomeLocsFromLocatables(GenomeLocParser parser, Collection<? extends Locatable> locatables) {
        Utils.nonNull(parser, "the input genome-loc parser cannot be null");
        Utils.nonNull(locatables, "the input locatable collection cannot be null");
        Utils.containsNoNull(locatables, "some element in the locatable input collection is null");
        List result = locatables.stream().map(parser::createGenomeLoc).collect(Collectors.toList());
        return Collections.unmodifiableList(result);
    }

    public static List<SimpleInterval> getAllIntervalsForReference(SAMSequenceDictionary sequenceDictionary) {
        return GenomeLocSortedSet.createSetFromSequenceDictionary(sequenceDictionary).stream().map(SimpleInterval::new).collect(Collectors.toList());
    }

    public static List<SimpleInterval> getResolvedIntervals(String intervalQueryString, SAMSequenceDictionary sequenceDictionary) {
        int lastColonIndex;
        Utils.nonNull(intervalQueryString);
        Utils.validateArg(!intervalQueryString.isEmpty(), "intervalQueryString should not be empty");
        ArrayList<SimpleInterval> resolvedIntervals = new ArrayList<SimpleInterval>();
        SAMSequenceRecord queryAsContigName = sequenceDictionary.getSequence(intervalQueryString);
        if (queryAsContigName != null) {
            resolvedIntervals.add(new SimpleInterval(intervalQueryString, 1, queryAsContigName.getSequenceLength()));
        }
        if ((lastColonIndex = intervalQueryString.lastIndexOf(58)) == -1) {
            return resolvedIntervals;
        }
        String prefix = intervalQueryString.substring(0, lastColonIndex);
        SAMSequenceRecord prefixSequence = sequenceDictionary.getSequence(prefix);
        if (prefixSequence == null) {
            return resolvedIntervals;
        }
        try {
            int endPos;
            int startPos;
            int lastDashIndex = intervalQueryString.lastIndexOf(45);
            if (intervalQueryString.endsWith("+")) {
                startPos = SimpleInterval.parsePositionThrowOnFailure(intervalQueryString.substring(lastColonIndex + 1, intervalQueryString.length() - 1));
                endPos = prefixSequence.getSequenceLength();
            } else if (lastDashIndex > lastColonIndex) {
                startPos = SimpleInterval.parsePositionThrowOnFailure(intervalQueryString.substring(lastColonIndex + 1, lastDashIndex));
                endPos = SimpleInterval.parsePositionThrowOnFailure(intervalQueryString.substring(lastDashIndex + 1, intervalQueryString.length()));
            } else {
                endPos = startPos = SimpleInterval.parsePositionThrowOnFailure(intervalQueryString.substring(lastColonIndex + 1, intervalQueryString.length()));
            }
            if (SimpleInterval.isValid(prefix, startPos, endPos)) {
                resolvedIntervals.add(new SimpleInterval(prefix, startPos, endPos));
            } else if (resolvedIntervals.isEmpty()) {
                SimpleInterval.validatePositions(prefix, startPos, endPos);
            }
        }
        catch (NumberFormatException e) {
            if (resolvedIntervals.isEmpty()) {
                throw e;
            }
            log.warn(String.format("The query interval string \"%s\" is interpreted as a query against the contig named \"%s\", but may have been intended as an (accidentally malformed) query against the contig named \"%s\"", intervalQueryString, ((SimpleInterval)resolvedIntervals.get(0)).getContig(), prefixSequence.getSequenceName()));
        }
        return resolvedIntervals;
    }

    public static SimpleInterval trimIntervalToContig(String contig, int start, int stop, int contigLength) {
        Utils.nonNull(contig);
        Utils.validateArg(contigLength >= 1, () -> "contigLength should be at least 1 but was " + contigLength);
        int boundedStart = Math.max(1, start);
        int boundedStop = Math.min(contigLength, stop);
        if (boundedStart > contigLength || boundedStop < 1) {
            return null;
        }
        return new SimpleInterval(contig, boundedStart, boundedStop);
    }

    public static boolean intervalIsOnDictionaryContig(SimpleInterval interval, SAMSequenceDictionary dictionary) {
        Utils.nonNull(interval);
        Utils.nonNull(dictionary);
        SAMSequenceRecord contigRecord = dictionary.getSequence(interval.getContig());
        if (contigRecord == null) {
            return false;
        }
        return interval.getEnd() <= contigRecord.getSequenceLength();
    }

    public static List<SimpleInterval> cutToShards(Iterable<SimpleInterval> intervals, int shardSize) {
        ArrayList<SimpleInterval> ret = new ArrayList<SimpleInterval>();
        for (SimpleInterval i : intervals) {
            int endShard;
            int beginShard = IntervalUtils.shardIndex(i.getStart(), shardSize);
            if (beginShard == (endShard = IntervalUtils.shardIndex(i.getEnd(), shardSize))) {
                ret.add(i);
                continue;
            }
            ret.add(new SimpleInterval(i.getContig(), i.getStart(), IntervalUtils.endOfShard(beginShard, shardSize)));
            for (int shard = beginShard + 1; shard < endShard; ++shard) {
                ret.add(new SimpleInterval(i.getContig(), IntervalUtils.beginOfShard(shard, shardSize), IntervalUtils.endOfShard(shard, shardSize)));
            }
            ret.add(new SimpleInterval(i.getContig(), IntervalUtils.beginOfShard(endShard, shardSize), i.getEnd()));
        }
        return ret;
    }

    public static int shardIndex(int oneBasedOffset, int shardSize) {
        return (oneBasedOffset - 1) / shardSize;
    }

    public static int beginOfShard(int shardIndex, int shardSize) {
        return shardIndex * shardSize + 1;
    }

    public static int endOfShard(int shardIndex, int shardSize) {
        return IntervalUtils.beginOfShard(shardIndex + 1, shardSize) - 1;
    }

    public static List<SimpleInterval> getSpanningIntervals(List<? extends Locatable> locations, SAMSequenceDictionary sequenceDictionary) {
        return locations.stream().collect(Collectors.groupingBy(Locatable::getContig)).values().stream().map(IntervalUtils::getSpanningInterval).sorted((i1, i2) -> IntervalUtils.compareLocatables(i1, i2, sequenceDictionary)).collect(Collectors.toList());
    }

    public static <T extends Locatable> List<Locatable> combineAndSortBreakpoints(List<T> unsortedLocatables1, List<T> unsortedLocatables2, SAMSequenceDictionary dictionary) {
        Utils.nonNull(dictionary);
        List<T> locatables1 = IntervalUtils.sortLocatablesBySequenceDictionary(unsortedLocatables1, dictionary);
        List<T> locatables2 = IntervalUtils.sortLocatablesBySequenceDictionary(unsortedLocatables2, dictionary);
        if (locatables1 == null && locatables2 == null) {
            return Collections.emptyList();
        }
        IntervalUtils.validateNoOverlappingIntervals(locatables1);
        IntervalUtils.validateNoOverlappingIntervals(locatables2);
        if (CollectionUtils.isEmpty(locatables1)) {
            return locatables2.stream().map(SimpleInterval::new).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(locatables2)) {
            return locatables1.stream().map(SimpleInterval::new).collect(Collectors.toList());
        }
        ArrayList<T> masterList = new ArrayList<T>();
        masterList.addAll(locatables1);
        masterList.addAll(locatables2);
        Set contigs = masterList.stream().map(Locatable::getContig).collect(Collectors.toSet());
        Map contigToBreakpoints = contigs.stream().collect(Collectors.toMap(Function.identity(), l -> new HashSet()));
        masterList.forEach(l -> {
            ((Set)contigToBreakpoints.get(l.getContig())).add(Pair.of((Object)l.getStart(), (Object)((Object)IntervalBreakpointType.START_BREAKPOINT)));
            ((Set)contigToBreakpoints.get(l.getContig())).add(Pair.of((Object)l.getEnd(), (Object)((Object)IntervalBreakpointType.END_BREAKPOINT)));
        });
        ArrayList<SimpleInterval> result = new ArrayList<SimpleInterval>();
        for (String contig : contigs) {
            ArrayList breakpoints = new ArrayList(contigToBreakpoints.get(contig));
            breakpoints.sort((p1, p2) -> {
                int firstComparison = ((Integer)p1.getLeft()).compareTo((Integer)p2.getLeft());
                if (firstComparison != 0) {
                    return firstComparison;
                }
                return ((IntervalBreakpointType)((Object)((Object)p1.getRight()))).compareTo((Enum)p2.getRight());
            });
            int numCurrentStarts = 0;
            int numCurrentEnds = 0;
            for (int i = 0; i < breakpoints.size() - 1; ++i) {
                boolean isBetweenIntervals;
                int currentBreakpoint = (Integer)((Pair)breakpoints.get(i)).getLeft();
                int nextBreakpoint = (Integer)((Pair)breakpoints.get(i + 1)).getLeft();
                boolean isCurrentBreakpointStart = ((Pair)breakpoints.get(i)).getRight() == IntervalBreakpointType.START_BREAKPOINT;
                boolean isNextBreakpointStart = ((Pair)breakpoints.get(i + 1)).getRight() == IntervalBreakpointType.START_BREAKPOINT;
                int start = !isCurrentBreakpointStart && !isNextBreakpointStart ? currentBreakpoint + 1 : currentBreakpoint;
                int end = isCurrentBreakpointStart && isNextBreakpointStart ? nextBreakpoint - 1 : nextBreakpoint;
                boolean bl = isBetweenIntervals = !isCurrentBreakpointStart && isNextBreakpointStart;
                if (isBetweenIntervals) {
                    ++start;
                    --end;
                }
                if (isCurrentBreakpointStart) {
                    ++numCurrentStarts;
                } else {
                    ++numCurrentEnds;
                }
                if (isBetweenIntervals && numCurrentStarts <= numCurrentEnds || start > end) continue;
                result.add(new SimpleInterval(contig, start, end));
            }
        }
        return IntervalUtils.sortLocatablesBySequenceDictionary(result, dictionary);
    }

    public static <T extends Locatable> void validateNoOverlappingIntervals(List<T> locatables) {
        if (CollectionUtils.isEmpty(locatables)) {
            return;
        }
        HashSet<T> locatablesSet = new HashSet<T>(locatables);
        if (locatablesSet.size() != locatables.size()) {
            throw new UserException.BadInput("Duplicate(s) detected in input:  " + locatables.size() + " intervals had " + (locatables.size() - locatablesSet.size()) + " duplicates.");
        }
        OverlapDetector overlapDetector = OverlapDetector.create(locatables);
        for (Locatable locatable : locatables) {
            Set overlaps = overlapDetector.getOverlaps(locatable);
            if (overlaps.size() <= 1) continue;
            throw new UserException.BadInput("Overlap detected in input:  " + locatable + " overlapped " + StringUtils.join((Iterable)overlaps, (String)", "));
        }
    }

    public static <T extends Locatable> List<T> sortLocatablesBySequenceDictionary(Collection<T> locatables, SAMSequenceDictionary dictionary) {
        ArrayList<T> result;
        Utils.nonNull(dictionary);
        ArrayList<T> arrayList = result = locatables == null ? null : new ArrayList<T>(locatables);
        if (result != null) {
            result.sort(IntervalUtils.getDictionaryOrderComparator(dictionary));
        }
        return result;
    }

    public static <T extends Locatable, U extends Locatable> Map<T, List<U>> createOverlapMap(List<T> keys, List<U> vals, SAMSequenceDictionary dictionary) {
        Utils.nonNull(keys);
        Utils.nonNull(vals);
        Utils.nonNull(dictionary);
        IntervalUtils.validateNoOverlappingIntervals(keys);
        IntervalUtils.validateNoOverlappingIntervals(vals);
        List<Locatable> sortedKeys = IntervalUtils.sortLocatablesBySequenceDictionary(keys, dictionary);
        OverlapDetector overlapDetector = OverlapDetector.create(vals);
        HashMap<Locatable, List<T>> result = new HashMap<Locatable, List<T>>();
        for (Locatable key : sortedKeys) {
            Set overlaps = overlapDetector.getOverlaps(key);
            List<T> overlapsAsList = IntervalUtils.sortLocatablesBySequenceDictionary(overlaps, dictionary);
            result.put(key, overlapsAsList);
        }
        return result;
    }

    public static <T extends Locatable> Comparator<T> getDictionaryOrderComparator(SAMSequenceDictionary dictionary) {
        Utils.nonNull(dictionary);
        return (o1, o2) -> IntervalUtils.compareLocatables(o1, o2, dictionary);
    }

    public static boolean isReciprocalOverlap(SimpleInterval interval1, SimpleInterval interval2, double reciprocalOverlapThreshold) {
        Utils.nonNull(interval1);
        Utils.nonNull(interval2);
        ParamUtils.inRange(reciprocalOverlapThreshold, 0.0, 1.0, "Reciprocal threshold must be between 0.0 and 1.0.");
        if (reciprocalOverlapThreshold == 0.0) {
            return true;
        }
        return interval1.overlaps(interval2) && (double)interval1.intersect(interval2).size() >= (double)interval2.size() * reciprocalOverlapThreshold && (double)interval2.intersect(interval1).size() >= (double)interval1.size() * reciprocalOverlapThreshold;
    }

    public static <L extends Locatable> Map<String, List<SimpleInterval>> sortAndMergeIntervals(Collection<SimpleInterval> input, SAMSequenceDictionary dictionary, IntervalMergingRule rule) {
        GenomeLocParser parser = new GenomeLocParser(dictionary);
        List<GenomeLoc> locs = input.stream().map(parser::createGenomeLoc).collect(Collectors.toList());
        GenomeLocSortedSet resultSet = IntervalUtils.sortAndMergeIntervals(parser, locs, rule);
        return resultSet.stream().map(loc -> new SimpleInterval(loc.contigName, loc.start, loc.stop)).collect(Collectors.groupingBy(SimpleInterval::getContig));
    }

    public static enum IntervalBreakpointType {
        START_BREAKPOINT,
        END_BREAKPOINT;

    }

    private static final class SplitLocusRecursive {
        final List<GenomeLoc> split;
        final LinkedList<GenomeLoc> remaining;

        private SplitLocusRecursive(List<GenomeLoc> split, LinkedList<GenomeLoc> remaining) {
            this.split = split;
            this.remaining = remaining;
        }
    }
}

