/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;

public final class SequenceDictionaryUtils {
    private static final Comparator<SAMSequenceRecord> SEQUENCE_INDEX_ORDER = Comparator.comparing(SAMSequenceRecord::getSequenceIndex);
    protected static final SAMSequenceRecord CHR1_HG18 = new SAMSequenceRecord("chr1", 247249719);
    protected static final SAMSequenceRecord CHR2_HG18 = new SAMSequenceRecord("chr2", 242951149);
    protected static final SAMSequenceRecord CHR10_HG18 = new SAMSequenceRecord("chr10", 135374737);
    protected static final SAMSequenceRecord CHR1_HG19 = new SAMSequenceRecord("chr1", 249250621);
    protected static final SAMSequenceRecord CHR2_HG19 = new SAMSequenceRecord("chr2", 243199373);
    protected static final SAMSequenceRecord CHR10_HG19 = new SAMSequenceRecord("chr10", 135534747);
    protected static final SAMSequenceRecord CHR1_B36 = new SAMSequenceRecord("1", 247249719);
    protected static final SAMSequenceRecord CHR2_B36 = new SAMSequenceRecord("2", 242951149);
    protected static final SAMSequenceRecord CHR10_B36 = new SAMSequenceRecord("10", 135374737);
    protected static final SAMSequenceRecord CHR1_B37 = new SAMSequenceRecord("1", 249250621);
    protected static final SAMSequenceRecord CHR2_B37 = new SAMSequenceRecord("2", 243199373);
    protected static final SAMSequenceRecord CHR10_B37 = new SAMSequenceRecord("10", 135534747);

    private SequenceDictionaryUtils() {
    }

    public static void validateDictionaries(String name1, SAMSequenceDictionary dict1, String name2, SAMSequenceDictionary dict2) {
        boolean requireSuperset = false;
        boolean checkContigOrdering = false;
        SequenceDictionaryUtils.validateDictionaries(name1, dict1, name2, dict2, false, false);
    }

    public static void validateCRAMDictionaryAgainstReference(SAMSequenceDictionary referenceDictionary, SAMSequenceDictionary cramDictionary) {
        boolean requireSuperset = true;
        boolean checkContigOrdering = false;
        SequenceDictionaryUtils.validateDictionaries("reference", referenceDictionary, "reads", cramDictionary, true, false);
    }

    public static void validateDictionaries(String name1, SAMSequenceDictionary dict1, String name2, SAMSequenceDictionary dict2, boolean requireSuperset, boolean checkContigOrdering) {
        Utils.nonNull(dict1, "Something went wrong with sequence dictionary detection, check that " + name1 + " has a valid sequence dictionary");
        Utils.nonNull(dict2, "Something went wrong with sequence dictionary detection, check that " + name2 + " has a valid sequence dictionary");
        SequenceDictionaryCompatibility type = SequenceDictionaryUtils.compareDictionaries(dict1, dict2, checkContigOrdering);
        switch (type) {
            case IDENTICAL: {
                return;
            }
            case SUPERSET: {
                return;
            }
            case COMMON_SUBSET: {
                if (requireSuperset) {
                    Set contigs1 = dict1.getSequences().stream().map(SAMSequenceRecord::getSequenceName).collect(Collectors.toSet());
                    List missingContigs = dict2.getSequences().stream().map(SAMSequenceRecord::getSequenceName).filter(contig -> !contigs1.contains(contig)).collect(Collectors.toList());
                    throw new UserException.IncompatibleSequenceDictionaries(String.format("Dictionary %s is missing contigs found in dictionary %s.  Missing contigs: \n %s \n", name1, name2, String.join((CharSequence)", ", missingContigs)), name1, dict1, name2, dict2);
                }
                return;
            }
            case NO_COMMON_CONTIGS: {
                throw new UserException.IncompatibleSequenceDictionaries("No overlapping contigs found", name1, dict1, name2, dict2);
            }
            case UNEQUAL_COMMON_CONTIGS: {
                List<SAMSequenceRecord> x = SequenceDictionaryUtils.findDisequalCommonContigs(SequenceDictionaryUtils.getCommonContigsByName(dict1, dict2), dict1, dict2);
                SAMSequenceRecord elt1 = x.get(0);
                SAMSequenceRecord elt2 = x.get(1);
                throw new UserException.IncompatibleSequenceDictionaries(String.format("Found contigs with the same name but different lengths:\n  contig %s = %s / %d\n  contig %s = %s / %d", name1, elt1.getSequenceName(), elt1.getSequenceLength(), name2, elt2.getSequenceName(), elt2.getSequenceLength()), name1, dict1, name2, dict2);
            }
            case NON_CANONICAL_HUMAN_ORDER: {
                UserException.LexicographicallySortedSequenceDictionary ex = SequenceDictionaryUtils.nonCanonicalHumanContigOrder(dict1) ? new UserException.LexicographicallySortedSequenceDictionary(name1, dict1) : new UserException.LexicographicallySortedSequenceDictionary(name2, dict2);
                throw ex;
            }
            case OUT_OF_ORDER: {
                throw new UserException.IncompatibleSequenceDictionaries("The relative ordering of the common contigs in " + name1 + " and " + name2 + " is not the same; to fix this please see: (https://www.broadinstitute.org/gatk/guide/article?id=1328),  which describes reordering contigs in BAM and VCF files.", name1, dict1, name2, dict2);
            }
            case DIFFERENT_INDICES: {
                String msg = "One or more contigs common to both dictionaries have different indices (ie., absolute positions) in each dictionary. Code that is sensitive to contig ordering can fail when this is the case. You should fix the sequence dictionaries so that all shared contigs occur at the same absolute positions in both dictionaries.";
                throw new UserException.IncompatibleSequenceDictionaries("One or more contigs common to both dictionaries have different indices (ie., absolute positions) in each dictionary. Code that is sensitive to contig ordering can fail when this is the case. You should fix the sequence dictionaries so that all shared contigs occur at the same absolute positions in both dictionaries.", name1, dict1, name2, dict2);
            }
        }
        throw new GATKException("Unexpected SequenceDictionaryComparison type: " + (Object)((Object)type));
    }

    public static SequenceDictionaryCompatibility compareDictionaries(SAMSequenceDictionary dict1, SAMSequenceDictionary dict2, boolean checkContigOrdering) {
        if (checkContigOrdering && (SequenceDictionaryUtils.nonCanonicalHumanContigOrder(dict1) || SequenceDictionaryUtils.nonCanonicalHumanContigOrder(dict2))) {
            return SequenceDictionaryCompatibility.NON_CANONICAL_HUMAN_ORDER;
        }
        Set<String> commonContigs = SequenceDictionaryUtils.getCommonContigsByName(dict1, dict2);
        if (commonContigs.isEmpty()) {
            return SequenceDictionaryCompatibility.NO_COMMON_CONTIGS;
        }
        if (!SequenceDictionaryUtils.commonContigsHaveSameLengths(commonContigs, dict1, dict2)) {
            return SequenceDictionaryCompatibility.UNEQUAL_COMMON_CONTIGS;
        }
        boolean commonContigsAreInSameRelativeOrder = SequenceDictionaryUtils.commonContigsAreInSameRelativeOrder(commonContigs, dict1, dict2);
        if (checkContigOrdering && !commonContigsAreInSameRelativeOrder) {
            return SequenceDictionaryCompatibility.OUT_OF_ORDER;
        }
        if (commonContigsAreInSameRelativeOrder && commonContigs.size() == dict1.size() && commonContigs.size() == dict2.size()) {
            return SequenceDictionaryCompatibility.IDENTICAL;
        }
        if (checkContigOrdering && !SequenceDictionaryUtils.commonContigsAreAtSameIndices(commonContigs, dict1, dict2)) {
            return SequenceDictionaryCompatibility.DIFFERENT_INDICES;
        }
        if (SequenceDictionaryUtils.supersets(dict1, dict2)) {
            return SequenceDictionaryCompatibility.SUPERSET;
        }
        return SequenceDictionaryCompatibility.COMMON_SUBSET;
    }

    private static boolean supersets(SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        for (SAMSequenceRecord dict2Record : dict2.getSequences()) {
            SAMSequenceRecord dict1Record = dict1.getSequence(dict2Record.getSequenceName());
            if (dict1Record != null && SequenceDictionaryUtils.sequenceRecordsAreEquivalent(dict2Record, dict1Record)) continue;
            return false;
        }
        return true;
    }

    private static boolean commonContigsHaveSameLengths(Set<String> commonContigs, SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        return SequenceDictionaryUtils.findDisequalCommonContigs(commonContigs, dict1, dict2) == null;
    }

    private static List<SAMSequenceRecord> findDisequalCommonContigs(Set<String> commonContigs, SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        for (String name : commonContigs) {
            SAMSequenceRecord elt2;
            SAMSequenceRecord elt1 = dict1.getSequence(name);
            if (SequenceDictionaryUtils.sequenceRecordsAreEquivalent(elt1, elt2 = dict2.getSequence(name))) continue;
            return Arrays.asList(elt1, elt2);
        }
        return null;
    }

    public static boolean sequenceRecordsAreEquivalent(SAMSequenceRecord first, SAMSequenceRecord second) {
        int length2;
        if (first == second) {
            return true;
        }
        if (first == null || second == null) {
            return false;
        }
        int length1 = first.getSequenceLength();
        if (length1 != (length2 = second.getSequenceLength()) && length1 != 0 && length2 != 0) {
            return false;
        }
        return first.getSequenceName().equals(second.getSequenceName());
    }

    private static boolean nonCanonicalHumanContigOrder(SAMSequenceDictionary dict) {
        SAMSequenceRecord chr1 = null;
        SAMSequenceRecord chr2 = null;
        SAMSequenceRecord chr10 = null;
        for (SAMSequenceRecord elt : dict.getSequences()) {
            if (SequenceDictionaryUtils.isHumanSeqRecord(elt, CHR1_HG18, CHR1_HG19, CHR1_B36, CHR1_B37)) {
                chr1 = elt;
            }
            if (SequenceDictionaryUtils.isHumanSeqRecord(elt, CHR2_HG18, CHR2_HG19, CHR2_B36, CHR2_B37)) {
                chr2 = elt;
            }
            if (!SequenceDictionaryUtils.isHumanSeqRecord(elt, CHR10_HG18, CHR10_HG19, CHR10_B36, CHR10_B37)) continue;
            chr10 = elt;
        }
        if (chr1 != null && chr2 != null && chr10 != null) {
            return chr1.getSequenceIndex() >= chr2.getSequenceIndex() || chr2.getSequenceIndex() >= chr10.getSequenceIndex();
        }
        return false;
    }

    private static boolean isHumanSeqRecord(SAMSequenceRecord elt, SAMSequenceRecord ... recs) {
        for (SAMSequenceRecord rec : recs) {
            if (elt.getSequenceLength() != rec.getSequenceLength() || !elt.getSequenceName().equals(rec.getSequenceName())) continue;
            return true;
        }
        return false;
    }

    private static boolean commonContigsAreInSameRelativeOrder(Set<String> commonContigs, SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        List<SAMSequenceRecord> list1 = SequenceDictionaryUtils.getSequencesOfName(commonContigs, dict1);
        List<SAMSequenceRecord> list2 = SequenceDictionaryUtils.getSequencesOfName(commonContigs, dict2);
        list1.sort(SEQUENCE_INDEX_ORDER);
        list2.sort(SEQUENCE_INDEX_ORDER);
        for (int i = 0; i < list1.size(); ++i) {
            SAMSequenceRecord elt1 = list1.get(i);
            SAMSequenceRecord elt2 = list2.get(i);
            if (elt1.getSequenceName().equals(elt2.getSequenceName())) continue;
            return false;
        }
        return true;
    }

    private static List<SAMSequenceRecord> getSequencesOfName(Set<String> commonContigs, SAMSequenceDictionary dict) {
        ArrayList<SAMSequenceRecord> l = new ArrayList<SAMSequenceRecord>(commonContigs.size());
        for (String name : commonContigs) {
            l.add(dict.getSequence(name));
        }
        return l;
    }

    private static boolean commonContigsAreAtSameIndices(Set<String> commonContigs, SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        for (String commonContig : commonContigs) {
            SAMSequenceRecord dict1Record = dict1.getSequence(commonContig);
            SAMSequenceRecord dict2Record = dict2.getSequence(commonContig);
            if (dict1Record.getSequenceIndex() == dict2Record.getSequenceIndex()) continue;
            return false;
        }
        return true;
    }

    public static Set<String> getCommonContigsByName(SAMSequenceDictionary dict1, SAMSequenceDictionary dict2) {
        Set<String> intersectingSequenceNames = SequenceDictionaryUtils.getContigNames(dict1);
        intersectingSequenceNames.retainAll(SequenceDictionaryUtils.getContigNames(dict2));
        return intersectingSequenceNames;
    }

    public static Set<String> getContigNames(SAMSequenceDictionary dict) {
        LinkedHashSet<String> contigNames = new LinkedHashSet<String>(Utils.optimumHashSize(dict.size()));
        for (SAMSequenceRecord dictionaryEntry : dict.getSequences()) {
            contigNames.add(dictionaryEntry.getSequenceName());
        }
        return contigNames;
    }

    public static List<String> getContigNamesList(SAMSequenceDictionary refSeqDict) {
        Utils.nonNull(refSeqDict, "provided reference sequence ditionary is null");
        return refSeqDict.getSequences().stream().map(SAMSequenceRecord::getSequenceName).collect(Collectors.toList());
    }

    public static String getDictionaryAsString(SAMSequenceDictionary dict) {
        Utils.nonNull(dict, "Sequence dictionary must be non-null");
        StringBuilder s = new StringBuilder("[ ");
        for (SAMSequenceRecord dictionaryEntry : dict.getSequences()) {
            s.append(dictionaryEntry.getSequenceName());
            s.append("(length:");
            s.append(dictionaryEntry.getSequenceLength());
            s.append(") ");
        }
        s.append("]");
        return s.toString();
    }

    public static enum SequenceDictionaryCompatibility {
        IDENTICAL,
        COMMON_SUBSET,
        SUPERSET,
        NO_COMMON_CONTIGS,
        UNEQUAL_COMMON_CONTIGS,
        NON_CANONICAL_HUMAN_ORDER,
        OUT_OF_ORDER,
        DIFFERENT_INDICES;

    }
}

