/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.bigquery;

import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryError;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldValue;
import com.google.cloud.bigquery.FieldValueList;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.QueryResponse;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableResult;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;
import org.apache.ivy.util.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.utils.bigquery.StorageAPIAvroReader;
import org.broadinstitute.hellbender.utils.bigquery.TableReference;

public final class BigQueryUtils {
    private static final Logger logger = LogManager.getLogger(BigQueryUtils.class);

    private BigQueryUtils() {
    }

    public static BigQuery getBigQueryEndPoint() {
        return (BigQuery)BigQueryOptions.getDefaultInstance().getService();
    }

    public static TableResult executeQuery(String queryString) {
        return BigQueryUtils.executeQuery(BigQueryUtils.getBigQueryEndPoint(), queryString, false);
    }

    public static TableResult executeQuery(String queryString, boolean runQueryInBatchMode) {
        return BigQueryUtils.executeQuery(BigQueryUtils.getBigQueryEndPoint(), queryString, runQueryInBatchMode);
    }

    public static TableResult executeQuery(BigQuery bigQuery, String queryString, boolean runQueryInBatchMode) {
        QueryJobConfiguration queryConfig = QueryJobConfiguration.newBuilder((String)queryString).setUseLegacySql(Boolean.valueOf(false)).setPriority(runQueryInBatchMode ? QueryJobConfiguration.Priority.BATCH : QueryJobConfiguration.Priority.INTERACTIVE).build();
        logger.info("Executing Query: \n\n" + queryString);
        TableResult result = BigQueryUtils.submitQueryAndWaitForResults(bigQuery, queryConfig);
        logger.info("Query returned " + result.getTotalRows() + " results.");
        return result;
    }

    public static TableResult executeQuery(BigQuery bigQuery, String projectID, String dataSet, String queryString) {
        QueryJobConfiguration queryConfig = QueryJobConfiguration.newBuilder((String)queryString).setUseLegacySql(Boolean.valueOf(false)).setDefaultDataset(DatasetId.of((String)projectID, (String)dataSet)).build();
        return BigQueryUtils.submitQueryAndWaitForResults(bigQuery, queryConfig);
    }

    public static String getResultDataPrettyString(TableResult result) {
        Schema schema = result.getSchema();
        List<Integer> columnWidths = BigQueryUtils.calculateColumnWidths(result);
        boolean rowsAllPrimitive = StreamSupport.stream(result.iterateAll().spliterator(), false).flatMap(row -> row.stream().map(v -> v.getAttribute() == FieldValue.Attribute.PRIMITIVE)).allMatch(b -> b);
        String headerFooter = "+" + columnWidths.stream().map(l -> StringUtils.repeat((String)"=", (int)(l + 2)) + "+").collect(Collectors.joining(""));
        String rowSeparator = headerFooter.replace('=', '-');
        StringBuilder sb = new StringBuilder();
        BigQueryUtils.addHeaderToStringBuilder(schema, columnWidths, headerFooter, sb);
        for (FieldValueList row2 : result.iterateAll()) {
            if (rowsAllPrimitive) {
                BigQueryUtils.addPrimitiveRowToStringBuilder(row2, columnWidths, sb);
            } else {
                BigQueryUtils.addComplexRowToStringBuilder(row2, schema, columnWidths, sb);
                sb.append(rowSeparator);
            }
            sb.append("\n");
        }
        sb.append(headerFooter);
        sb.append("\n");
        return sb.toString();
    }

    private static void addHeaderToStringBuilder(Schema schema, List<Integer> columnWidths, String headerFooter, StringBuilder sb) {
        sb.append(headerFooter);
        sb.append("\n");
        sb.append("|");
        sb.append(IntStream.range(0, columnWidths.size()).boxed().map(i -> String.format(" %-" + columnWidths.get((int)i) + "s |", schema.getFields().get(i.intValue()).getName())).collect(Collectors.joining()));
        sb.append("\n");
        sb.append(headerFooter);
        sb.append("\n");
    }

    private static void addComplexRowToStringBuilder(FieldValueList row, Schema schema, List<Integer> columnWidths, StringBuilder sb) {
        int maxNumValuesInRow = 1;
        for (int i = 0; i < row.size(); ++i) {
            FieldValue value = row.get(schema.getFields().get(i).getName());
            if (value.isNull() || value.getAttribute() == FieldValue.Attribute.PRIMITIVE || maxNumValuesInRow > value.getRecordValue().size()) continue;
            maxNumValuesInRow = value.getRecordValue().size();
        }
        for (int currentFieldNum = 0; currentFieldNum < maxNumValuesInRow; ++currentFieldNum) {
            sb.append("|");
            for (int i = 0; i < row.size(); ++i) {
                FieldValue value = row.get(i);
                if (value.isNull()) {
                    sb.append(BigQueryUtils.getEmptyColumnString(columnWidths, i));
                    continue;
                }
                if (value.getAttribute() == FieldValue.Attribute.PRIMITIVE) {
                    if (currentFieldNum == 0) {
                        sb.append(String.format(" %-" + columnWidths.get(i) + "s |", value.getStringValue()));
                        continue;
                    }
                    sb.append(BigQueryUtils.getEmptyColumnString(columnWidths, i));
                    continue;
                }
                if (value.getRepeatedValue().size() == 0) {
                    sb.append(BigQueryUtils.getEmptyColumnString(columnWidths, i));
                    continue;
                }
                if (currentFieldNum < value.getRepeatedValue().size()) {
                    if (((FieldValue)value.getRepeatedValue().get(currentFieldNum)).getAttribute() == FieldValue.Attribute.PRIMITIVE) {
                        sb.append(String.format(" %-" + columnWidths.get(i) + "s |", ((FieldValue)value.getRepeatedValue().get(currentFieldNum)).getStringValue()));
                        continue;
                    }
                    sb.append(String.format(" %-" + columnWidths.get(i) + "s |", ((FieldValue)value.getRepeatedValue().get(currentFieldNum)).getRecordValue().get(0).getStringValue()));
                    continue;
                }
                sb.append(BigQueryUtils.getEmptyColumnString(columnWidths, i));
            }
            sb.append("\n");
        }
    }

    private static String getEmptyColumnString(List<Integer> columnWidths, int i) {
        return String.format(" %-" + columnWidths.get(i) + "s |", "");
    }

    private static void addPrimitiveRowToStringBuilder(FieldValueList row, List<Integer> columnWidths, StringBuilder sb) {
        sb.append("|");
        sb.append(IntStream.range(0, row.size()).boxed().map(i -> String.format(" %-" + columnWidths.get((int)i) + "s |", row.get(i.intValue()).getStringValue())).collect(Collectors.joining()));
    }

    public static void logResultDataPretty(TableResult result, Logger theLogger) {
        for (String line : BigQueryUtils.getResultDataPrettyString(result).split("\n")) {
            theLogger.info(line);
        }
    }

    private static List<Integer> calculateColumnWidths(TableResult result) {
        ArrayList<Integer> columnWidths = new ArrayList<Integer>(result.getSchema().getFields().size());
        for (Field field : result.getSchema().getFields()) {
            columnWidths.add(field.getName().length());
        }
        for (FieldValueList row : result.iterateAll()) {
            for (int i = 0; i < row.size(); ++i) {
                if (row.get(i).isNull()) continue;
                if (row.get(i).getAttribute() == FieldValue.Attribute.PRIMITIVE) {
                    if ((Integer)columnWidths.get(i) >= row.get(i).getStringValue().length()) continue;
                    columnWidths.set(i, row.get(i).getStringValue().length());
                    continue;
                }
                for (int j = 0; j < row.get(i).getRepeatedValue().size(); ++j) {
                    String stringValue = ((FieldValue)row.get(i).getRepeatedValue().get(j)).getAttribute() == FieldValue.Attribute.PRIMITIVE ? ((FieldValue)row.get(i).getRepeatedValue().get(j)).getStringValue() : ((FieldValue)row.get(i).getRepeatedValue().get(j)).getRecordValue().get(0).getStringValue();
                    if ((Integer)columnWidths.get(i) >= stringValue.length()) continue;
                    columnWidths.set(i, stringValue.length());
                }
            }
        }
        return columnWidths;
    }

    private static TableResult submitQueryAndWaitForResults(BigQuery bigQuery, QueryJobConfiguration queryJobConfiguration) {
        TableResult result;
        JobId jobId = JobId.of((String)UUID.randomUUID().toString());
        logger.info("Sending query to server...");
        Job queryJob = bigQuery.create(JobInfo.newBuilder((JobConfiguration)queryJobConfiguration).setJobId(jobId).build(), new BigQuery.JobOption[0]);
        try {
            logger.info("Waiting for query to complete...");
            queryJob = queryJob.waitFor(new RetryOption[0]);
        }
        catch (InterruptedException ex) {
            throw new GATKException("Interrupted while waiting for query job to complete", ex);
        }
        if (queryJob == null) {
            throw new GATKException("Query job no longer exists");
        }
        if (queryJob.getStatus().getError() != null) {
            for (BigQueryError bigQueryError : queryJob.getStatus().getExecutionErrors()) {
                logger.error("Encountered BigQuery Execution Error: " + bigQueryError.toString());
            }
            throw new GATKException(queryJob.getStatus().getError().toString());
        }
        logger.info("Retrieving query results...");
        QueryResponse response = bigQuery.getQueryResults(jobId, new BigQuery.QueryResultsOption[0]);
        try {
            result = queryJob.getQueryResults(new BigQuery.QueryResultsOption[0]);
            long bytesProcessed = ((JobStatistics.QueryStatistics)queryJob.getStatistics()).getTotalBytesProcessed();
            logger.info(String.format("%.2f MB actually scanned", (double)bytesProcessed / 1000000.0));
        }
        catch (InterruptedException ex) {
            throw new GATKException("Interrupted while waiting for query job to complete", ex);
        }
        return result;
    }

    private static long getQueryCostBytesProcessedEstimate(String queryString) {
        QueryJobConfiguration dryRunQueryConfig = QueryJobConfiguration.newBuilder((String)queryString).setUseLegacySql(Boolean.valueOf(false)).setDryRun(Boolean.valueOf(true)).setUseQueryCache(Boolean.valueOf(false)).setPriority(QueryJobConfiguration.Priority.INTERACTIVE).build();
        Job dryRunJob = BigQueryUtils.getBigQueryEndPoint().create(JobInfo.newBuilder((JobConfiguration)dryRunQueryConfig).build(), new BigQuery.JobOption[0]);
        long bytesProcessed = ((JobStatistics.QueryStatistics)dryRunJob.getStatistics()).getTotalBytesProcessed();
        return bytesProcessed;
    }

    public static StorageAPIAvroReader executeQueryWithStorageAPI(String queryString, List<String> fieldsToRetrieve, String projectID) {
        return BigQueryUtils.executeQueryWithStorageAPI(queryString, fieldsToRetrieve, projectID, false);
    }

    public static StorageAPIAvroReader executeQueryWithStorageAPI(String queryString, List<String> fieldsToRetrieve, String projectID, boolean runQueryInBatchMode) {
        String tempTableDataset = "temp_tables";
        String tempTableName = UUID.randomUUID().toString().replace('-', '_');
        String tempTableFullyQualified = String.format("%s.%s.%s", projectID, "temp_tables", tempTableName);
        long bytesProcessed = BigQueryUtils.getQueryCostBytesProcessedEstimate(queryString);
        logger.info(String.format("Estimated %s MB scanned", bytesProcessed / 1000000L));
        String queryStringIntoTempTable = "CREATE TABLE `" + tempTableFullyQualified + "`\nOPTIONS(\n  expiration_timestamp=TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 1 DAY)\n) AS\n" + queryString;
        BigQueryUtils.executeQuery(queryStringIntoTempTable, runQueryInBatchMode);
        Table tableInfo = BigQueryUtils.getBigQueryEndPoint().getTable(TableId.of((String)projectID, (String)"temp_tables", (String)tempTableName), new BigQuery.TableOption[0]);
        logger.info(String.format("Query temp table created with %s rows and %s bytes in size", tableInfo.getNumRows(), tableInfo.getNumBytes()));
        TableReference tr = new TableReference(tempTableFullyQualified, fieldsToRetrieve);
        return new StorageAPIAvroReader(tr);
    }
}

