/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.clipping;

import htsjdk.samtools.Cigar;
import htsjdk.samtools.CigarElement;
import htsjdk.samtools.CigarOperator;
import java.util.Arrays;
import java.util.List;
import org.broadinstitute.hellbender.utils.Nucleotide;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.clipping.ClippingRepresentation;
import org.broadinstitute.hellbender.utils.read.CigarUtils;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.ReadUtils;

public final class ClippingOp {
    public final int start;
    public final int stop;

    public ClippingOp(int start, int stop) {
        this.start = start;
        this.stop = stop;
    }

    public int getLength() {
        return this.stop - this.start + 1;
    }

    GATKRead apply(ClippingRepresentation algorithm, GATKRead originalRead) {
        switch (algorithm) {
            case WRITE_NS: {
                GATKRead readCopied = originalRead.copy();
                this.applyWriteNs(readCopied);
                return readCopied;
            }
            case WRITE_Q0S: {
                GATKRead readCopied = originalRead.copy();
                this.applyWriteQ0s(readCopied);
                return readCopied;
            }
            case WRITE_NS_Q0S: {
                GATKRead readCopied = originalRead.copy();
                this.applyWriteNs(readCopied);
                this.applyWriteQ0s(readCopied);
                return readCopied;
            }
            case HARDCLIP_BASES: {
                return this.applyHardClipBases(originalRead, this.start, this.stop);
            }
            case SOFTCLIP_BASES: {
                return this.applySoftClipBases(originalRead.copy());
            }
            case REVERT_SOFTCLIPPED_BASES: {
                return this.applyRevertSoftClippedBases(originalRead.copy());
            }
        }
        throw new IllegalStateException("Unexpected Clipping operator type " + (Object)((Object)algorithm));
    }

    private GATKRead applySoftClipBases(GATKRead readCopied) {
        Utils.validateArg(!readCopied.isUnmapped(), "Read Clipper cannot soft clip unmapped reads");
        if (readCopied.getLength() <= 2) {
            return readCopied;
        }
        int myStop = Math.min(this.stop, this.start + readCopied.getLength() - 2);
        Utils.validate(this.start <= 0 || myStop == readCopied.getLength() - 1, () -> String.format("Cannot apply soft clipping operator to the middle of a read: %s to be clipped at %d-%d", readCopied.getName(), this.start, myStop));
        Cigar oldCigar = readCopied.getCigar();
        Cigar newCigar = CigarUtils.clipCigar(oldCigar, this.start, myStop + 1, CigarOperator.SOFT_CLIP);
        readCopied.setCigar(newCigar);
        int alignmentStartShift = this.start == 0 ? CigarUtils.alignmentStartShift(oldCigar, this.stop + 1) : 0;
        int newStart = readCopied.getStart() + alignmentStartShift;
        readCopied.setPosition(readCopied.getContig(), newStart);
        return readCopied;
    }

    private void applyWriteQ0s(GATKRead readCopied) {
        byte[] newQuals = readCopied.getBaseQualities();
        this.overwriteFromStartToStop(newQuals, (byte)0);
        readCopied.setBaseQualities(newQuals);
    }

    private void applyWriteNs(GATKRead readCopied) {
        byte[] newBases = readCopied.getBases();
        this.overwriteFromStartToStop(newBases, Nucleotide.N.encodeAsByte());
        readCopied.setBases(newBases);
    }

    private void overwriteFromStartToStop(byte[] arr, byte newVal) {
        Arrays.fill(arr, this.start, Math.min(arr.length, this.stop + 1), newVal);
    }

    private GATKRead applyRevertSoftClippedBases(GATKRead read) {
        Cigar originalCigar = read.getCigar();
        List originalElements = originalCigar.getCigarElements();
        if (originalElements.isEmpty() || !((CigarElement)originalElements.get(0)).getOperator().isClipping() && !((CigarElement)originalElements.get(originalElements.size() - 1)).getOperator().isClipping()) {
            return read;
        }
        GATKRead unclipped = read.copy();
        Cigar unclippedCigar = CigarUtils.revertSoftClips(originalCigar);
        unclipped.setCigar(unclippedCigar);
        int newStart = read.getSoftStart();
        if (newStart <= 0) {
            unclipped.setPosition(unclipped.getContig(), 1);
            unclipped = this.applyHardClipBases(unclipped, 0, -newStart);
            if (!unclipped.isUnmapped()) {
                unclipped.setPosition(unclipped.getContig(), 1);
            }
            return unclipped;
        }
        unclipped.setPosition(unclipped.getContig(), newStart);
        return unclipped;
    }

    private GATKRead applyHardClipBases(GATKRead read, int start, int stop) {
        int newLength = read.getLength() - (stop - start + 1);
        if (newLength == 0) {
            return ReadUtils.emptyRead(read);
        }
        Cigar cigar = read.getCigar();
        Cigar newCigar = read.isUnmapped() ? new Cigar() : CigarUtils.clipCigar(cigar, start, stop + 1, CigarOperator.HARD_CLIP);
        byte[] newBases = new byte[newLength];
        byte[] newQuals = new byte[newLength];
        int copyStart = start == 0 ? stop + 1 : 0;
        System.arraycopy(read.getBases(), copyStart, newBases, 0, newLength);
        System.arraycopy(read.getBaseQualities(), copyStart, newQuals, 0, newLength);
        GATKRead hardClippedRead = read.copy();
        hardClippedRead.setBaseQualities(newQuals);
        hardClippedRead.setBases(newBases);
        hardClippedRead.setCigar(newCigar);
        if (start == 0 && !read.isUnmapped()) {
            hardClippedRead.setPosition(read.getContig(), read.getStart() + CigarUtils.alignmentStartShift(cigar, stop + 1));
        }
        if (ReadUtils.hasBaseIndelQualities(read)) {
            byte[] newBaseInsertionQuals = new byte[newLength];
            byte[] newBaseDeletionQuals = new byte[newLength];
            System.arraycopy(ReadUtils.getBaseInsertionQualities(read), copyStart, newBaseInsertionQuals, 0, newLength);
            System.arraycopy(ReadUtils.getBaseDeletionQualities(read), copyStart, newBaseDeletionQuals, 0, newLength);
            ReadUtils.setInsertionBaseQualities(hardClippedRead, newBaseInsertionQuals);
            ReadUtils.setDeletionBaseQualities(hardClippedRead, newBaseDeletionQuals);
        }
        return hardClippedRead;
    }
}

