/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.dragstr;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.dragstr.DragstrParams;
import org.broadinstitute.hellbender.utils.functional.IntToDoubleBiFunction;

public final class DragstrParamUtils {
    public static final String GOP_TABLE_NAME = "GOP";
    public static final String GCP_TABLE_NAME = "GCP";
    public static final String API_TABLE_NAME = "API";
    private static int LINE_BUILDER_BUFFER_SIZE = 1024;

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static DragstrParams parse(GATKPath path) {
        if (path == null) {
            return null;
        }
        try (BufferedReader reader = DragstrParamUtils.openBufferedReader(path.toString());){
            DragstrParams dragstrParams = DragstrParamUtils.parse(reader, path.toString());
            return dragstrParams;
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile(path.toString(), (Exception)e);
        }
    }

    public static void print(DragstrParams params, GATKPath path, Object ... annotations) {
        Utils.nonNull(params, "the input params cannot be null");
        Utils.nonNull(path, "the input path cannot be null");
        try (PrintWriter pw = new PrintWriter(path.getOutputStream());){
            DragstrParamUtils.print(params, pw, annotations);
        }
        params.setName(path.toString());
    }

    private static void print(DragstrParams params, PrintWriter writer, Object ... annotations) {
        writer.println("############################################################################################");
        writer.println("# DragstrParams");
        writer.println("# -------------------------");
        int i = 0;
        while (i < annotations.length) {
            Object name = annotations[i++];
            Object value = i < annotations.length ? annotations[i++] : null;
            writer.println("# " + name + (value != null ? " = " + value : ""));
        }
        writer.println("############################################################################################");
        DragstrParamUtils.printTables(params, writer);
    }

    private static BufferedReader openBufferedReader(String path) {
        try {
            return Files.newBufferedReader(Paths.get(path, new String[0]));
        }
        catch (IOException ex) {
            throw new UserException.CouldNotReadInputFile(path, (Exception)ex);
        }
    }

    private static DragstrParams parse(BufferedReader reader, String name) {
        try {
            String header;
            while ((header = reader.readLine()) != null && header.startsWith("#")) {
            }
            if (header == null) {
                throw new UserException.BadInput("there is no content in the dragstr-params file " + name);
            }
            String[] headerParts = header.split("\\s+");
            int[] repeats = Arrays.stream(headerParts).filter(str -> !str.isEmpty()).mapToInt(str -> {
                try {
                    return Integer.parseInt(str);
                }
                catch (NumberFormatException ex) {
                    throw new UserException.BadInput("bad format for an integer", ex);
                }
            }).toArray();
            int maxRepeats = repeats.length;
            for (int i = 0; i < repeats.length; ++i) {
                if (repeats[i] == i + 1) continue;
                throw new UserException.BadInput("the DRAGstr parameter file header line must contain integers starting at 1 " + Arrays.toString(repeats));
            }
            HashMap<String, double[][]> tables = new HashMap<String, double[][]>();
            String line = reader.readLine();
            if (line == null) {
                throw new UserException.BadInput("end of table list before expected");
            }
            String tableName = line.replaceAll(":$", "");
            ArrayList<String> tableLines = new ArrayList<String>();
            while ((line = reader.readLine()) != null) {
                if (line.charAt(line.length() - 1) == ':') {
                    tables.put(tableName, DragstrParamUtils.linesToMatrix(tableLines, repeats.length));
                    tableName = line.replaceAll(":$", "");
                    tableLines.clear();
                    continue;
                }
                tableLines.add(line);
            }
            if (tableName.isEmpty()) {
                throw new UserException.BadInput("table with no name");
            }
            tables.put(tableName, DragstrParamUtils.linesToMatrix(tableLines, repeats.length));
            double[][] gopMatrix = DragstrParamUtils.mandatoryMatrix(tables, GOP_TABLE_NAME);
            double[][] gcpMatrix = DragstrParamUtils.mandatoryMatrix(tables, GCP_TABLE_NAME);
            double[][] apiMatrix = DragstrParamUtils.mandatoryMatrix(tables, API_TABLE_NAME);
            int maxPeriod = gopMatrix.length;
            return DragstrParams.of(maxPeriod, maxRepeats, gopMatrix, gcpMatrix, apiMatrix, name);
        }
        catch (IOException ex) {
            throw new UserException.CouldNotReadInputFile(name, (Exception)ex);
        }
    }

    private static void printTable(PrintWriter printWriter, StringBuilder lineBuilder, String tableName, int maxPeriod, int maxRepeat, IntToDoubleBiFunction func) {
        printWriter.println(tableName + ":");
        for (int i = 1; i <= maxPeriod; ++i) {
            lineBuilder.setLength(0);
            lineBuilder.append(String.format("%5s", String.format("%.2f", func.apply(i, 1))));
            for (int j = 2; j <= maxRepeat; ++j) {
                lineBuilder.append("  ");
                lineBuilder.append(String.format("%5s", String.format("%.2f", func.apply(i, j))));
            }
            printWriter.println(lineBuilder.toString());
        }
    }

    private static void printTables(DragstrParams params, PrintWriter printWriter) {
        StringBuilder lineBuilder = new StringBuilder(LINE_BUILDER_BUFFER_SIZE);
        lineBuilder.append(String.format("%5s", "1"));
        for (int i = 2; i <= params.maximumRepeats(); ++i) {
            lineBuilder.append("  ");
            lineBuilder.append(String.format("%5s", i));
        }
        printWriter.println(lineBuilder.toString());
        DragstrParamUtils.printTable(printWriter, lineBuilder, GOP_TABLE_NAME, params.maximumPeriod(), params.maximumRepeats(), params::gop);
        DragstrParamUtils.printTable(printWriter, lineBuilder, GCP_TABLE_NAME, params.maximumPeriod(), params.maximumRepeats(), params::gcp);
        DragstrParamUtils.printTable(printWriter, lineBuilder, API_TABLE_NAME, params.maximumPeriod(), params.maximumRepeats(), params::api);
    }

    private static double[][] mandatoryMatrix(Map<String, double[][]> tableData, String name) {
        double[][] result = tableData.get(name);
        if (result == null) {
            throw new UserException.BadInput("missing matrix " + name);
        }
        return result;
    }

    private static double[][] linesToMatrix(List<String> lines, int expectedNumberOfColumns) {
        double[][] result = new double[lines.size()][expectedNumberOfColumns];
        for (int i = 0; i < lines.size(); ++i) {
            String line = lines.get(i);
            String[] parts = line.split("\\s+");
            if (parts.length < expectedNumberOfColumns) {
                throw new UserException.BadInput("line has the wrong number of columns");
            }
            int k = 0;
            for (int j = 0; j < parts.length; ++j) {
                if (parts[j].isEmpty()) continue;
                if (k >= expectedNumberOfColumns) {
                    throw new UserException.BadInput("line has the wrong number of columns");
                }
                try {
                    double val = Double.parseDouble(parts[j]);
                    if (Double.isNaN(val) || Double.isInfinite(val) || val < 0.0) {
                        throw new NullPointerException();
                    }
                    result[i][k++] = val;
                    continue;
                }
                catch (NumberFormatException ex) {
                    throw new UserException.BadInput(String.format("score is not a valid Phred value (%d,%d) == %s", i + 1, j + 1, parts[j]));
                }
            }
        }
        return result;
    }
}

