/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.dragstr;

import com.google.common.io.Files;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceDictionaryCodec;
import htsjdk.samtools.SAMSequenceRecord;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.tools.dragstr.DragstrLocus;
import org.broadinstitute.hellbender.tools.dragstr.DragstrLocusUtils;
import org.broadinstitute.hellbender.tools.dragstr.STRDecimationTable;
import org.broadinstitute.hellbender.utils.BinaryTableWriter;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.ZipUtils;
import org.broadinstitute.hellbender.utils.tsv.TableWriter;

public final class STRTableFileBuilder
implements AutoCloseable {
    private boolean closed;
    private final File dir;
    private final Map<String, String> annotations;
    private int maxPeriod;
    private int maxRepeatLength;
    private final BinaryTableWriter<DragstrLocus> sitesWriter;
    private final TableWriter<DragstrLocus> textSitesWriter;
    private final SAMSequenceDictionary dictionary;
    private final STRDecimationTable decimationTable;
    private final long[][] emittedCounts;
    private final long[][] totalCounts;

    private STRTableFileBuilder(File dir, boolean generateTextSitesFile, SAMSequenceDictionary dictionary, STRDecimationTable decimationTable, int maxPeriod, int maxRepeatLength) {
        this.maxPeriod = maxPeriod;
        this.maxRepeatLength = maxRepeatLength;
        this.emittedCounts = new long[maxPeriod + 1][maxRepeatLength + 1];
        this.totalCounts = new long[maxPeriod + 1][maxRepeatLength + 1];
        this.dir = dir;
        this.dictionary = dictionary;
        this.decimationTable = decimationTable;
        this.annotations = new LinkedHashMap<String, String>();
        try {
            this.sitesWriter = DragstrLocusUtils.binaryWriter(new File(dir, "sites.bin"), new File(dir, "sites.idx"));
        }
        catch (FileNotFoundException ex) {
            throw new GATKException("possible bug, the parent directory " + dir + " must exists at this point", ex);
        }
        try {
            this.textSitesWriter = generateTextSitesFile ? DragstrLocusUtils.textWriter(new FileOutputStream(new File(dir, "sites.txt")), dictionary) : null;
        }
        catch (IOException e) {
            throw new GATKException("possible bug", e);
        }
        STRTableFileBuilder.writeReferenceDictionary(dir, dictionary);
        STRTableFileBuilder.writeDecimationTable(dir, decimationTable);
    }

    public static STRTableFileBuilder newInstance(SAMSequenceDictionary dictionary, STRDecimationTable decimationTable, boolean generateTextSitesFile, int maxPeriod, int maxRepeatLength) {
        Utils.validateArg(maxPeriod >= 1, "max period must be positive");
        Utils.validateArg(maxRepeatLength >= 1, "max repeat length must be positive");
        Utils.nonNull(decimationTable, "decimation table must not be negative");
        Utils.nonNull(dictionary, "dictionary must not be negative");
        File tempDir = Files.createTempDir();
        return new STRTableFileBuilder(tempDir, generateTextSitesFile, dictionary, decimationTable, maxPeriod, maxRepeatLength);
    }

    public void annotate(String name, String value) {
        Utils.nonNull(name);
        Utils.nonNull(value);
        this.annotations.put(name, value);
    }

    private static void writeReferenceDictionary(File dir, SAMSequenceDictionary dictionary) {
        File dictionaryFile = new File(dir, "reference.dict");
        try (PrintWriter dictWriter = new PrintWriter(new FileWriter(dictionaryFile));){
            SAMSequenceDictionaryCodec codec = new SAMSequenceDictionaryCodec((Writer)dictWriter);
            codec.encode(dictionary);
        }
        catch (IOException e) {
            throw new GATKException("issues writing dictionary file in stage directory " + dir, e);
        }
    }

    private static void writeDecimationTable(File dir, STRDecimationTable decimationTable) {
        File decimationTableFile = new File(dir, "decimation.txt");
        try (PrintWriter deciWriter = new PrintWriter(new FileWriter(decimationTableFile));){
            decimationTable.print(deciWriter);
        }
        catch (IOException e) {
            throw new GATKException("issues writing dictionary file in stage directory " + dir, e);
        }
    }

    public void decimate(int period, int repeatLength) {
        this.checkIsNotClosed();
        int effectiveRepeatLength = Math.min(this.maxRepeatLength, repeatLength);
        int effectivePeriod = Math.min(this.maxPeriod, period);
        long[] lArray = this.totalCounts[effectivePeriod];
        int n = effectiveRepeatLength;
        lArray[n] = lArray[n] + 1L;
    }

    public void emit(DragstrLocus locus) throws GATKException {
        this.checkIsNotClosed();
        this.checkLocusIsValid(locus);
        int effectiveRepeatLength = Math.min(this.maxRepeatLength, locus.getRepeats());
        int effectivePeriod = Math.min(this.maxPeriod, locus.getPeriod());
        long[] lArray = this.totalCounts[effectivePeriod];
        int n = effectiveRepeatLength;
        lArray[n] = lArray[n] + 1L;
        long[] lArray2 = this.emittedCounts[effectivePeriod];
        int n2 = effectiveRepeatLength;
        lArray2[n2] = lArray2[n2] + 1L;
        try {
            this.sitesWriter.write(locus);
            if (this.textSitesWriter != null) {
                this.textSitesWriter.writeRecord(locus);
            }
        }
        catch (IOException ex) {
            throw new GATKException("issues writing loci to the staging files in " + this.dir, ex);
        }
    }

    private void checkLocusIsValid(DragstrLocus locus) {
        Utils.nonNull(locus, "the locus cannot be null");
        SAMSequenceRecord seq = this.dictionary.getSequence(locus.getChromosomeIndex());
        Utils.nonNull(seq, "the locus chr idx is out of range");
        Utils.validateArg(locus.getStart() >= 1L, "the start coordinate must be positive");
        Utils.validateArg(locus.getEnd() <= (long)seq.getSequenceLength(), "the end position is beyond the seq's end");
    }

    private void writeSummary() {
        File summaryFile = new File(this.dir, "summary.txt");
        try (PrintWriter writer = new PrintWriter(new FileWriter(summaryFile));){
            writer.println("##########################################################################################");
            writer.println("# STRTableSummary");
            writer.println("# ---------------------------------------");
            writer.println("# maxPeriod = " + this.maxPeriod);
            writer.println("# maxRepeatLength = " + this.maxRepeatLength);
            for (String name : this.annotations.keySet()) {
                writer.println("# " + name + " = " + this.annotations.get(name));
            }
            writer.println("##########################################################################################");
            writer.println(String.join((CharSequence)"\t", "period", "repeatLength", "totalCounts", "emittedCounts", "intendedDecimation", "actualDecimation"));
            for (int period = 1; period <= this.maxPeriod; ++period) {
                int repeatLength;
                int n = repeatLength = period == 1 ? 1 : 2;
                while (repeatLength <= this.maxRepeatLength) {
                    long total = this.totalCounts[period][repeatLength];
                    long emitted = this.emittedCounts[period][repeatLength];
                    int decimation = this.decimationTable.decimationBit(period, repeatLength);
                    double actualDecimation = total > 0L ? MathUtils.INV_LOG_2 * (Math.log(total) - Math.log(emitted)) : 0.0;
                    writer.println(Utils.join((CharSequence)"\t", period, repeatLength, total, emitted, decimation, (double)Math.round(actualDecimation * 100.0) / 100.0));
                    ++repeatLength;
                }
            }
        }
        catch (IOException e) {
            throw new GATKException("unexpected issues writing summary file in " + this.dir);
        }
    }

    public void store(GATKPath path) {
        this.checkIsNotClosed();
        try {
            this.sitesWriter.flush();
            if (this.textSitesWriter != null) {
                this.textSitesWriter.flush();
            }
        }
        catch (IOException ex) {
            throw new GATKException("problems flushing the str-table-file content to " + this.dir, ex);
        }
        this.writeSummary();
        try {
            ZipUtils.zip(this.dir, path, new String[0]);
        }
        catch (GATKException ex) {
            throw new GATKException("problems flushing the str-table-file content from " + this.dir + " to " + path, ex.getCause());
        }
    }

    private void checkIsNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("the writer is already closed");
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            try {
                if (this.sitesWriter != null) {
                    this.sitesWriter.close();
                }
                if (this.textSitesWriter != null) {
                    this.textSitesWriter.close();
                }
                if (this.dir.exists()) {
                    FileUtils.deleteDirectory((File)this.dir);
                }
            }
            catch (IOException ex) {
                throw new GATKException("issues finishing writing the sites files in the stage directory " + this.dir);
            }
        }
    }
}

