/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.fasta;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.GZIIndex;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.BaseUtils;
import org.broadinstitute.hellbender.utils.Utils;

public final class CachingIndexedFastaSequenceFile
implements ReferenceSequenceFile {
    protected static final Logger logger = LogManager.getLogger(CachingIndexedFastaSequenceFile.class);
    private final ReferenceSequenceFile sequenceFile;
    private static final boolean PRINT_EFFICIENCY = false;
    private static final int PRINT_FREQUENCY = 10000;
    public static final long DEFAULT_CACHE_SIZE = 1000000L;
    private final long cacheSize;
    private final long cacheMissBackup;
    private final boolean preserveCase;
    private final boolean preserveIUPAC;
    long cacheHits = 0L;
    long cacheMisses = 0L;
    private final Cache cache = new Cache();

    private static ReferenceSequenceFile requireIndex(Path fasta, ReferenceSequenceFile referenceSequenceFile) {
        if (!referenceSequenceFile.isIndexed()) {
            throw new GATKException("Could not load " + fasta.toUri().toString() + " as an indexed fasta despite passing checks before loading.");
        }
        return referenceSequenceFile;
    }

    public CachingIndexedFastaSequenceFile(GATKPath fasta) {
        this(fasta.toPath(), false);
    }

    public CachingIndexedFastaSequenceFile(Path fasta) {
        this(fasta, false);
    }

    public CachingIndexedFastaSequenceFile(Path fasta, boolean preserveAmbiguityCodesAndCapitalization) {
        this(fasta, 1000000L, preserveAmbiguityCodesAndCapitalization, preserveAmbiguityCodesAndCapitalization);
    }

    public CachingIndexedFastaSequenceFile(Path fasta, long cacheSize, boolean preserveCase, boolean preserveIUPAC) {
        CachingIndexedFastaSequenceFile.checkFastaPath(fasta);
        Utils.validate(cacheSize > 0L, () -> "Cache size must be > 0 but was " + cacheSize);
        try {
            ReferenceSequenceFile referenceSequenceFile = ReferenceSequenceFileFactory.getReferenceSequenceFile((Path)fasta, (boolean)true, (boolean)true);
            this.sequenceFile = CachingIndexedFastaSequenceFile.requireIndex(fasta, referenceSequenceFile);
            this.cacheSize = cacheSize;
            this.cacheMissBackup = Math.max(cacheSize / 1000L, 1L);
            this.preserveCase = preserveCase;
            this.preserveIUPAC = preserveIUPAC;
        }
        catch (IllegalArgumentException e) {
            throw new UserException.CouldNotReadInputFile(fasta, "Could not read reference sequence.  The FASTA must have either a .fasta or .fa extension", (Throwable)e);
        }
        catch (Exception e) {
            throw new UserException.CouldNotReadInputFile(fasta, e);
        }
    }

    private static void checkFastaPath(Path fastaPath) {
        if (!Files.exists(fastaPath, new LinkOption[0])) {
            throw new UserException.MissingReference("The specified fasta file (" + fastaPath.toUri() + ") does not exist.");
        }
        Path indexPath = ReferenceSequenceFileFactory.getFastaIndexFileName((Path)fastaPath);
        Path dictPath = ReferenceSequenceFileFactory.getDefaultDictionaryForReferenceSequence((Path)fastaPath);
        if (!Files.exists(indexPath, new LinkOption[0])) {
            throw new UserException.MissingReferenceFaiFile(indexPath, fastaPath);
        }
        if (!Files.exists(dictPath, new LinkOption[0])) {
            throw new UserException.MissingReferenceDictFile(dictPath, fastaPath);
        }
        try {
            Path gziPath = GZIIndex.resolveIndexNameForBgzipFile((Path)fastaPath);
            if (IOUtil.isBlockCompressed((Path)fastaPath, (boolean)true) && !Files.exists(gziPath, new LinkOption[0])) {
                throw new UserException.MissingReferenceGziFile(gziPath, fastaPath);
            }
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile("Couldn't open fasta file: " + fastaPath.toUri().toString() + ".", (Exception)e);
        }
    }

    public void printEfficiency(Level priority) {
        logger.log(priority, String.format("### CachingIndexedFastaReader: hits=%d misses=%d efficiency %.6f%%", this.cacheHits, this.cacheMisses, this.calcEfficiency()));
    }

    public double calcEfficiency() {
        return 100.0 * (double)this.cacheHits / ((double)this.cacheMisses + (double)this.cacheHits * 1.0);
    }

    public long getCacheHits() {
        return this.cacheHits;
    }

    public long getCacheMisses() {
        return this.cacheMisses;
    }

    public long getCacheSize() {
        return this.cacheSize;
    }

    public boolean isPreservingCase() {
        return this.preserveCase;
    }

    public SAMSequenceDictionary getSequenceDictionary() {
        return this.sequenceFile.getSequenceDictionary();
    }

    public ReferenceSequence nextSequence() {
        return this.sequenceFile.nextSequence();
    }

    public void reset() {
        this.sequenceFile.reset();
    }

    public boolean isIndexed() {
        return true;
    }

    public ReferenceSequence getSequence(String contig) {
        SAMSequenceRecord sequence = Utils.nonNull(this.getSequenceDictionary().getSequence(contig), () -> "Contig: " + contig + " not found in sequence dictionary.");
        return this.getSubsequenceAt(contig, 1L, sequence.getSequenceLength());
    }

    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        ReferenceSequence result;
        if (stop - start + 1L > this.cacheSize) {
            ++this.cacheMisses;
            result = this.sequenceFile.getSubsequenceAt(contig, start, stop);
            if (!this.preserveCase) {
                StringUtil.toUpperCase((byte[])result.getBases());
            }
            if (!this.preserveIUPAC) {
                BaseUtils.convertIUPACtoN(result.getBases(), true, start < 1L);
            }
        } else {
            SAMSequenceRecord contigInfo = this.sequenceFile.getSequenceDictionary().getSequence(contig);
            if (contigInfo == null) {
                throw new UserException.MissingContigInSequenceDictionary(contig, this.sequenceFile.getSequenceDictionary());
            }
            if (stop > (long)contigInfo.getSequenceLength()) {
                throw new SAMException("Query asks for data past end of contig. Query contig " + contig + " start:" + start + " stop:" + stop + " contigLength:" + contigInfo.getSequenceLength());
            }
            if (start < this.cache.start || stop > this.cache.stop || this.cache.seq == null || this.cache.seq.getContigIndex() != contigInfo.getSequenceIndex()) {
                ++this.cacheMisses;
                this.cache.start = Math.max(start - this.cacheMissBackup, 0L);
                this.cache.stop = Math.min(start + this.cacheSize + this.cacheMissBackup, (long)contigInfo.getSequenceLength());
                this.cache.seq = this.sequenceFile.getSubsequenceAt(contig, this.cache.start, this.cache.stop);
                if (!this.preserveCase) {
                    StringUtil.toUpperCase((byte[])this.cache.seq.getBases());
                }
                if (!this.preserveIUPAC) {
                    BaseUtils.convertIUPACtoN(this.cache.seq.getBases(), true, this.cache.start == 0L);
                }
            } else {
                ++this.cacheHits;
            }
            int cacheOffsetStart = (int)(start - this.cache.start);
            int cacheOffsetStop = (int)(stop - start + (long)cacheOffsetStart + 1L);
            try {
                result = new ReferenceSequence(this.cache.seq.getName(), this.cache.seq.getContigIndex(), Arrays.copyOfRange(this.cache.seq.getBases(), cacheOffsetStart, cacheOffsetStop));
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new GATKException(String.format("BUG: bad array indexing.  Cache start %d and end %d, request start %d end %d, offset start %d and end %d, base size %d", this.cache.start, this.cache.stop, start, stop, cacheOffsetStart, cacheOffsetStop, this.cache.seq.getBases().length), e);
            }
        }
        return result;
    }

    public void close() {
        try {
            this.sequenceFile.close();
        }
        catch (IOException e) {
            throw new GATKException("Error closing file: " + this.sequenceFile.toString(), e);
        }
    }

    private static class Cache {
        long start = -1L;
        long stop = -1L;
        ReferenceSequence seq = null;

        private Cache() {
        }
    }
}

