/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.gcs;

import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.HttpMethod;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import com.google.cloud.storage.contrib.nio.CloudStorageConfiguration;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystem;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystemProvider;
import com.google.cloud.storage.contrib.nio.SeekableByteChannelPrefetcher;
import com.google.common.base.Strings;
import com.google.common.io.ByteStreams;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.RuntimeIOException;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.io.IOUtils;
import shaded.cloud_nio.com.google.api.gax.retrying.RetrySettings;
import shaded.cloud_nio.com.google.auth.Credentials;
import shaded.cloud_nio.com.google.auth.oauth2.GoogleCredentials;
import shaded.cloud_nio.com.google.cloud.TransportOptions;
import shaded.cloud_nio.com.google.cloud.http.HttpTransportOptions;
import shaded.cloud_nio.org.threeten.bp.Duration;

public final class BucketUtils {
    public static final String GCS_PREFIX = "gs://";
    public static final String HTTP_PREFIX = "http://";
    public static final String HTTPS_PREFIX = "https://";
    public static final String HDFS_SCHEME = "hdfs";
    public static final String HDFS_PREFIX = "hdfs://";
    public static final String FILE_PREFIX = "file:";

    private BucketUtils() {
    }

    public static boolean isGcsUrl(String path) {
        Utils.nonNull(path);
        return path.startsWith(GCS_PREFIX);
    }

    public static boolean isGcsUrl(GATKPath pathSpec) {
        Utils.nonNull(pathSpec);
        return pathSpec.getScheme().equals("gs");
    }

    public static boolean isEligibleForPrefetching(GATKPath pathSpec) {
        Utils.nonNull(pathSpec);
        return BucketUtils.isEligibleForPrefetching(pathSpec.getScheme());
    }

    public static boolean isEligibleForPrefetching(java.nio.file.Path path) {
        Utils.nonNull(path);
        return BucketUtils.isEligibleForPrefetching(path.toUri().getScheme());
    }

    private static boolean isEligibleForPrefetching(String scheme) {
        return scheme != null && (scheme.equals("gs") || scheme.equals("http") || scheme.equals("https"));
    }

    public static boolean isHttpUrl(String path) {
        return path.startsWith(HTTP_PREFIX) || path.startsWith(HTTPS_PREFIX);
    }

    public static boolean isHadoopUrl(String path) {
        return path.startsWith(HDFS_PREFIX);
    }

    public static boolean isRemoteStorageUrl(String path) {
        return BucketUtils.isGcsUrl(path) || BucketUtils.isHadoopUrl(path) || BucketUtils.isHttpUrl(path);
    }

    public static String makeFilePathAbsolute(String path) {
        if (BucketUtils.isGcsUrl(path) || BucketUtils.isHadoopUrl(path) || BucketUtils.isFileUrl(path) || BucketUtils.isHttpUrl(path)) {
            return path;
        }
        return new File(path).getAbsolutePath();
    }

    public static InputStream openFile(String path) {
        try {
            InputStream inputStream;
            Utils.nonNull(path);
            if (BucketUtils.isGcsUrl(path)) {
                java.nio.file.Path p = BucketUtils.getPathOnGcs(path);
                inputStream = Files.newInputStream(p, new OpenOption[0]);
            } else if (BucketUtils.isHadoopUrl(path)) {
                Path file = new Path(path);
                org.apache.hadoop.fs.FileSystem fs = file.getFileSystem(new Configuration());
                inputStream = fs.open(file);
            } else {
                inputStream = new FileInputStream(path);
            }
            if (IOUtil.hasBlockCompressedExtension((String)path)) {
                return IOUtils.makeZippedInputStream(new BufferedInputStream(inputStream));
            }
            return inputStream;
        }
        catch (IOException x) {
            throw new UserException.CouldNotReadInputFile(path, (Exception)x);
        }
    }

    public static OutputStream createFile(String path) {
        Utils.nonNull(path);
        try {
            if (BucketUtils.isGcsUrl(path)) {
                java.nio.file.Path p = BucketUtils.getPathOnGcs(path);
                return Files.newOutputStream(p, new OpenOption[0]);
            }
            if (BucketUtils.isHadoopUrl(path)) {
                Path file = new Path(path);
                org.apache.hadoop.fs.FileSystem fs = file.getFileSystem(new Configuration());
                return fs.create(file);
            }
            return new FileOutputStream(path);
        }
        catch (IOException x) {
            throw new UserException.CouldNotCreateOutputFile("Could not create file at path:" + path + " due to " + x.getMessage(), (Exception)x);
        }
    }

    public static void copyFile(String sourcePath, String destPath) throws IOException {
        try (InputStream in = BucketUtils.openFile(sourcePath);
             OutputStream fout = BucketUtils.createFile(destPath);){
            ByteStreams.copy((InputStream)in, (OutputStream)fout);
        }
    }

    public static void deleteFile(String pathToDelete) throws IOException {
        if (BucketUtils.isGcsUrl(pathToDelete)) {
            java.nio.file.Path p = BucketUtils.getPathOnGcs(pathToDelete);
            Files.delete(p);
        } else if (BucketUtils.isHadoopUrl(pathToDelete)) {
            Path file = new Path(pathToDelete);
            org.apache.hadoop.fs.FileSystem fs = file.getFileSystem(new Configuration());
            fs.delete(file, false);
        } else {
            boolean ok = new File(pathToDelete).delete();
            if (!ok) {
                throw new IOException("Unable to delete '" + pathToDelete + "'");
            }
        }
    }

    public static String getTempFilePath(String prefix, String extension) {
        if (BucketUtils.isGcsUrl(prefix) || BucketUtils.isHadoopUrl(prefix)) {
            String path = BucketUtils.randomRemotePath(prefix, "", extension);
            IOUtils.deleteOnExit(IOUtils.getPath(path));
            IOUtils.deleteOnExit(IOUtils.getPath(path + ".idx"));
            IOUtils.deleteOnExit(IOUtils.getPath(path + ".tbi"));
            IOUtils.deleteOnExit(IOUtils.getPath(path + ".bai"));
            IOUtils.deleteOnExit(IOUtils.getPath(path + ".md5"));
            IOUtils.deleteOnExit(IOUtils.getPath(path.replaceAll(extension + "$", ".bai")));
            return path;
        }
        return IOUtils.createTempFile(prefix, extension).getAbsolutePath();
    }

    public static String randomRemotePath(String stagingLocation, String prefix, String suffix) {
        if (BucketUtils.isGcsUrl(stagingLocation)) {
            return BucketUtils.getPathOnGcs(stagingLocation).resolve(prefix + UUID.randomUUID().toString() + suffix).toUri().toString();
        }
        if (BucketUtils.isHadoopUrl(stagingLocation)) {
            return new Path(stagingLocation, prefix + UUID.randomUUID().toString() + suffix).toString();
        }
        throw new IllegalArgumentException("Staging location is not remote: " + stagingLocation);
    }

    public static boolean fileExists(String path) {
        boolean MAYBE = false;
        try (InputStream inputStream = BucketUtils.openFile(path);){
            int n = inputStream.read();
        }
        catch (UserException.CouldNotReadInputFile notthere) {
            return false;
        }
        catch (FileNotFoundException x) {
            return false;
        }
        catch (IOException x) {
            return false;
        }
        return true;
    }

    public static long fileSize(String path) throws IOException {
        if (BucketUtils.isGcsUrl(path)) {
            java.nio.file.Path p = BucketUtils.getPathOnGcs(path);
            return Files.size(p);
        }
        if (BucketUtils.isHadoopUrl(path)) {
            Path hadoopPath = new Path(path);
            org.apache.hadoop.fs.FileSystem fs = hadoopPath.getFileSystem(new Configuration());
            return fs.getFileStatus(hadoopPath).getLen();
        }
        return new File(path).length();
    }

    public static long dirSize(GATKPath pathSpecifier) {
        try {
            if (BucketUtils.isGcsUrl(pathSpecifier)) {
                java.nio.file.Path p = BucketUtils.getPathOnGcs(pathSpecifier.getRawInputString());
                if (Files.isRegularFile(p, new LinkOption[0])) {
                    return Files.size(p);
                }
                return Files.list(p).mapToLong(q -> {
                    try {
                        return Files.isRegularFile(q, new LinkOption[0]) ? Files.size(q) : 0L;
                    }
                    catch (IOException e) {
                        throw new RuntimeIOException((Throwable)e);
                    }
                }).sum();
            }
            Path hadoopPath = new Path(pathSpecifier.getURIString());
            org.apache.hadoop.fs.FileSystem fs = new Path(pathSpecifier.getURIString()).getFileSystem(new Configuration());
            FileStatus status = fs.getFileStatus(hadoopPath);
            if (status == null) {
                throw new UserException.CouldNotReadInputFile(pathSpecifier.getRawInputString(), "File not found.");
            }
            long size = 0L;
            if (status.isDirectory()) {
                for (FileStatus st : fs.listStatus(status.getPath())) {
                    if (!st.isFile()) continue;
                    size += st.getLen();
                }
            } else {
                size += status.getLen();
            }
            return size;
        }
        catch (RuntimeIOException | IOException e) {
            throw new UserException("Failed to determine total input size of " + pathSpecifier.getRawInputString() + "\n Caused by:" + e.getMessage(), e);
        }
    }

    public static boolean isFileUrl(String path) {
        return path.startsWith(FILE_PREFIX);
    }

    public static String getBucket(String path) {
        return path.split("/")[2];
    }

    public static String getPathWithoutBucket(String path) {
        String[] split = path.split("/");
        return String.join((CharSequence)"/", Arrays.copyOfRange(split, 3, split.length));
    }

    public static void setGlobalNIODefaultOptions(int maxReopens, String requesterProject) {
        CloudStorageFileSystemProvider.setDefaultCloudStorageConfiguration((CloudStorageConfiguration)BucketUtils.getCloudStorageConfiguration(maxReopens, requesterProject));
        CloudStorageFileSystemProvider.setStorageOptions((StorageOptions)BucketUtils.setGenerousTimeouts(StorageOptions.newBuilder()).build());
    }

    public static java.nio.file.Path getPathOnGcs(String gcsUrl) {
        String[] split = gcsUrl.split("/", -1);
        String BUCKET = split[2];
        String pathWithoutBucket = String.join((CharSequence)"/", Arrays.copyOfRange(split, 3, split.length));
        return CloudStorageFileSystem.forBucket((String)BUCKET).getPath(pathWithoutBucket, new String[0]);
    }

    public static CloudStorageConfiguration getCloudStorageConfiguration(int maxReopens, String requesterProject) {
        CloudStorageConfiguration.Builder builder = CloudStorageConfiguration.builder().maxChannelReopens(maxReopens);
        if (!Strings.isNullOrEmpty((String)requesterProject)) {
            builder = builder.autoDetectRequesterPays(true).userProject(requesterProject);
        }
        builder.usePseudoDirectories(true);
        return builder.build();
    }

    private static StorageOptions.Builder setGenerousTimeouts(StorageOptions.Builder builder) {
        return (StorageOptions.Builder)builder.setTransportOptions((TransportOptions)HttpTransportOptions.newBuilder().setConnectTimeout(120000).setReadTimeout(120000).build()).setRetrySettings(RetrySettings.newBuilder().setMaxAttempts(15).setMaxRetryDelay(Duration.ofMillis((long)256000L)).setTotalTimeout(Duration.ofMillis((long)4000000L)).setInitialRetryDelay(Duration.ofMillis((long)1000L)).setRetryDelayMultiplier(2.0).setInitialRpcTimeout(Duration.ofMillis((long)180000L)).setRpcTimeoutMultiplier(1.0).setMaxRpcTimeout(Duration.ofMillis((long)180000L)).build());
    }

    public static FileSystem getAuthenticatedGcs(String projectId, String bucket, byte[] credentials) throws IOException {
        StorageOptions.Builder builder = (StorageOptions.Builder)StorageOptions.newBuilder().setProjectId(projectId);
        if (null != credentials) {
            builder = (StorageOptions.Builder)builder.setCredentials((Credentials)GoogleCredentials.fromStream((InputStream)new ByteArrayInputStream(credentials)));
        }
        StorageOptions storageOptions = BucketUtils.setGenerousTimeouts(builder).build();
        return CloudStorageFileSystem.forBucket((String)bucket, (CloudStorageConfiguration)CloudStorageConfiguration.DEFAULT, (StorageOptions)storageOptions);
    }

    public static SeekableByteChannel addPrefetcher(int bufferSizeMB, SeekableByteChannel channel) {
        try {
            return SeekableByteChannelPrefetcher.addPrefetcher((int)bufferSizeMB, (SeekableByteChannel)channel);
        }
        catch (IOException ex) {
            throw new GATKException("Unable to initialize the prefetcher: " + ex);
        }
    }

    public static Function<SeekableByteChannel, SeekableByteChannel> getPrefetchingWrapper(int cloudPrefetchBuffer) {
        return cloudPrefetchBuffer > 0 ? rawChannel -> BucketUtils.addPrefetcher(cloudPrefetchBuffer, rawChannel) : Utils.identityFunction();
    }

    public static String createSignedUrlToGcsObject(String path, long hoursToLive) {
        Storage storage = (Storage)StorageOptions.getDefaultInstance().getService();
        BlobInfo info = BlobInfo.newBuilder((String)BucketUtils.getBucket(path), (String)BucketUtils.getPathWithoutBucket(path)).build();
        URL signedUrl = storage.signUrl(info, hoursToLive, TimeUnit.HOURS, new Storage.SignUrlOption[]{Storage.SignUrlOption.httpMethod((HttpMethod)HttpMethod.GET)});
        return signedUrl.toString();
    }

    public static String bucketPathToPublicHttpUrl(String path) {
        return String.format("https://storage.googleapis.com/%s/%s", BucketUtils.getBucket(path), BucketUtils.getPathWithoutBucket(path));
    }
}

