/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.io;

import com.google.cloud.storage.StorageException;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.IOUtil;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.GATKPath;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.gcs.BucketUtils;
import org.broadinstitute.hellbender.utils.io.DeleteRecursivelyOnExitPathHook;
import org.broadinstitute.hellbender.utils.io.Resource;
import org.broadinstitute.hellbender.utils.runtime.ProcessController;
import org.broadinstitute.hellbender.utils.runtime.ProcessOutput;
import org.broadinstitute.hellbender.utils.runtime.ProcessSettings;

public final class IOUtils {
    private static final Logger logger = LogManager.getLogger(IOUtils.class);
    private static final File DEV_DIR = new File("/dev");
    private static final byte[] hdf5HeaderSignature = new byte[]{-119, 72, 68, 70, 13, 10, 26, 10};
    public static final String GENOMIC_DB_URI_SCHEME = "gendb";
    private static final Pattern GENOMICSDB_URI_PATTERN = Pattern.compile("^gendb(\\.?)(.*)(://)(.*)");

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isHDF5File(Path hdf5Candidate) {
        try (DataInputStream candidateStream = new DataInputStream(Files.newInputStream(hdf5Candidate, new OpenOption[0]));){
            byte[] candidateHeader = new byte[hdf5HeaderSignature.length];
            candidateStream.read(candidateHeader, 0, candidateHeader.length);
            boolean bl = Arrays.equals(candidateHeader, hdf5HeaderSignature);
            return bl;
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile(String.format("I/O error reading from input stream %s", hdf5Candidate), (Exception)e);
        }
    }

    public static File createTempDir(String prefix) {
        try {
            Path tmpDir = Files.createTempDirectory(prefix, new FileAttribute[0]).normalize();
            IOUtils.deleteOnExit(tmpDir);
            return tmpDir.toFile();
        }
        catch (IOException | SecurityException e) {
            throw new UserException.BadTempDir(e.getMessage(), e);
        }
    }

    public static File writeTempFile(String content, String prefix, String suffix) {
        return IOUtils.writeTempFile(content, prefix, suffix, null);
    }

    public static File writeTempFile(String content, String prefix, String suffix, File directory) {
        try {
            File tempFile = File.createTempFile(prefix, suffix, directory).toPath().normalize().toFile();
            FileUtils.writeStringToFile((File)tempFile, (String)content, (Charset)StandardCharsets.UTF_8);
            return tempFile;
        }
        catch (IOException e) {
            throw new UserException.BadTempDir(e.getMessage(), e);
        }
    }

    public static File writeTempFile(List<String> content, String prefix, String suffix) {
        try {
            File tempFile = IOUtils.createTempFile(prefix, suffix);
            FileUtils.writeLines((File)tempFile, content);
            return tempFile;
        }
        catch (IOException e) {
            throw new UserException.BadTempDir(e.getMessage(), e);
        }
    }

    public static boolean isSpecialFile(File file) {
        return file != null && (file.getAbsolutePath().startsWith("/dev/") || file.equals(DEV_DIR));
    }

    public static boolean tryDelete(File file) {
        if (IOUtils.isSpecialFile(file)) {
            logger.debug("Not trying to delete " + file);
            return false;
        }
        boolean deleted = FileUtils.deleteQuietly((File)file);
        if (deleted) {
            logger.debug("Deleted " + file);
        } else if (file.exists()) {
            logger.warn("Unable to delete " + file);
        }
        return deleted;
    }

    public static File writeTempResource(Resource resource) {
        File tempFile = IOUtils.createTempFile(FilenameUtils.getBaseName((String)resource.getPath()) + ".", "." + FilenameUtils.getExtension((String)resource.getPath()));
        IOUtils.writeResource(resource, tempFile);
        return tempFile;
    }

    public static File writeTempResourceFromPath(String resourcePath, Class<?> relativeClass) {
        Utils.nonNull(resourcePath, "A resource path must be provided");
        Resource resource = new Resource(resourcePath, relativeClass);
        return IOUtils.writeTempResource(resource);
    }

    public static void writeResource(Resource resource, File file) {
        String path = resource.getPath();
        InputStream inputStream = resource.getResourceContentsAsStream();
        FileOutputStream outputStream = null;
        try {
            outputStream = FileUtils.openOutputStream((File)file);
            org.apache.commons.io.IOUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
        }
        catch (IOException e) {
            throw new GATKException(String.format("Unable to copy resource '%s' to '%s'", path, file), e);
        }
        finally {
            org.apache.commons.io.IOUtils.closeQuietly((InputStream)inputStream);
            org.apache.commons.io.IOUtils.closeQuietly((OutputStream)outputStream);
        }
    }

    public static byte[] readFileIntoByteArray(File source) {
        return IOUtils.readFileIntoByteArray(source, 4096);
    }

    public static byte[] readFileIntoByteArray(File source, int readBufferSize) {
        byte[] fileContents;
        if (source == null) {
            throw new GATKException("Source file was null");
        }
        try {
            fileContents = IOUtils.readStreamIntoByteArray(new FileInputStream(source), readBufferSize);
        }
        catch (FileNotFoundException e) {
            throw new UserException.CouldNotReadInputFile(source.getAbsolutePath(), (Exception)e);
        }
        if ((long)fileContents.length != source.length()) {
            throw new UserException.CouldNotReadInputFile(String.format("Unable to completely read file %s: read only %d/%d bytes", source.getAbsolutePath(), fileContents.length, source.length()));
        }
        return fileContents;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] readStreamIntoByteArray(InputStream in, int readBufferSize) {
        if (in == null) {
            throw new IllegalArgumentException("Input stream was null");
        }
        if (readBufferSize <= 0) {
            throw new IllegalArgumentException("Read buffer size must be > 0");
        }
        byte[] readBuffer = new byte[readBufferSize];
        ByteArrayOutputStream fileBuffer = new ByteArrayOutputStream(readBufferSize * 4);
        try {
            try {
                int currentBytesRead;
                while ((currentBytesRead = in.read(readBuffer, 0, readBuffer.length)) >= 0) {
                    fileBuffer.write(readBuffer, 0, currentBytesRead);
                }
            }
            finally {
                in.close();
            }
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile("I/O error reading from input stream", (Exception)e);
        }
        return fileBuffer.toByteArray();
    }

    public static void writeByteArrayToFile(byte[] bytes, File destination) {
        if (destination == null) {
            throw new GATKException("Destination file was null");
        }
        try {
            IOUtils.writeByteArrayToStream(bytes, new FileOutputStream(destination));
        }
        catch (FileNotFoundException e) {
            throw new UserException.CouldNotCreateOutputFile(destination, (Exception)e);
        }
    }

    public static void writeByteArrayToStream(byte[] bytes, OutputStream out) {
        if (bytes == null || out == null) {
            throw new GATKException("Data to write or output stream was null");
        }
        try {
            try {
                out.write(bytes);
            }
            finally {
                out.close();
            }
        }
        catch (IOException e) {
            throw new UserException.CouldNotCreateOutputFile("I/O error writing to output stream", (Exception)e);
        }
    }

    public static void gunzip(File input, File output) {
        try (GZIPInputStream in = new GZIPInputStream(new FileInputStream(input));
             FileOutputStream out = new FileOutputStream(output);){
            int len;
            byte[] buf = new byte[4096];
            while ((len = in.read(buf)) > 0) {
                ((OutputStream)out).write(buf, 0, len);
            }
        }
        catch (IOException e) {
            throw new GATKException("Exception while unzipping a file:" + input + " to:" + output, e);
        }
    }

    public static File gunzipToTempIfNeeded(File maybeGzipedFile) {
        if (!maybeGzipedFile.getPath().endsWith(".gz")) {
            return maybeGzipedFile;
        }
        File result = IOUtils.createTempFile("unzippedFile", "tmp");
        IOUtils.gunzip(maybeGzipedFile, result);
        return result;
    }

    public static Reader makeReaderMaybeGzipped(Path path) throws IOException {
        BufferedInputStream in = new BufferedInputStream(Files.newInputStream(path, new OpenOption[0]));
        return IOUtils.makeReaderMaybeGzipped(in, path.toString().endsWith(".gz"));
    }

    public static Reader makeReaderMaybeGzipped(InputStream in, boolean zipped) throws IOException {
        if (zipped) {
            return new InputStreamReader(IOUtils.makeZippedInputStream(in));
        }
        return new InputStreamReader(in);
    }

    public static InputStream makeZippedInputStream(InputStream in) throws IOException {
        Utils.nonNull(in);
        if (BlockCompressedInputStream.isValidFile((InputStream)in)) {
            return new BlockCompressedInputStream(in);
        }
        return new GZIPInputStream(in);
    }

    public static void extractTarGz(Path tarGzFilePath) {
        IOUtils.extractTarGz(tarGzFilePath, tarGzFilePath.getParent(), false);
    }

    public static void extractTarGz(Path tarGzFilePath, Path destDir) {
        IOUtils.extractTarGz(tarGzFilePath, destDir, false);
    }

    public static void extractTarGz(Path tarGzFilePath, Path destDir, boolean overwriteExistingFiles) {
        logger.info("Extracting data from archive: " + tarGzFilePath.toUri());
        try (InputStream fi = Files.newInputStream(tarGzFilePath, new OpenOption[0]);
             BufferedInputStream bi = new BufferedInputStream(fi);
             GzipCompressorInputStream gzi = new GzipCompressorInputStream((InputStream)bi);
             TarArchiveInputStream archiveStream = new TarArchiveInputStream((InputStream)gzi);){
            IOUtils.extractFilesFromArchiveStream(archiveStream, tarGzFilePath, destDir, overwriteExistingFiles);
        }
        catch (IOException ex) {
            throw new UserException("Could not extract data from: " + tarGzFilePath.toUri(), ex);
        }
    }

    private static void extractFilesFromArchiveStream(TarArchiveInputStream archiveStream, Path localTarGzPath, Path destDir, boolean overwriteExistingFiles) throws IOException {
        TarArchiveEntry entry;
        while ((entry = archiveStream.getNextTarEntry()) != null) {
            String linkName;
            logger.info("Extracting file: " + entry.getName());
            if (!archiveStream.canReadEntryData((ArchiveEntry)entry)) {
                throw new UserException("Could not read data from archive file(" + localTarGzPath.toUri() + "): " + entry.getName());
            }
            Path extractedEntryPath = destDir.resolve(entry.getName()).normalize();
            IOUtils.ensurePathIsOkForOutput(extractedEntryPath, overwriteExistingFiles);
            if (entry.isDirectory()) {
                Files.createDirectories(extractedEntryPath, new FileAttribute[0]);
                continue;
            }
            Files.createDirectories(extractedEntryPath.getParent(), new FileAttribute[0]);
            if (entry.isFIFO()) {
                IOUtils.createFifoFile(extractedEntryPath, overwriteExistingFiles);
                continue;
            }
            if (entry.isSymbolicLink()) {
                linkName = entry.getLinkName();
                if (Files.exists(extractedEntryPath, new LinkOption[0]) && overwriteExistingFiles) {
                    IOUtils.removeFileWithWarning(extractedEntryPath);
                }
                Files.createSymbolicLink(extractedEntryPath, Paths.get(linkName, new String[0]), new FileAttribute[0]);
                continue;
            }
            if (entry.isLink()) {
                linkName = entry.getLinkName();
                if (Files.exists(extractedEntryPath, new LinkOption[0]) && overwriteExistingFiles) {
                    IOUtils.removeFileWithWarning(extractedEntryPath);
                }
                Files.createLink(extractedEntryPath, Paths.get(linkName, new String[0]));
                continue;
            }
            if (entry.isFile()) {
                OutputStream o = Files.newOutputStream(extractedEntryPath, new OpenOption[0]);
                Throwable throwable = null;
                try {
                    org.apache.commons.io.IOUtils.copy((InputStream)archiveStream, (OutputStream)o);
                    continue;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (o == null) continue;
                    if (throwable != null) {
                        try {
                            o.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    o.close();
                    continue;
                }
            }
            throw new UserException("Cannot extract file from tar.gz (unknown type): " + entry.toString());
        }
    }

    public static void writeTarGz(String name, File ... files) throws IOException {
        try (TarArchiveOutputStream taos = new TarArchiveOutputStream((OutputStream)new GzipCompressorOutputStream((OutputStream)new FileOutputStream(name)));){
            taos.setBigNumberMode(1);
            taos.setLongFileMode(2);
            taos.setAddPaxHeadersForNonAsciiNames(true);
            for (File file : files) {
                IOUtils.addToTar(taos, file, ".");
            }
        }
    }

    private static void addToTar(TarArchiveOutputStream out, File file, String dir) throws IOException {
        String entry = dir + File.separator + file.getName();
        if (file.isFile()) {
            out.putArchiveEntry((ArchiveEntry)new TarArchiveEntry(file, entry));
            try (FileInputStream in = new FileInputStream(file);){
                org.apache.commons.compress.utils.IOUtils.copy((InputStream)in, (OutputStream)out);
            }
            out.closeArchiveEntry();
        } else if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children != null) {
                for (File child : children) {
                    IOUtils.addToTar(out, child, entry);
                }
            }
        } else {
            System.out.println(file.getName() + " is not supported");
        }
    }

    private static void ensurePathIsOkForOutput(Path p, boolean overwriteExistingFiles) {
        if (Files.exists(p, new LinkOption[0])) {
            if (overwriteExistingFiles) {
                logger.warn("Overwriting existing output destination: " + p.toUri());
            } else {
                throw new UserException("Output destination already exists: " + p.toUri());
            }
        }
    }

    public static File createFifoFile(Path fifoFilePath) {
        return IOUtils.createFifoFile(fifoFilePath, false);
    }

    private static void removeFileWithWarning(Path filePath) {
        logger.warn("File already exists in path.  Replacing existing file: " + filePath.toUri());
        try {
            Files.delete(filePath);
        }
        catch (IOException ex) {
            throw new UserException("Could not replace existing file: " + filePath.toUri());
        }
    }

    public static File createFifoFile(Path fifoFilePath, boolean overwriteExisting) {
        if (Files.exists(fifoFilePath, new LinkOption[0])) {
            if (!overwriteExisting) {
                throw new UserException("Cannot create fifo file.  File already exists: " + fifoFilePath.toUri());
            }
            IOUtils.removeFileWithWarning(fifoFilePath);
        }
        ProcessSettings mkFIFOSettings = new ProcessSettings(new String[]{"mkfifo", fifoFilePath.toFile().getAbsolutePath()});
        mkFIFOSettings.getStdoutSettings().setBufferSize(-1);
        mkFIFOSettings.setRedirectErrorStream(true);
        ProcessController mkFIFOController = new ProcessController();
        ProcessOutput result = mkFIFOController.exec(mkFIFOSettings);
        int exitValue = result.getExitValue();
        File fifoFile = fifoFilePath.toFile();
        if (exitValue != 0) {
            throw new GATKException(String.format("Failure creating FIFO named (%s). Got exit code (%d) stderr (%s) and stdout (%s)", fifoFilePath.toFile().getAbsolutePath(), exitValue, result.getStderr() == null ? "" : result.getStderr().getBufferString(), result.getStdout() == null ? "" : result.getStdout().getBufferString()));
        }
        if (!fifoFile.exists()) {
            throw new GATKException(String.format("FIFO (%s) created but doesn't exist", fifoFilePath.toFile().getAbsolutePath()));
        }
        if (!fifoFile.canWrite()) {
            throw new GATKException(String.format("FIFO (%s) created isn't writable", fifoFilePath.toFile().getAbsolutePath()));
        }
        return fifoFile;
    }

    public static PrintStream makePrintStreamMaybeGzipped(GATKPath filePath) throws IOException {
        if (filePath.hasExtension(".gz")) {
            return new PrintStream(new GZIPOutputStream(filePath.getOutputStream()));
        }
        return new PrintStream(filePath.getOutputStream());
    }

    public static File createTempFile(String name, String extension) {
        return IOUtils.createTempFileInDirectory(name, extension, null);
    }

    public static File createTempFileInDirectory(String name, String extension, File targetDir) {
        try {
            if (!extension.startsWith(".")) {
                extension = "." + extension;
            }
            File file = File.createTempFile(name, extension, targetDir);
            file.deleteOnExit();
            new File(file.getAbsolutePath() + ".idx").deleteOnExit();
            new File(file.getAbsolutePath() + ".tbi").deleteOnExit();
            new File(file.getAbsolutePath() + ".bai").deleteOnExit();
            new File(file.getAbsolutePath() + ".md5").deleteOnExit();
            new File(file.getAbsolutePath().replaceAll(extension + "$", ".bai")).deleteOnExit();
            return file;
        }
        catch (IOException ex) {
            throw new GATKException("Cannot create temp file: " + ex.getMessage(), ex);
        }
    }

    public static Path createTempPath(String name, String extension) {
        try {
            if (!extension.startsWith(".")) {
                extension = "." + extension;
            }
            Path path = Files.createTempFile(IOUtils.getPath(System.getProperty("java.io.tmpdir")), name, extension, new FileAttribute[0]);
            IOUtils.deleteOnExit(path);
            String filename = path.getFileName().toString();
            IOUtils.deleteOnExit(path.resolveSibling(filename + ".idx"));
            IOUtils.deleteOnExit(path.resolveSibling(filename + ".tbi"));
            IOUtils.deleteOnExit(path.resolveSibling(filename + ".bai"));
            IOUtils.deleteOnExit(path.resolveSibling(filename.replaceAll(extension + "$", ".bai")));
            IOUtils.deleteOnExit(path.resolveSibling(filename + ".md5"));
            return path;
        }
        catch (IOException ex) {
            throw new GATKException("Cannot create temp file: " + ex.getMessage(), ex);
        }
    }

    public static String replaceExtension(String path, String extension) {
        Utils.nonNull(path);
        Utils.nonNull(extension);
        String extensionNoLeadingDot = StringUtils.stripStart((String)extension, (String)".");
        return FilenameUtils.removeExtension((String)path) + '.' + extensionNoLeadingDot;
    }

    public static File replaceExtension(File file, String extension) {
        return new File(IOUtils.replaceExtension(file.getPath(), extension));
    }

    public static Path getPath(String uriString) {
        URI uri;
        Utils.nonNull(uriString);
        try {
            uri = URI.create(uriString);
        }
        catch (IllegalArgumentException x) {
            return Paths.get(uriString, new String[0]);
        }
        try {
            if ("gs".equals(uri.getScheme())) {
                return BucketUtils.getPathOnGcs(uriString);
            }
            return uri.getScheme() == null ? Paths.get(uriString, new String[0]) : Paths.get(uri);
        }
        catch (FileSystemNotFoundException e) {
            try {
                ClassLoader cl = Thread.currentThread().getContextClassLoader();
                if (cl == null) {
                    throw e;
                }
                return FileSystems.newFileSystem(uri, new HashMap(), cl).provider().getPath(uri);
            }
            catch (ProviderNotFoundException x) {
                return Paths.get(uriString, new String[0]);
            }
            catch (IOException io) {
                throw new UserException(uriString + " is not a supported path", io);
            }
        }
    }

    public static String appendPathToDir(String dir, String path) {
        if (path.startsWith("/")) {
            return path;
        }
        if (BucketUtils.isRemoteStorageUrl(dir) || BucketUtils.isFileUrl(dir)) {
            Path dirPath = IOUtils.getPath(dir);
            return dirPath.resolve(path).toUri().toString();
        }
        return new File(dir, path).getPath();
    }

    public static String getAbsolutePathWithoutFileProtocol(Path path) {
        return path.toAbsolutePath().toUri().toString().replaceFirst("^file://", "");
    }

    public static void assertFileIsReadable(Path path) {
        Utils.nonNull(path);
        try {
            if (!Files.exists(path, new LinkOption[0])) {
                throw new UserException.CouldNotReadInputFile(path, "It doesn't exist.");
            }
            if (!Files.isRegularFile(path, new LinkOption[0])) {
                throw new UserException.CouldNotReadInputFile(path, "It isn't a regular file");
            }
            if (!Files.isReadable(path)) {
                throw new UserException.CouldNotReadInputFile(path, "It is not readable, check the file permissions");
            }
        }
        catch (StorageException cloudBoom) {
            throw new UserException.CouldNotReadInputFile(path, cloudBoom.getCode() + ": " + cloudBoom.getMessage(), (Throwable)cloudBoom);
        }
    }

    public static void assertPathsAreReadable(String ... paths) {
        for (String path : paths) {
            IOUtils.assertFileIsReadable(IOUtils.getPath(path));
        }
    }

    public static void canReadFile(File ... files) {
        Utils.nonNull(files, "Unexpected null input.");
        for (File file : files) {
            Utils.nonNull(file, "Unexpected null file reference.");
            if (!file.exists()) {
                throw new UserException.CouldNotReadInputFile(file.getAbsolutePath(), "The input file does not exist.");
            }
            if (!file.isFile()) {
                throw new UserException.CouldNotReadInputFile(file.getAbsolutePath(), "The input file is not a regular file");
            }
            if (file.canRead()) continue;
            throw new UserException.CouldNotReadInputFile(file.getAbsolutePath(), "The input file cannot be read.  Check the permissions.");
        }
    }

    public static void createDirectory(String pathString) throws IOException {
        Utils.nonNull(pathString);
        Files.createDirectory(IOUtils.getPath(pathString), new FileAttribute[0]);
    }

    public static String urlEncode(String string) {
        try {
            return URLEncoder.encode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            throw new UserException("Could not encode sample name", ex);
        }
    }

    public static String urlDecode(String string) {
        try {
            return URLDecoder.decode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            throw new UserException("Could not decode sample name", ex);
        }
    }

    public static boolean isGenomicsDBPath(GATKPath pathSpec) {
        return IOUtils.getGenomicsDBPath(pathSpec) != null;
    }

    public static boolean isGenomicsDBPath(String path) {
        return IOUtils.getGenomicsDBPath(path) != null;
    }

    public static String getAbsolutePathWithGenomicsDBURIScheme(GATKPath genomicsDBPath) {
        String path = IOUtils.getGenomicsDBAbsolutePath(genomicsDBPath);
        if (path == null) {
            return null;
        }
        if (path.contains("://")) {
            return "gendb." + path;
        }
        return "gendb://" + path;
    }

    public static String getGenomicsDBAbsolutePath(GATKPath gendbPath) {
        String path = IOUtils.getGenomicsDBPath(gendbPath);
        if (path == null) {
            return null;
        }
        if (path.contains("://")) {
            return path;
        }
        return new File(path).getAbsolutePath();
    }

    public static String getGenomicsDBPath(GATKPath path) {
        return IOUtils.getGenomicsDBPath(path.getRawInputString());
    }

    public static String getGenomicsDBPath(String path) {
        Matcher matcher;
        String genomicsdbPath = null;
        if (path != null && path.startsWith(GENOMIC_DB_URI_SCHEME) && (matcher = GENOMICSDB_URI_PATTERN.matcher(path)).find() && !matcher.group(3).isEmpty()) {
            if (!matcher.group(1).isEmpty()) {
                if (!matcher.group(2).isEmpty()) {
                    genomicsdbPath = matcher.group(2) + matcher.group(3) + matcher.group(4);
                }
            } else if (matcher.group(2).isEmpty()) {
                genomicsdbPath = matcher.group(4);
            }
        }
        return genomicsdbPath;
    }

    public static void deleteOnExit(Path fileToDelete) {
        DeleteRecursivelyOnExitPathHook.add(fileToDelete);
    }

    public static void deleteRecursively(Path rootPath) {
        IOUtil.recursiveDelete((Path)rootPath);
    }

    public static Path fileToPath(File toConvert) {
        return null == toConvert ? null : toConvert.toPath();
    }
}

