/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.mcmc;

import com.google.common.primitives.Doubles;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.commons.math3.optim.MaxEval;
import org.apache.commons.math3.optim.OptimizationData;
import org.apache.commons.math3.optim.nonlinear.scalar.GoalType;
import org.apache.commons.math3.optim.univariate.BrentOptimizer;
import org.apache.commons.math3.optim.univariate.SearchInterval;
import org.apache.commons.math3.optim.univariate.UnivariateObjectiveFunction;
import org.apache.commons.math3.stat.descriptive.moment.Mean;
import org.apache.commons.math3.stat.descriptive.moment.StandardDeviation;
import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.mllib.stat.KernelDensity;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.mcmc.DecileCollection;
import org.broadinstitute.hellbender.utils.mcmc.PosteriorSummary;

public class PosteriorSummaryUtils {
    public static final double SILVERMANS_RULE_CONSTANT = 1.06;
    public static final double SILVERMANS_RULE_EXPONENT = -0.2;
    private static final MaxEval BRENT_MAX_EVAL = new MaxEval(100);
    private static final double RELATIVE_TOLERANCE = 0.01;

    private PosteriorSummaryUtils() {
    }

    public static PosteriorSummary calculateHighestPosteriorDensitySummary(List<Double> samples, double alpha, JavaSparkContext ctx) {
        Utils.nonNull(samples);
        Utils.validateArg(samples.size() > 0, "Number of samples must be greater than zero.");
        Utils.validateArg(0.0 < alpha && alpha < 1.0, "Alpha must be in (0, 1).");
        double central = PosteriorSummaryUtils.calculatePosteriorMode(samples, ctx);
        if (Double.isNaN(central)) {
            return new PosteriorSummary(Double.NaN, Double.NaN, Double.NaN);
        }
        ArrayList<Double> sortedSamples = new ArrayList<Double>(samples);
        Collections.sort(sortedSamples);
        int n = sortedSamples.size();
        double lower = (Double)sortedSamples.get(0);
        double upper = (Double)sortedSamples.get(n - 1);
        double minIntervalWidth = (Double)sortedSamples.get(n - 1) - (Double)sortedSamples.get(0);
        int numSamplesInInterval = (int)Math.floor((1.0 - alpha) * (double)n);
        for (int i = 0; i < n - numSamplesInInterval; ++i) {
            double intervalWidth = (Double)sortedSamples.get(i + numSamplesInInterval) - (Double)sortedSamples.get(i);
            if (!(intervalWidth < minIntervalWidth)) continue;
            minIntervalWidth = intervalWidth;
            lower = (Double)sortedSamples.get(i);
            upper = (Double)sortedSamples.get(i + numSamplesInInterval);
        }
        return new PosteriorSummary(central, lower, upper);
    }

    public static PosteriorSummary calculateHighestPosteriorDensityAndDecilesSummary(List<Double> samples, double alpha, JavaSparkContext ctx) {
        Utils.nonNull(samples);
        Utils.validateArg(samples.size() > 0, "Number of samples must be greater than zero.");
        Utils.validateArg(0.0 < alpha && alpha < 1.0, "Alpha must be in (0, 1).");
        PosteriorSummary posteriorSummary = PosteriorSummaryUtils.calculateHighestPosteriorDensitySummary(samples, alpha, ctx);
        DecileCollection deciles = new DecileCollection(samples);
        posteriorSummary.setDeciles(deciles);
        return posteriorSummary;
    }

    public static double calculatePosteriorMode(List<Double> samples, JavaSparkContext ctx) {
        Utils.nonNull(samples);
        Utils.validateArg(samples.size() > 0, "Number of samples must be greater than zero.");
        double sampleMin = Collections.min(samples);
        double sampleMax = Collections.max(samples);
        double sampleMean = new Mean().evaluate(Doubles.toArray(samples));
        double sampleStandardDeviation = new StandardDeviation().evaluate(Doubles.toArray(samples));
        if (sampleStandardDeviation == 0.0 || Double.isNaN(sampleMean)) {
            return sampleMean;
        }
        double bandwidth = 1.06 * sampleStandardDeviation * Math.pow(samples.size(), -0.2);
        KernelDensity pdf = new KernelDensity().setSample(ctx.parallelize(samples, 1)).setBandwidth(bandwidth);
        BrentOptimizer optimizer = new BrentOptimizer(0.01, 0.01 * (sampleMax - sampleMin));
        UnivariateObjectiveFunction objective = new UnivariateObjectiveFunction(f -> pdf.estimate(new double[]{f})[0]);
        SearchInterval searchInterval = new SearchInterval(sampleMin, sampleMax, sampleMean);
        return optimizer.optimize(new OptimizationData[]{objective, GoalType.MAXIMIZE, searchInterval, BRENT_MAX_EVAL}).getPoint();
    }
}

