/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.nio;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Queue;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.collections4.queue.CircularFifoQueue;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.nio.NioFileCopierWithProgressMeterResults;

public class NioFileCopierWithProgressMeter {
    private static final Logger logger = LogManager.getLogger(NioFileCopierWithProgressMeter.class);
    protected static final int BUFFER_SIZE_BYTES = 0x100000;
    protected static final double DEFAULT_PROGRESS_DISPLAY_PERCENT_INCREMENT = 0.25;
    protected static final long SECOND_IN_MS = 1000L;
    protected static final long MINUTE_IN_MS = 60000L;
    protected static final long HOUR_IN_MS = 3600000L;
    protected static final long DAY_IN_MS = 86400000L;
    protected static final int KB_TO_BYTES = 1024;
    protected static final int MS_TO_SEC = 1000;
    protected static final int NANOS_TO_MILLIS = 1000000;
    protected static final int NANOS_TO_SECONDS = 1000000000;
    protected static final int COPY_SPEED_HISTORY_SIZE = 10;
    protected static final boolean OVERWRITE_EXISTING_DEFAULT = false;
    protected static final Verbosity VERBOSITY_DEFAULT = Verbosity.MODERATE;
    protected final Path source;
    protected final Path dest;
    protected long srcFileSize;
    protected int srcFileSizeNumDigits;
    protected String checksum = "";
    protected MessageDigest messageDigest = null;
    protected String expectedChecksum = "";
    protected boolean overwriteExisting = false;
    protected Verbosity verbosity = Verbosity.MODERATE;
    protected boolean formatTimeRemainingAsTimestamp = true;
    protected final byte[] copyBuffer = new byte[0x100000];
    protected double progressPercentDisplayIncrement = 0.25;
    protected final Queue<Double> downloadBytesPerMilliSecond = new CircularFifoQueue(10);
    protected boolean copyComplete = false;
    protected long totalBytesRead = 0L;
    protected long progressBytesRead = 0L;
    protected double lastProgressValue = 0.0;
    protected long lastProgressTime_ns;

    protected NioFileCopierWithProgressMeter(Path source, Path dest, boolean overwriteExisting, Verbosity verbosity) {
        this.source = source.toAbsolutePath();
        this.dest = dest.toAbsolutePath();
        this.overwriteExisting = overwriteExisting;
        this.verbosity = verbosity;
    }

    public static NioFileCopierWithProgressMeter create(Path source, Path dest) {
        return NioFileCopierWithProgressMeter.create(source, dest, false);
    }

    public static NioFileCopierWithProgressMeter create(Path source, Path dest, boolean overwriteExisting) {
        return new NioFileCopierWithProgressMeter(source, dest, overwriteExisting, VERBOSITY_DEFAULT);
    }

    public static NioFileCopierWithProgressMeter create(Path source, Path dest, boolean overwriteExisting, Verbosity verbosity) {
        return new NioFileCopierWithProgressMeter(source, dest, overwriteExisting, verbosity);
    }

    public Path getSource() {
        return this.source;
    }

    public Path getDest() {
        return this.dest;
    }

    public boolean isCopyComplete() {
        return this.copyComplete;
    }

    public boolean isOverwriteExisting() {
        return this.overwriteExisting;
    }

    public NioFileCopierWithProgressMeter setOverwriteExisting(boolean overwriteExisting) {
        this.overwriteExisting = overwriteExisting;
        return this;
    }

    public NioFileCopierWithProgressMeter setChecksumAlgorithmAndExpectedChecksum(String algorithm, String expectedChecksum) {
        try {
            this.messageDigest = MessageDigest.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new IllegalArgumentException("Provided checksum algorithm does not exist: " + algorithm, ex);
        }
        this.expectedChecksum = expectedChecksum;
        return this;
    }

    public NioFileCopierWithProgressMeter setFormatTimeRemainingAsTimestamp() {
        this.formatTimeRemainingAsTimestamp = true;
        return this;
    }

    public NioFileCopierWithProgressMeter setFormatTimeRemainingAsWords() {
        this.formatTimeRemainingAsTimestamp = false;
        return this;
    }

    public Verbosity getVerbosity() {
        return this.verbosity;
    }

    public NioFileCopierWithProgressMeter setVerbosity(Verbosity verbosity) {
        this.verbosity = verbosity;
        return this;
    }

    protected void updateMessageDigest(byte[] copyBuffer, int startIndex, int endIndex) {
        if (this.messageDigest != null) {
            this.messageDigest.update(copyBuffer, startIndex, endIndex - startIndex);
        }
    }

    protected void calculateChecksumFromMessageDigest() {
        if (this.messageDigest != null) {
            this.checksum = DatatypeConverter.printHexBinary((byte[])this.messageDigest.digest());
        }
    }

    protected boolean isSilent() {
        return this.verbosity == Verbosity.SILENT;
    }

    protected String formatMillisecondsTime(long time_ms) {
        if (this.formatTimeRemainingAsTimestamp) {
            return new AsTimeTimeFormatter(time_ms).format();
        }
        return new AsWordsTimeFormatter(time_ms).format();
    }

    protected void logProgress(double progressValue, long totalBytesRead, double bytesPerMillisecond) {
        if (this.verbosity == Verbosity.VERBOSE) {
            this.logProgressVerbose(progressValue, totalBytesRead, bytesPerMillisecond);
        } else if (this.verbosity.isAbove(Verbosity.MINIMAL)) {
            this.logProgressSimple(progressValue, totalBytesRead, bytesPerMillisecond);
        }
    }

    protected Duration getRemainingDuration(long totalBytesRead, double bytesPerMillisecond) {
        long remainingFileSize_bytes = this.srcFileSize - totalBytesRead;
        double estTimeRemaining_ms = (double)remainingFileSize_bytes / bytesPerMillisecond;
        return Duration.ofMillis((long)estTimeRemaining_ms);
    }

    protected void logProgressSimple(double progressValue, long totalBytesRead, double bytesPerMillisecond) {
        Duration estTimeRemainingDuration = this.getRemainingDuration(totalBytesRead, bytesPerMillisecond);
        logger.info(String.format("    Transfer: % 2.2f%% complete.  Est. time remaining: %s (@%3.02f kbps)", progressValue, this.formatMillisecondsTime(estTimeRemainingDuration.toMillis()), bytesPerMillisecond / 1024.0 * 1000.0));
    }

    protected void logProgressVerbose(double progressValue, long totalBytesRead, double bytesPerMillisecond) {
        Duration estTimeRemainingDuration = this.getRemainingDuration(totalBytesRead, bytesPerMillisecond);
        Instant endTime = Instant.now().plus(estTimeRemainingDuration);
        logger.info(String.format("    Transfer: % 2.2f%% complete (%" + this.srcFileSizeNumDigits + "d bytes; %6s).  Est. time remaining: %s (Complete time: %s) (@%3.02f kbps)", progressValue, totalBytesRead, FileUtils.byteCountToDisplaySize((long)totalBytesRead), this.formatMillisecondsTime(estTimeRemainingDuration.toMillis()), endTime.atZone(ZoneId.systemDefault()).toLocalDateTime().toString(), bytesPerMillisecond / 1024.0 * 1000.0));
    }

    private void initializeCopyProgressTime(long startTime_ns) {
        this.lastProgressTime_ns = startTime_ns;
    }

    protected void updateCopyProgress(int bytesRead) {
        if (!this.isSilent()) {
            this.totalBytesRead += (long)bytesRead;
            this.progressBytesRead += (long)bytesRead;
            double rawProgressValuePercent = (double)this.totalBytesRead / (double)this.srcFileSize * 100.0;
            double progressValue = this.progressPercentDisplayIncrement * Math.floor(Math.abs(rawProgressValuePercent / this.progressPercentDisplayIncrement));
            if (progressValue != this.lastProgressValue) {
                long currentProgressTime_ns = System.nanoTime();
                long dt_ms = (currentProgressTime_ns - this.lastProgressTime_ns) / 1000000L;
                double bytesPerMs = (double)this.progressBytesRead / (double)dt_ms;
                this.lastProgressTime_ns = currentProgressTime_ns;
                this.downloadBytesPerMilliSecond.add(bytesPerMs);
                this.logProgress(progressValue, this.totalBytesRead, this.downloadBytesPerMilliSecond.stream().mapToDouble(x -> x).average().orElse(dt_ms));
                this.lastProgressValue = progressValue;
                this.progressBytesRead = 0L;
            }
        }
    }

    protected void determineProgessDisplayIncrement(long fileSize) {
        long SIZE_STEP = 1024L;
        long KB = 1024L;
        long MB = 0x100000L;
        long GB = 0x40000000L;
        this.progressPercentDisplayIncrement = fileSize >= 0x1900000000L ? 0.1 : (fileSize >= 0x280000000L ? 0.25 : (fileSize >= 0x140000000L ? 0.5 : (fileSize >= 0x40000000L ? 1.0 : (fileSize >= 0x6400000L ? 5.0 : (fileSize >= 0x100000L ? 10.0 : 25.0)))));
    }

    protected void doCopy() {
        try (InputStream inputStream = Files.newInputStream(this.getSource(), new OpenOption[0]);
             OutputStream outputStream = Files.newOutputStream(this.getDest(), new OpenOption[0]);){
            int bytesRead;
            this.srcFileSize = Files.size(this.getSource());
            this.srcFileSizeNumDigits = (int)Math.ceil(Math.log10(this.srcFileSize));
            this.determineProgessDisplayIncrement(this.srcFileSize);
            if (this.verbosity.isAbove(Verbosity.SILENT)) {
                logger.info("Initiating copy from " + this.getSource().toUri().toString() + " to " + this.getDest().toUri().toString());
                logger.info("File size: " + this.srcFileSize + " bytes (" + FileUtils.byteCountToDisplaySize((long)this.srcFileSize) + ").");
                logger.info("Please wait.  This could take a while...");
            }
            while ((bytesRead = inputStream.read(this.copyBuffer)) != -1) {
                outputStream.write(this.copyBuffer, 0, bytesRead);
                this.updateMessageDigest(this.copyBuffer, 0, bytesRead);
                this.updateCopyProgress(bytesRead);
            }
            this.calculateChecksumFromMessageDigest();
        }
        catch (IOException ex) {
            throw new UserException("Could not copy file: " + this.source.toUri().toString() + " -> " + this.dest.toUri().toString(), ex);
        }
    }

    public NioFileCopierWithProgressMeterResults initiateCopy() {
        if (this.copyComplete) {
            throw new GATKException("Attempted multiple file copies.  NioFileCopierWithProgressMeter can copy a file only once!");
        }
        if (Files.exists(this.getDest(), new LinkOption[0])) {
            if (!this.isOverwriteExisting()) {
                throw new UserException.CouldNotCreateOutputFile(this.getDest().toUri().toString(), "Download aborted!  Output data sources file already exists!");
            }
            if (this.verbosity.isAbove(Verbosity.SILENT)) {
                logger.warn("Destination already exists.  Overwriting file at location: " + this.getDest().toUri().toString());
            }
        }
        long startTime_ns = System.nanoTime();
        this.initializeCopyProgressTime(startTime_ns);
        this.doCopy();
        if (this.verbosity.isAbove(Verbosity.SILENT)) {
            logger.info(String.format("Download Complete! - Total elapsed time: %ds", (System.nanoTime() - startTime_ns) / 1000000000L));
        }
        this.copyComplete = true;
        return new NioFileCopierWithProgressMeterResults(this.source, this.dest, this.srcFileSize, this.messageDigest != null, this.checksum, this.messageDigest == null ? "" : this.messageDigest.getAlgorithm(), this.expectedChecksum);
    }

    private class AsTimeTimeFormatter
    extends SimpleTimeFormatter {
        AsTimeTimeFormatter(long time_ms) {
            super(time_ms);
        }

        @Override
        public String format() {
            if (this.days > 0L) {
                return String.format("%d:%02d:%02d:%02d.%03d", this.days, this.hours, this.minutes, this.seconds, this.millis);
            }
            if (this.hours > 0L) {
                return String.format("%02d:%02d:%02d.%03d", this.hours, this.minutes, this.seconds, this.millis);
            }
            if (this.minutes > 0L) {
                return String.format("%02d:%02d.%03d", this.minutes, this.seconds, this.millis);
            }
            if (this.seconds > 0L) {
                return String.format("%02d.%03d", this.seconds, this.millis);
            }
            return String.format("00.%03d", this.millis);
        }
    }

    private class AsWordsTimeFormatter
    extends SimpleTimeFormatter {
        AsWordsTimeFormatter(long time_ms) {
            super(time_ms);
        }

        @Override
        public String format() {
            if (this.days > 0L) {
                return String.format("%d day" + this.sHelper(this.days) + ", %02d hour" + this.sHelper(this.hours) + ", %02d minute" + this.sHelper(this.minutes) + ", %2d.%03d seconds", this.days, this.hours, this.minutes, this.seconds, this.millis);
            }
            if (this.hours > 0L) {
                return String.format("%02d hour" + this.sHelper(this.hours) + ", %02d minute" + this.sHelper(this.minutes) + ", %02d.%03d seconds", this.hours, this.minutes, this.seconds, this.millis);
            }
            if (this.minutes > 0L) {
                return String.format("%02d minute" + this.sHelper(this.minutes) + ", %02d.%03d seconds", this.minutes, this.seconds, this.millis);
            }
            if (this.seconds > 0L) {
                return String.format("%02d.%03d seconds", this.seconds, this.millis);
            }
            return String.format("0.%03d seconds", this.millis);
        }
    }

    private abstract class SimpleTimeFormatter {
        final long rawTime_ms;
        final long days;
        final long hours;
        final long minutes;
        final long seconds;
        final long millis;

        SimpleTimeFormatter(long time_ms) {
            this.rawTime_ms = time_ms;
            long remainder = time_ms;
            this.days = this.formatTimeHelper(remainder, 86400000L);
            this.hours = this.formatTimeHelper(remainder -= this.days * 86400000L, 3600000L);
            this.minutes = this.formatTimeHelper(remainder -= this.hours * 3600000L, 60000L);
            this.seconds = this.formatTimeHelper(remainder -= this.minutes * 60000L, 1000L);
            this.millis = remainder -= this.seconds * 1000L;
        }

        private long formatTimeHelper(long duration, long conversionFactor) {
            long outTime = duration > conversionFactor ? Math.floorDiv(duration, conversionFactor) : 0L;
            return outTime;
        }

        protected String sHelper(long value) {
            return value == 1L ? "" : "s";
        }

        public abstract String format();
    }

    public static interface ChecksumCalculator {
        public String calculateChecksumOnInputStream(InputStream var1) throws IOException;
    }

    public static enum Verbosity {
        SILENT(0),
        MINIMAL(1),
        MODERATE(2),
        VERBOSE(3);

        private final int sev;

        private Verbosity(int sev) {
            this.sev = sev;
        }

        public boolean isAbove(Verbosity other) {
            return this.sev > other.sev;
        }
    }
}

