/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.pairhmm;

import com.google.common.annotations.VisibleForTesting;
import java.util.Arrays;
import org.apache.commons.lang.ArrayUtils;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.param.ParamUtils;
import org.broadinstitute.hellbender.utils.read.GATKRead;

public final class DragstrReadSTRAnalyzer {
    private final int[][] repeatsByPeriodAndPosition;
    private final int[] periodWithMostRepeats;
    private final int maxPeriod;
    private final int seqLength;

    private DragstrReadSTRAnalyzer(int seqLength, int[][] repeatsByPeriodAndPosition, int[] periodWithMostRepeats, int maxPeriod) {
        this.repeatsByPeriodAndPosition = repeatsByPeriodAndPosition;
        this.periodWithMostRepeats = periodWithMostRepeats;
        this.maxPeriod = maxPeriod;
        this.seqLength = seqLength;
    }

    public static DragstrReadSTRAnalyzer of(GATKRead read, int maxPeriod) {
        Utils.nonNull(read, "the input read cannot be null");
        byte[] bases = Utils.nonNull(read.getBasesNoCopy(), "the input read must have bases");
        return DragstrReadSTRAnalyzer.of(bases, maxPeriod);
    }

    @VisibleForTesting
    static DragstrReadSTRAnalyzer of(byte[] bases, int maxPeriod) {
        ParamUtils.isPositive(maxPeriod, "the input max period must be 1 or greater");
        int seqLength = bases.length;
        int[][] repeatsByPeriodAndPosition = new int[maxPeriod][seqLength];
        int[] periodWithMostRepeats = new int[seqLength];
        DragstrReadSTRAnalyzer.analyzeBases(bases, seqLength, repeatsByPeriodAndPosition, periodWithMostRepeats, maxPeriod);
        return new DragstrReadSTRAnalyzer(seqLength, repeatsByPeriodAndPosition, periodWithMostRepeats, maxPeriod);
    }

    public int numberOfRepeats(int position, int period) {
        if (period <= 0 || period > this.maxPeriod) {
            return 0;
        }
        if (position < 0 || position >= this.seqLength) {
            throw new IllegalArgumentException("cannot query outside requested boundaries");
        }
        return this.repeatsByPeriodAndPosition[period - 1][position];
    }

    public int mostRepeatedPeriod(int position) {
        if (position >= 0 && position < this.seqLength) {
            return this.periodWithMostRepeats[position];
        }
        throw new IllegalArgumentException(String.format("cannot query outside requested boundaries [%d, %d): %d", 0, this.seqLength, position));
    }

    public int numberOfMostRepeats(int position) {
        if (position >= 0 && position < this.seqLength) {
            return this.repeatsByPeriodAndPosition[this.periodWithMostRepeats[position] - 1][position];
        }
        throw new IllegalArgumentException("cannot query outside requested boundaries");
    }

    private static void analyzeBases(byte[] bases, int seqLength, int[][] repeatsByPeriodAndPosition, int[] periodWithMostRepeats, int maxPeriod) {
        DragstrReadSTRAnalyzer.calculateRepeatsForPeriodOne(bases, seqLength, repeatsByPeriodAndPosition[0]);
        int[] runLengthBuffer = new int[seqLength + 1];
        int[] heap = new int[maxPeriod + 1];
        for (int period = 2; period <= maxPeriod; ++period) {
            DragstrReadSTRAnalyzer.calculateRepeatsForPeriodTwoAndAbove(bases, seqLength, period, runLengthBuffer, heap, repeatsByPeriodAndPosition[period - 1]);
        }
        Arrays.fill(periodWithMostRepeats, 0, seqLength, 1);
        int[] mostRepeats = (int[])repeatsByPeriodAndPosition[0].clone();
        for (int periodIndex = 1; periodIndex < maxPeriod; ++periodIndex) {
            int periodLength = periodIndex + 1;
            int[] periodValues = repeatsByPeriodAndPosition[periodIndex];
            for (int position = 0; position < seqLength; ++position) {
                int repeats = periodValues[position];
                if (repeats <= mostRepeats[position]) continue;
                mostRepeats[position] = repeats;
                periodWithMostRepeats[position] = periodLength;
            }
        }
    }

    private static void calculateRepeatsForPeriodTwoAndAbove(byte[] bases, int seqLength, int period, int[] runLengthBuffer, int[] heap, int[] output) {
        int cycleIndex;
        if (seqLength < period) {
            Arrays.fill(output, 0, seqLength, 0);
            return;
        }
        int position = seqLength - 1;
        for (cycleIndex = period; cycleIndex > 1; --cycleIndex) {
            runLengthBuffer[position] = 0;
            --position;
        }
        runLengthBuffer[position--] = 1;
        int carryBack = 1;
        int positionPlusPeriod = position + period;
        int matchedCycles = 0;
        while (position >= 0) {
            if (bases[position] == bases[positionPlusPeriod]) {
                if (++matchedCycles == period) {
                    runLengthBuffer[position] = ++carryBack;
                    matchedCycles = 0;
                } else {
                    runLengthBuffer[position] = carryBack;
                }
            } else {
                runLengthBuffer[position] = 1;
                carryBack = 1;
                matchedCycles = 0;
            }
            --position;
            --positionPlusPeriod;
        }
        for (cycleIndex = 0; cycleIndex < period; ++cycleIndex) {
            for (position = cycleIndex; position < seqLength; position += period) {
                int totalRunLength = runLengthBuffer[position];
                for (int repeatInRun = 1; repeatInRun < totalRunLength; ++repeatInRun) {
                    runLengthBuffer[position += period] = totalRunLength;
                }
            }
        }
        if (runLengthBuffer[0] > 1) {
            runLengthBuffer[0] = runLengthBuffer[0] - 1;
        }
        int heapSize = period + 1;
        Arrays.fill(heap, 0, heapSize, 0);
        int currentMax = heap[0] = runLengthBuffer[0];
        int stop0 = Math.min(period, seqLength - 1);
        position = 0;
        while (position < stop0) {
            int n = position++;
            heap[position] = runLengthBuffer[position];
            output[n] = currentMax = Math.max(currentMax, heap[position]);
            DragstrReadSTRAnalyzer.fixHeap(heap, position, heapSize);
        }
        runLengthBuffer[seqLength] = 1;
        int outPosition = 0;
        while (position < seqLength) {
            int valueOut;
            int valueIn;
            if ((valueIn = runLengthBuffer[++position]) != (valueOut = runLengthBuffer[outPosition++])) {
                int fixHeapIdx = ArrayUtils.indexOf((int[])heap, (int)valueOut);
                heap[fixHeapIdx] = valueIn;
                DragstrReadSTRAnalyzer.fixHeap(heap, fixHeapIdx, heapSize);
                currentMax = heap[0];
            }
            output[position - 1] = currentMax;
        }
    }

    private static void fixHeap(int[] heap, int idx, int heapSize) {
        if (idx == 0 || heap[idx - 1 >> 1] > heap[idx]) {
            DragstrReadSTRAnalyzer.fixHeapDown(heap, idx, heapSize);
        } else {
            DragstrReadSTRAnalyzer.fixHeapUp(heap, idx);
        }
    }

    private static void fixHeapUp(int[] heap, int idx) {
        int upIdx;
        int upValue;
        int value = heap[idx];
        while ((upValue = heap[upIdx = idx - 1 >> 1]) < value) {
            heap[idx] = upValue;
            idx = upIdx;
            if (idx > 0) continue;
        }
        heap[idx] = value;
    }

    private static boolean checkHeap(int[] heap, int heapSize) {
        for (int i = 1; i < heapSize; ++i) {
            if (heap[i - 1 >> 1] >= heap[i]) continue;
            return false;
        }
        return true;
    }

    private static void fixHeapDown(int[] heap, int idx, int heapSize) {
        int value = heap[idx];
        while (true) {
            int leftValue;
            int idxRight;
            int rightValue = (idxRight = idx + 1 << 1) < heapSize ? heap[idxRight] : -1;
            int idxLeft = idxRight - 1;
            int n = leftValue = idxLeft < heapSize ? heap[idxLeft] : -1;
            if (rightValue > value) {
                if (rightValue > leftValue) {
                    heap[idx] = rightValue;
                    idx = idxRight;
                    continue;
                }
                heap[idx] = leftValue;
                idx = idxLeft;
                continue;
            }
            if (leftValue <= value) break;
            heap[idx] = leftValue;
            idx = idxLeft;
        }
        heap[idx] = value;
    }

    private static void calculateRepeatsForPeriodOne(byte[] bases, int seqLength, int[] output) {
        int rightMargin = seqLength - 1;
        byte last = bases[rightMargin];
        output[rightMargin] = 1;
        int carryBack = 1;
        for (int position = rightMargin - 1; position >= 0; --position) {
            byte next = bases[position];
            output[position] = next == last ? ++carryBack : (carryBack = 1);
            last = next;
        }
        int prevRunLength = output[0];
        for (int position = 1; position <= rightMargin; ++position) {
            byte next = bases[position];
            if (next == last) {
                output[position] = prevRunLength;
                continue;
            }
            int thisRunLength = output[position];
            if (prevRunLength < thisRunLength) {
                output[position - 1] = thisRunLength;
            }
            last = next;
            prevRunLength = thisRunLength;
        }
    }
}

