/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.pairhmm;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.variant.variantcontext.Allele;
import java.io.Closeable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.gatk.nativebindings.pairhmm.PairHMMNativeArguments;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.genotyper.LikelihoodMatrix;
import org.broadinstitute.hellbender.utils.haplotype.Haplotype;
import org.broadinstitute.hellbender.utils.pairhmm.Log10PairHMM;
import org.broadinstitute.hellbender.utils.pairhmm.LoglessPairHMM;
import org.broadinstitute.hellbender.utils.pairhmm.PairHMMInputScoreImputation;
import org.broadinstitute.hellbender.utils.pairhmm.PairHMMInputScoreImputator;
import org.broadinstitute.hellbender.utils.pairhmm.VectorLoglessPairHMM;
import org.broadinstitute.hellbender.utils.read.GATKRead;

public abstract class PairHMM
implements Closeable {
    protected static final Logger logger = LogManager.getLogger(PairHMM.class);
    public static final byte BASE_QUALITY_SCORE_THRESHOLD = 18;
    protected boolean constantsAreInitialized = false;
    protected byte[] previousHaplotypeBases;
    protected int hapStartIndex;
    protected int maxHaplotypeLength;
    protected int maxReadLength;
    protected int paddedMaxReadLength;
    protected int paddedMaxHaplotypeLength;
    protected int paddedReadLength;
    protected int paddedHaplotypeLength;
    protected boolean initialized = false;
    protected boolean doNotUseTristateCorrection = false;
    protected double[] mLogLikelihoodArray;
    protected static Boolean doProfiling = true;
    protected static long pairHMMComputeTime = 0L;
    protected long threadLocalPairHMMComputeTimeDiff = 0L;
    protected long startTime = 0L;

    protected void doNotUseTristateCorrection() {
        this.doNotUseTristateCorrection = true;
    }

    public void initialize(int readMaxLength, int haplotypeMaxLength) throws IllegalArgumentException {
        Utils.validateArg(haplotypeMaxLength > 0, () -> "haplotypeMaxLength must be > 0 but got " + haplotypeMaxLength);
        this.maxHaplotypeLength = haplotypeMaxLength;
        this.maxReadLength = readMaxLength;
        this.paddedMaxReadLength = readMaxLength + 1;
        this.paddedMaxHaplotypeLength = haplotypeMaxLength + 1;
        this.previousHaplotypeBases = null;
        this.constantsAreInitialized = false;
        this.initialized = true;
    }

    public void initialize(List<Haplotype> haplotypes, Map<String, List<GATKRead>> perSampleReadList, int readMaxLength, int haplotypeMaxLength) {
        this.initialize(readMaxLength, haplotypeMaxLength);
    }

    private static int findMaxAlleleLength(List<? extends Allele> alleles) {
        int max = 0;
        for (Allele allele : alleles) {
            int alleleLength = allele.length();
            if (max >= alleleLength) continue;
            max = alleleLength;
        }
        return max;
    }

    static int findMaxReadLength(List<GATKRead> reads) {
        int listMaxReadLength = 0;
        for (GATKRead read : reads) {
            int readLength = read.getLength();
            if (readLength <= listMaxReadLength) continue;
            listMaxReadLength = readLength;
        }
        return listMaxReadLength;
    }

    public void computeLog10Likelihoods(LikelihoodMatrix<GATKRead, Haplotype> logLikelihoods, List<GATKRead> processedReads, PairHMMInputScoreImputator inputScoreImputator) {
        if (processedReads.isEmpty()) {
            return;
        }
        if (doProfiling.booleanValue()) {
            this.startTime = System.nanoTime();
        }
        int readMaxLength = PairHMM.findMaxReadLength(processedReads);
        int haplotypeMaxLength = PairHMM.findMaxAlleleLength(logLikelihoods.alleles());
        if (!this.initialized || readMaxLength > this.maxReadLength || haplotypeMaxLength > this.maxHaplotypeLength) {
            this.initialize(readMaxLength, haplotypeMaxLength);
        }
        int readCount = processedReads.size();
        List<Haplotype> alleles = logLikelihoods.alleles();
        int alleleCount = alleles.size();
        this.mLogLikelihoodArray = new double[readCount * alleleCount];
        int idx = 0;
        int readIndex = 0;
        for (GATKRead read : processedReads) {
            PairHMMInputScoreImputation inputScoreImputation = inputScoreImputator.impute(read);
            byte[] readBases = read.getBases();
            byte[] readQuals = read.getBaseQualities();
            byte[] readInsQuals = inputScoreImputation.insOpenPenalties();
            byte[] readDelQuals = inputScoreImputation.delOpenPenalties();
            byte[] overallGCP = inputScoreImputation.gapContinuationPenalties();
            boolean isFirstHaplotype = true;
            for (int a = 0; a < alleleCount; ++a) {
                Allele allele = alleles.get(a);
                byte[] alleleBases = allele.getBases();
                byte[] nextAlleleBases = a == alleles.size() - 1 ? null : alleles.get(a + 1).getBases();
                double lk = this.computeReadLikelihoodGivenHaplotypeLog10(alleleBases, readBases, readQuals, readInsQuals, readDelQuals, overallGCP, true, nextAlleleBases);
                logLikelihoods.set(a, readIndex, lk);
                this.mLogLikelihoodArray[idx++] = lk;
            }
            ++readIndex;
        }
        if (doProfiling.booleanValue()) {
            this.threadLocalPairHMMComputeTimeDiff = System.nanoTime() - this.startTime;
            pairHMMComputeTime += this.threadLocalPairHMMComputeTimeDiff;
        }
    }

    @VisibleForTesting
    double computeReadLikelihoodGivenHaplotypeLog10(byte[] haplotypeBases, byte[] readBases, byte[] readQuals, byte[] insertionGOP, byte[] deletionGOP, byte[] overallGCP, boolean recacheReadValues, byte[] nextHaplotypeBases) throws IllegalStateException, IllegalArgumentException {
        Utils.validate(this.initialized, "Must call initialize before calling computeReadLikelihoodGivenHaplotypeLog10");
        Utils.nonNull(haplotypeBases, "haplotypeBases may not be null");
        Utils.validateArg(haplotypeBases.length <= this.maxHaplotypeLength, () -> "Haplotype bases is too long, got " + haplotypeBases.length + " but max is " + this.maxHaplotypeLength);
        Utils.nonNull(readBases);
        Utils.validateArg(readBases.length <= this.maxReadLength, () -> "readBases is too long, got " + readBases.length + " but max is " + this.maxReadLength);
        Utils.validateArg(readQuals.length == readBases.length, () -> "Read bases and read quals aren't the same size: " + readBases.length + " vs " + readQuals.length);
        Utils.validateArg(insertionGOP.length == readBases.length, () -> "Read bases and read insertion quals aren't the same size: " + readBases.length + " vs " + insertionGOP.length);
        Utils.validateArg(deletionGOP.length == readBases.length, () -> "Read bases and read deletion quals aren't the same size: " + readBases.length + " vs " + deletionGOP.length);
        Utils.validateArg(overallGCP.length == readBases.length, () -> "Read bases and overall GCP aren't the same size: " + readBases.length + " vs " + overallGCP.length);
        this.paddedReadLength = readBases.length + 1;
        this.paddedHaplotypeLength = haplotypeBases.length + 1;
        this.hapStartIndex = recacheReadValues ? 0 : this.hapStartIndex;
        int nextHapStartIndex = nextHaplotypeBases == null || haplotypeBases.length != nextHaplotypeBases.length ? 0 : PairHMM.findFirstPositionWhereHaplotypesDiffer(haplotypeBases, nextHaplotypeBases);
        double result = this.subComputeReadLikelihoodGivenHaplotypeLog10(haplotypeBases, readBases, readQuals, insertionGOP, deletionGOP, overallGCP, this.hapStartIndex, recacheReadValues, nextHapStartIndex);
        Utils.validate(result <= 0.0, () -> "PairHMM Log Probability cannot be greater than 0: " + String.format("haplotype: %s, read: %s, result: %f, PairHMM: %s", new String(haplotypeBases), new String(readBases), result, this.getClass().getSimpleName()));
        Utils.validate(MathUtils.isValidLog10Probability(result), () -> "Invalid Log Probability: " + result);
        this.previousHaplotypeBases = haplotypeBases;
        this.hapStartIndex = nextHapStartIndex < this.hapStartIndex ? 0 : nextHapStartIndex;
        return result;
    }

    protected abstract double subComputeReadLikelihoodGivenHaplotypeLog10(byte[] var1, byte[] var2, byte[] var3, byte[] var4, byte[] var5, byte[] var6, int var7, boolean var8, int var9);

    public static int findFirstPositionWhereHaplotypesDiffer(byte[] haplotype1, byte[] haplotype2) throws IllegalArgumentException {
        if (haplotype1 == null || haplotype1.length == 0) {
            throw new IllegalArgumentException("Haplotype1 is bad " + Arrays.toString(haplotype1));
        }
        if (haplotype2 == null || haplotype2.length == 0) {
            throw new IllegalArgumentException("Haplotype2 is bad " + Arrays.toString(haplotype2));
        }
        for (int i = 0; i < haplotype1.length && i < haplotype2.length; ++i) {
            if (haplotype1[i] == haplotype2[i]) continue;
            return i;
        }
        return Math.min(haplotype1.length, haplotype2.length);
    }

    public double[] getLogLikelihoodArray() {
        return this.mLogLikelihoodArray;
    }

    @Override
    public void close() {
        if (doProfiling.booleanValue()) {
            logger.info("Total compute time in PairHMM computeLogLikelihoods() : " + (double)pairHMMComputeTime * 1.0E-9);
        }
    }

    public static enum Implementation {
        EXACT(args -> {
            Log10PairHMM hmm = new Log10PairHMM(true);
            logger.info("Using the non-hardware accelerated Java EXACT PairHMM implementation");
            return hmm;
        }),
        ORIGINAL(args -> {
            Log10PairHMM hmm = new Log10PairHMM(false);
            logger.info("Using the non-hardware-accelerated Java ORIGINAL PairHMM implementation");
            return hmm;
        }),
        LOGLESS_CACHING(args -> {
            LoglessPairHMM hmm = new LoglessPairHMM();
            logger.info("Using the non-hardware-accelerated Java LOGLESS_CACHING PairHMM implementation");
            return hmm;
        }),
        AVX_LOGLESS_CACHING(args -> {
            VectorLoglessPairHMM hmm = new VectorLoglessPairHMM(VectorLoglessPairHMM.Implementation.AVX, (PairHMMNativeArguments)args);
            logger.info("Using the AVX-accelerated native PairHMM implementation");
            return hmm;
        }),
        AVX_LOGLESS_CACHING_OMP(args -> {
            VectorLoglessPairHMM hmm = new VectorLoglessPairHMM(VectorLoglessPairHMM.Implementation.OMP, (PairHMMNativeArguments)args);
            logger.info("Using the OpenMP multi-threaded AVX-accelerated native PairHMM implementation");
            return hmm;
        }),
        EXPERIMENTAL_FPGA_LOGLESS_CACHING(args -> {
            VectorLoglessPairHMM hmm = new VectorLoglessPairHMM(VectorLoglessPairHMM.Implementation.FPGA, (PairHMMNativeArguments)args);
            logger.info("Using the FPGA-accelerated native PairHMM implementation");
            return hmm;
        }),
        FASTEST_AVAILABLE(args -> {
            try {
                VectorLoglessPairHMM hmm = new VectorLoglessPairHMM(VectorLoglessPairHMM.Implementation.OMP, (PairHMMNativeArguments)args);
                logger.info("Using the OpenMP multi-threaded AVX-accelerated native PairHMM implementation");
                return hmm;
            }
            catch (UserException.HardwareFeatureException e) {
                logger.info("OpenMP multi-threaded AVX-accelerated native PairHMM implementation is not supported");
                try {
                    VectorLoglessPairHMM hmm = new VectorLoglessPairHMM(VectorLoglessPairHMM.Implementation.AVX, (PairHMMNativeArguments)args);
                    logger.info("Using the AVX-accelerated native PairHMM implementation");
                    return hmm;
                }
                catch (UserException.HardwareFeatureException e2) {
                    logger.warn("***WARNING: Machine does not have the AVX instruction set support needed for the accelerated AVX PairHmm. Falling back to the MUCH slower LOGLESS_CACHING implementation!");
                    return new LoglessPairHMM();
                }
            }
        });

        private final Function<PairHMMNativeArguments, PairHMM> makeHmm;

        private Implementation(Function<PairHMMNativeArguments, PairHMM> makeHmm) {
            this.makeHmm = makeHmm;
        }

        public PairHMM makeNewHMM(PairHMMNativeArguments args) {
            return this.makeHmm.apply(args);
        }
    }
}

