/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.read;

import com.google.common.collect.Lists;
import htsjdk.samtools.Cigar;
import htsjdk.samtools.CigarElement;
import htsjdk.samtools.CigarOperator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.broadinstitute.gatk.nativebindings.smithwaterman.SWOverhangStrategy;
import org.broadinstitute.gatk.nativebindings.smithwaterman.SWParameters;
import org.broadinstitute.hellbender.utils.Tail;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.read.AlignmentUtils;
import org.broadinstitute.hellbender.utils.read.CigarBuilder;
import org.broadinstitute.hellbender.utils.smithwaterman.SmithWatermanAligner;
import org.broadinstitute.hellbender.utils.smithwaterman.SmithWatermanAlignment;

public final class CigarUtils {
    public static final SWParameters NEW_SW_PARAMETERS = new SWParameters(200, -150, -260, -11);
    public static final SWParameters ALIGNMENT_TO_BEST_HAPLOTYPE_SW_PARAMETERS = new SWParameters(10, -15, -30, -5);
    private static final String SW_PAD = "NNNNNNNNNN";

    private CigarUtils() {
    }

    public static Cigar invertCigar(Cigar cigar) {
        Utils.nonNull(cigar);
        ArrayList els = new ArrayList(cigar.getCigarElements());
        Collections.reverse(els);
        return new Cigar(els);
    }

    public static int countRefBasesAndClips(List<CigarElement> elems, int cigarStartIndex, int cigarEndIndex) {
        return CigarUtils.countRefBasesAndMaybeAlsoClips(elems, cigarStartIndex, cigarEndIndex, true, true);
    }

    public static int countRefBasesAndSoftClips(List<CigarElement> elems, int cigarStartIndex, int cigarEndIndex) {
        return CigarUtils.countRefBasesAndMaybeAlsoClips(elems, cigarStartIndex, cigarEndIndex, true, false);
    }

    private static int countRefBasesAndMaybeAlsoClips(List<CigarElement> elems, int cigarStartIndex, int cigarEndIndex, boolean includeSoftClips, boolean includeHardClips) {
        Utils.nonNull(elems);
        Utils.validateArg(cigarStartIndex >= 0 && cigarEndIndex <= elems.size() && cigarStartIndex <= cigarEndIndex, () -> "invalid index:0 -" + elems.size());
        int result = 0;
        for (CigarElement elem : elems.subList(cigarStartIndex, cigarEndIndex)) {
            CigarOperator op = elem.getOperator();
            if (!op.consumesReferenceBases() && (!includeSoftClips || op != CigarOperator.SOFT_CLIP) && (!includeHardClips || op != CigarOperator.HARD_CLIP)) continue;
            result += elem.getLength();
        }
        return result;
    }

    public static Cigar removeClipsAndPadding(Cigar cigar) {
        Utils.nonNull(cigar, "cigar is null");
        ArrayList<CigarElement> elements = new ArrayList<CigarElement>(cigar.numCigarElements());
        for (CigarElement ce : cigar.getCigarElements()) {
            if (CigarUtils.isClipOrPaddingOperator(ce.getOperator())) continue;
            elements.add(ce);
        }
        return new Cigar(elements);
    }

    private static boolean isClipOrPaddingOperator(CigarOperator op) {
        return op == CigarOperator.S || op == CigarOperator.H || op == CigarOperator.P;
    }

    public static boolean containsNOperator(List<CigarElement> cigarElements) {
        return Utils.nonNull(cigarElements).stream().anyMatch(el -> el.getOperator() == CigarOperator.N);
    }

    public static boolean isGood(Cigar c) {
        Utils.nonNull(c, "cigar is null");
        if (c.isValid(null, -1L) != null) {
            return false;
        }
        List elems = c.getCigarElements();
        return !CigarUtils.hasConsecutiveIndels(elems) && !CigarUtils.startsOrEndsWithDeletionIgnoringClips(elems);
    }

    private static boolean hasConsecutiveIndels(List<CigarElement> elems) {
        boolean prevIndel = false;
        for (CigarElement elem : elems) {
            boolean isIndel;
            CigarOperator op = elem.getOperator();
            boolean bl = isIndel = op == CigarOperator.INSERTION || op == CigarOperator.DELETION;
            if (prevIndel && isIndel) {
                return true;
            }
            prevIndel = isIndel;
        }
        return false;
    }

    private static boolean startsOrEndsWithDeletionIgnoringClips(List<CigarElement> elems) {
        block0: for (boolean leftSide : new boolean[]{true, false}) {
            for (CigarElement elem : leftSide ? elems : Lists.reverse(elems)) {
                CigarOperator op = elem.getOperator();
                if (op == CigarOperator.DELETION) {
                    return true;
                }
                if (op.isClipping()) continue;
                continue block0;
            }
        }
        return false;
    }

    public static Cigar calculateCigar(byte[] refSeq, byte[] altSeq, SmithWatermanAligner aligner, SWOverhangStrategy strategy) {
        Utils.nonNull(refSeq, "refSeq");
        Utils.nonNull(altSeq, "altSeq");
        if (altSeq.length == 0) {
            return new Cigar(Collections.singletonList(new CigarElement(refSeq.length, CigarOperator.D)));
        }
        if (altSeq.length == refSeq.length) {
            int mismatchCount = 0;
            for (int n = 0; n < refSeq.length && mismatchCount <= 2; mismatchCount += altSeq[n] == refSeq[n] ? 0 : 1, ++n) {
            }
            if (mismatchCount <= 2) {
                Cigar matching = new Cigar();
                matching.add(new CigarElement(refSeq.length, CigarOperator.MATCH_OR_MISMATCH));
                return matching;
            }
        }
        String paddedRef = SW_PAD + new String(refSeq) + SW_PAD;
        String paddedPath = SW_PAD + new String(altSeq) + SW_PAD;
        SmithWatermanAlignment alignment = aligner.align(paddedRef.getBytes(), paddedPath.getBytes(), NEW_SW_PARAMETERS, strategy);
        if (CigarUtils.isSWFailure(alignment)) {
            return null;
        }
        int baseStart = SW_PAD.length();
        int baseEnd = paddedPath.length() - SW_PAD.length() - 1;
        CigarBuilder.Result trimmedCigarAndDeletionsRemoved = AlignmentUtils.trimCigarByBases(alignment.getCigar(), baseStart, baseEnd);
        Cigar nonStandard = trimmedCigarAndDeletionsRemoved.getCigar();
        int trimmedLeadingDeletions = trimmedCigarAndDeletionsRemoved.getLeadingDeletionBasesRemoved();
        int trimmedTrailingDeletions = trimmedCigarAndDeletionsRemoved.getTrailingDeletionBasesRemoved();
        if (trimmedTrailingDeletions > 0) {
            nonStandard.add(new CigarElement(trimmedTrailingDeletions, CigarOperator.D));
        }
        CigarBuilder.Result leftAlignmentResult = AlignmentUtils.leftAlignIndels(nonStandard, refSeq, altSeq, trimmedLeadingDeletions);
        int totalLeadingDeletionsRemoved = trimmedLeadingDeletions + leftAlignmentResult.getLeadingDeletionBasesRemoved();
        int totalTrailingDeletionsRemoved = leftAlignmentResult.getTrailingDeletionBasesRemoved();
        if (totalLeadingDeletionsRemoved == 0 && totalTrailingDeletionsRemoved == 0) {
            return leftAlignmentResult.getCigar();
        }
        ArrayList<CigarElement> resultElements = new ArrayList<CigarElement>();
        if (totalLeadingDeletionsRemoved > 0) {
            resultElements.add(new CigarElement(totalLeadingDeletionsRemoved, CigarOperator.D));
        }
        resultElements.addAll(leftAlignmentResult.getCigar().getCigarElements());
        if (totalTrailingDeletionsRemoved > 0) {
            resultElements.add(new CigarElement(totalTrailingDeletionsRemoved, CigarOperator.D));
        }
        return new Cigar(resultElements);
    }

    private static boolean isSWFailure(SmithWatermanAlignment alignment) {
        if (alignment.getAlignmentOffset() > 0) {
            return true;
        }
        for (CigarElement ce : alignment.getCigar().getCigarElements()) {
            if (ce.getOperator() != CigarOperator.S) continue;
            return true;
        }
        return false;
    }

    public static int countUnclippedReadBases(Cigar cigar) {
        Utils.nonNull(cigar, "the input cigar cannot be null");
        return cigar.getCigarElements().stream().filter(ce -> {
            CigarOperator operator = ce.getOperator();
            return operator.isClipping() || operator.consumesReadBases();
        }).mapToInt(CigarElement::getLength).sum();
    }

    public static int countClippedBases(Cigar cigar, Tail tail, CigarOperator typeOfClip) {
        Utils.validateArg(typeOfClip.isClipping(), "typeOfClip must be a clipping operator");
        int size = cigar.numCigarElements();
        if (size < 2) {
            Utils.validateArg(size == 1 && !cigar.getFirstCigarElement().getOperator().isClipping(), "cigar is empty or completely clipped.");
            return 0;
        }
        int result = 0;
        for (int n = 0; n < size; ++n) {
            int index = tail == Tail.LEFT ? n : size - n - 1;
            CigarElement element = cigar.getCigarElement(index);
            if (!element.getOperator().isClipping()) {
                return result;
            }
            if (element.getOperator() != typeOfClip) continue;
            result += element.getLength();
        }
        throw new IllegalArgumentException("Input cigar " + cigar + " is completely clipped.");
    }

    public static int countClippedBases(Cigar cigar, Tail tail) {
        return CigarUtils.countClippedBases(cigar, tail, CigarOperator.SOFT_CLIP) + CigarUtils.countClippedBases(cigar, tail, CigarOperator.HARD_CLIP);
    }

    public static int countClippedBases(Cigar cigar, CigarOperator clippingType) {
        return CigarUtils.countClippedBases(cigar, Tail.LEFT, clippingType) + CigarUtils.countClippedBases(cigar, Tail.RIGHT, clippingType);
    }

    public static int countClippedBases(Cigar cigar) {
        return CigarUtils.countClippedBases(cigar, Tail.LEFT) + CigarUtils.countClippedBases(cigar, Tail.RIGHT);
    }

    public static int countAlignedBases(Cigar cigar) {
        return Utils.nonNull(cigar).getCigarElements().stream().filter(cigarElement -> cigarElement.getOperator().isAlignment()).mapToInt(CigarElement::getLength).sum();
    }

    public static Cigar revertSoftClips(Cigar originalCigar) {
        CigarBuilder builder = new CigarBuilder();
        for (CigarElement element : originalCigar.getCigarElements()) {
            if (element.getOperator() == CigarOperator.SOFT_CLIP) {
                builder.add(new CigarElement(element.getLength(), CigarOperator.MATCH_OR_MISMATCH));
                continue;
            }
            builder.add(element);
        }
        return builder.make();
    }

    public static Cigar clipCigar(Cigar cigar, int start, int stop, CigarOperator clippingOperator) {
        Utils.validateArg(clippingOperator.isClipping(), "Not a clipping operator");
        boolean clipLeft = start == 0;
        CigarBuilder newCigar = new CigarBuilder();
        int elementStart = 0;
        for (CigarElement element : cigar.getCigarElements()) {
            CigarOperator operator = element.getOperator();
            if (operator == CigarOperator.HARD_CLIP) {
                newCigar.add(new CigarElement(element.getLength(), element.getOperator()));
                continue;
            }
            int elementEnd = elementStart + (operator.consumesReadBases() ? element.getLength() : 0);
            if (elementEnd <= start || elementStart >= stop) {
                if (operator.consumesReadBases() || elementStart != start && elementStart != stop) {
                    newCigar.add(new CigarElement(element.getLength(), operator));
                }
            } else {
                int unclippedLength = clipLeft ? elementEnd - stop : start - elementStart;
                int clippedLength = element.getLength() - unclippedLength;
                if (unclippedLength <= 0) {
                    if (operator.consumesReadBases()) {
                        newCigar.add(new CigarElement(element.getLength(), clippingOperator));
                    }
                } else if (clipLeft) {
                    newCigar.add(new CigarElement(clippedLength, clippingOperator));
                    newCigar.add(new CigarElement(unclippedLength, operator));
                } else {
                    newCigar.add(new CigarElement(unclippedLength, operator));
                    newCigar.add(new CigarElement(clippedLength, clippingOperator));
                }
            }
            elementStart = elementEnd;
        }
        return newCigar.make();
    }

    public static int alignmentStartShift(Cigar cigar, int numClipped) {
        int refBasesClipped = 0;
        int elementStart = 0;
        for (CigarElement element : cigar.getCigarElements()) {
            CigarOperator operator = element.getOperator();
            if (operator == CigarOperator.HARD_CLIP) continue;
            int elementEnd = elementStart + (operator.consumesReadBases() ? element.getLength() : 0);
            if (elementEnd <= numClipped) {
                refBasesClipped += operator.consumesReferenceBases() ? element.getLength() : 0;
            } else if (elementStart < numClipped) {
                int clippedLength = numClipped - elementStart;
                refBasesClipped += operator.consumesReferenceBases() ? clippedLength : 0;
                break;
            }
            elementStart = elementEnd;
        }
        return refBasesClipped;
    }

    public static int computeAssociatedDistOnRead(Cigar cigar, int start, int refDist, boolean backward) {
        Utils.validateArg(refDist > 0 && start > 0, () -> "start " + start + " or distance " + refDist + " is non-positive.");
        List elements = backward ? Lists.reverse((List)cigar.getCigarElements()) : cigar.getCigarElements();
        int readLength = elements.stream().mapToInt(ce -> ce.getOperator().consumesReadBases() ? ce.getLength() : 0).sum();
        int readBasesToSkip = backward ? readLength - start : start - 1;
        int readBasesConsumed = 0;
        int refBasesConsumed = 0;
        for (CigarElement element : elements) {
            int readBasesConsumedBeforeElement = readBasesConsumed;
            if ((readBasesConsumed += element.getOperator().consumesReadBases() ? element.getLength() : 0) <= readBasesToSkip || (refBasesConsumed += element.getOperator().consumesReferenceBases() ? element.getLength() - Math.max(readBasesToSkip - readBasesConsumedBeforeElement, 0) : 0) < refDist) continue;
            int excessRefBasesInElement = Math.max(refBasesConsumed - refDist, 0);
            return readBasesConsumed - readBasesToSkip - (element.getOperator().consumesReadBases() ? excessRefBasesInElement : 0);
        }
        throw new IllegalArgumentException("Cigar " + cigar + "does not contain at least " + refDist + " reference bases past red start " + start + ".");
    }

    public static Cigar convertTerminalInsertionToSoftClip(Cigar cigar) {
        if (cigar.numCigarElements() < 2) {
            return cigar;
        }
        CigarBuilder builder = new CigarBuilder();
        for (int n = 0; n < cigar.numCigarElements(); ++n) {
            CigarElement element = cigar.getCigarElement(n);
            if (element.getOperator() != CigarOperator.INSERTION) {
                builder.add(element);
                continue;
            }
            if (n == 0 || n == cigar.numCigarElements() - 1) {
                builder.add(new CigarElement(element.getLength(), CigarOperator.SOFT_CLIP));
                continue;
            }
            if (cigar.getCigarElement(n - 1).getOperator().isClipping() || cigar.getCigarElement(n + 1).getOperator().isClipping()) {
                builder.add(new CigarElement(element.getLength(), CigarOperator.SOFT_CLIP));
                continue;
            }
            builder.add(element);
        }
        return builder.make();
    }
}

