/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.recalibration;

import htsjdk.samtools.Cigar;
import htsjdk.samtools.CigarElement;
import htsjdk.samtools.CigarOperator;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.util.Locatable;
import java.io.Serializable;
import java.util.Arrays;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.engine.ReferenceDataSource;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.exceptions.UserException;
import org.broadinstitute.hellbender.transformers.ReadTransformer;
import org.broadinstitute.hellbender.utils.BaseUtils;
import org.broadinstitute.hellbender.utils.MathUtils;
import org.broadinstitute.hellbender.utils.SerializableFunction;
import org.broadinstitute.hellbender.utils.SimpleInterval;
import org.broadinstitute.hellbender.utils.Utils;
import org.broadinstitute.hellbender.utils.baq.BAQ;
import org.broadinstitute.hellbender.utils.clipping.ReadClipper;
import org.broadinstitute.hellbender.utils.collections.NestedIntegerArray;
import org.broadinstitute.hellbender.utils.read.CigarBuilder;
import org.broadinstitute.hellbender.utils.read.GATKRead;
import org.broadinstitute.hellbender.utils.read.ReadUtils;
import org.broadinstitute.hellbender.utils.recalibration.EventType;
import org.broadinstitute.hellbender.utils.recalibration.ReadRecalibrationInfo;
import org.broadinstitute.hellbender.utils.recalibration.RecalDatum;
import org.broadinstitute.hellbender.utils.recalibration.RecalUtils;
import org.broadinstitute.hellbender.utils.recalibration.RecalibrationArgumentCollection;
import org.broadinstitute.hellbender.utils.recalibration.RecalibrationTables;
import org.broadinstitute.hellbender.utils.recalibration.covariates.Covariate;
import org.broadinstitute.hellbender.utils.recalibration.covariates.CovariateKeyCache;
import org.broadinstitute.hellbender.utils.recalibration.covariates.ReadCovariates;
import org.broadinstitute.hellbender.utils.recalibration.covariates.StandardCovariateList;

public final class BaseRecalibrationEngine
implements Serializable {
    private static final long serialVersionUID = 1L;
    protected static final Logger logger = LogManager.getLogger(BaseRecalibrationEngine.class);
    private final CovariateKeyCache keyCache;
    private final EventType[] cachedEventTypes;
    public static final SerializableFunction<GATKRead, SimpleInterval> BQSR_REFERENCE_WINDOW_FUNCTION = new BQSRReferenceWindowFunction();
    private RecalibrationArgumentCollection recalArgs;
    private RecalibrationTables recalTables;
    private SAMFileHeader readsHeader;
    private StandardCovariateList covariates;
    private BAQ baq;
    private static final byte NO_BAQ_UNCERTAINTY = 64;
    private long numReadsProcessed = 0L;
    private boolean finalized = false;

    public BaseRecalibrationEngine(RecalibrationArgumentCollection recalArgs, SAMFileHeader readsHeader) {
        EventType[] eventTypeArray;
        this.recalArgs = recalArgs;
        this.readsHeader = readsHeader;
        this.baq = recalArgs.enableBAQ ? new BAQ(recalArgs.BAQGOP) : null;
        this.covariates = new StandardCovariateList(recalArgs, readsHeader);
        int numReadGroups = readsHeader.getReadGroups().size();
        if (numReadGroups < 1) {
            throw new UserException("Number of read groups must be >= 1, but is " + numReadGroups);
        }
        this.recalTables = new RecalibrationTables(this.covariates, numReadGroups);
        this.keyCache = new CovariateKeyCache();
        if (recalArgs.computeIndelBQSRTables) {
            eventTypeArray = EventType.values();
        } else {
            EventType[] eventTypeArray2 = new EventType[1];
            eventTypeArray = eventTypeArray2;
            eventTypeArray2[0] = EventType.BASE_SUBSTITUTION;
        }
        this.cachedEventTypes = eventTypeArray;
    }

    public void logCovariatesUsed() {
        logger.info("The covariates being used here: ");
        for (Covariate cov : this.covariates) {
            logger.info('\t' + cov.getClass().getSimpleName());
        }
    }

    public void processRead(GATKRead originalRead, ReferenceDataSource refDS, Iterable<? extends Locatable> knownSites) {
        byte[] baqArray;
        ReadTransformer transform = this.makeReadTransform();
        GATKRead read = (GATKRead)transform.apply(originalRead);
        if (read.isEmpty()) {
            return;
        }
        RecalUtils.parsePlatformForRead(read, this.readsHeader, this.recalArgs);
        int[] isSNP = new int[read.getLength()];
        int[] isInsertion = new int[isSNP.length];
        int[] isDeletion = new int[isSNP.length];
        int nErrors = BaseRecalibrationEngine.calculateIsSNPOrIndel(read, refDS, isSNP, isInsertion, isDeletion);
        byte[] byArray = baqArray = nErrors == 0 || !this.recalArgs.enableBAQ ? BaseRecalibrationEngine.flatBAQArray(read) : this.calculateBAQArray(read, refDS);
        if (baqArray != null) {
            ReadCovariates covariates = RecalUtils.computeCovariates(read, this.readsHeader, this.covariates, true, this.keyCache);
            boolean[] skip = this.calculateSkipArray(read, knownSites);
            double[] snpErrors = BaseRecalibrationEngine.calculateFractionalErrorArray(isSNP, baqArray);
            double[] insertionErrors = BaseRecalibrationEngine.calculateFractionalErrorArray(isInsertion, baqArray);
            double[] deletionErrors = BaseRecalibrationEngine.calculateFractionalErrorArray(isDeletion, baqArray);
            ReadRecalibrationInfo info = new ReadRecalibrationInfo(read, covariates, skip, snpErrors, insertionErrors, deletionErrors);
            this.updateRecalTablesForRead(info);
        }
        ++this.numReadsProcessed;
    }

    public void finalizeData() {
        Utils.validate(!this.finalized, "FinalizeData() has already been called");
        BaseRecalibrationEngine.finalizeRecalibrationTables(this.recalTables);
        this.finalized = true;
    }

    public static void finalizeRecalibrationTables(RecalibrationTables tables) {
        Utils.nonNull(tables);
        NestedIntegerArray<RecalDatum> byReadGroupTable = tables.getReadGroupTable();
        NestedIntegerArray<RecalDatum> byQualTable = tables.getQualityScoreTable();
        for (NestedIntegerArray.Leaf<RecalDatum> leaf : byQualTable.getAllLeaves()) {
            int rgKey = leaf.keys[0];
            int eventIndex = leaf.keys[2];
            RecalDatum rgDatum = byReadGroupTable.get(rgKey, eventIndex);
            RecalDatum qualDatum = (RecalDatum)leaf.value;
            if (rgDatum == null) {
                byReadGroupTable.put(new RecalDatum(qualDatum), rgKey, eventIndex);
                continue;
            }
            rgDatum.combine(qualDatum);
        }
        BaseRecalibrationEngine.roundTableValues(tables);
    }

    private static void roundTableValues(RecalibrationTables rt) {
        for (int i = 0; i < rt.numTables(); ++i) {
            for (NestedIntegerArray.Leaf<RecalDatum> leaf : rt.getTable(i).getAllLeaves()) {
                ((RecalDatum)leaf.value).setNumMismatches(MathUtils.roundToNDecimalPlaces(((RecalDatum)leaf.value).getNumMismatches(), 2));
                ((RecalDatum)leaf.value).setEmpiricalQuality(MathUtils.roundToNDecimalPlaces(((RecalDatum)leaf.value).getEmpiricalQuality(), 4));
                ((RecalDatum)leaf.value).setEstimatedQReported(MathUtils.roundToNDecimalPlaces(((RecalDatum)leaf.value).getEstimatedQReported(), 4));
            }
        }
    }

    public RecalibrationTables getRecalibrationTables() {
        return this.recalTables;
    }

    public RecalibrationTables getFinalRecalibrationTables() {
        Utils.validate(this.finalized, "Cannot get final recalibration tables until finalizeData() has been called");
        return this.recalTables;
    }

    public StandardCovariateList getCovariates() {
        return this.covariates;
    }

    public long getNumReadsProcessed() {
        return this.numReadsProcessed;
    }

    private void updateRecalTablesForRead(ReadRecalibrationInfo recalInfo) {
        Utils.validate(!this.finalized, "FinalizeData() has already been called");
        GATKRead read = recalInfo.getRead();
        ReadCovariates readCovariates = recalInfo.getCovariatesValues();
        NestedIntegerArray<RecalDatum> qualityScoreTable = this.recalTables.getQualityScoreTable();
        int nCovariates = this.covariates.size();
        int nSpecialCovariates = this.covariates.numberOfSpecialCovariates();
        int readLength = read.getLength();
        for (int offset = 0; offset < readLength; ++offset) {
            if (recalInfo.skip(offset)) continue;
            for (int idx = 0; idx < this.cachedEventTypes.length; ++idx) {
                EventType eventType = this.cachedEventTypes[idx];
                int[] keys = readCovariates.getKeySet(offset, eventType);
                int eventIndex = eventType.ordinal();
                byte qual = recalInfo.getQual(eventType, offset);
                double isError = recalInfo.getErrorFraction(eventType, offset);
                int key0 = keys[0];
                int key1 = keys[1];
                RecalUtils.incrementDatumOrPutIfNecessary3keys(qualityScoreTable, qual, isError, key0, key1, eventIndex);
                for (int i = nSpecialCovariates; i < nCovariates; ++i) {
                    int keyi = keys[i];
                    if (keyi < 0) continue;
                    RecalUtils.incrementDatumOrPutIfNecessary4keys(this.recalTables.getTable(i), qual, isError, key0, key1, keyi, eventIndex);
                }
            }
        }
    }

    private ReadTransformer makeReadTransform() {
        ReadTransformer f0 = BaseRecalibrationEngine::consolidateCigar;
        ReadTransformer f = f0.andThen(this::setDefaultBaseQualities).andThen(this::resetOriginalBaseQualities).andThen(ReadClipper::hardClipAdaptorSequence).andThen(ReadClipper::hardClipSoftClippedBases);
        return f;
    }

    private static GATKRead consolidateCigar(GATKRead read) {
        read.setCigar(new CigarBuilder().addAll((Iterable<CigarElement>)read.getCigar()).make());
        return read;
    }

    private GATKRead resetOriginalBaseQualities(GATKRead read) {
        if (!this.recalArgs.useOriginalBaseQualities.booleanValue()) {
            return read;
        }
        return ReadUtils.resetOriginalBaseQualities(read);
    }

    private GATKRead setDefaultBaseQualities(GATKRead read) {
        if (this.recalArgs.defaultBaseQualities < 0) {
            return read;
        }
        byte[] reads = read.getBases();
        byte[] quals = read.getBaseQualities();
        if (quals == null || quals.length < reads.length) {
            byte[] new_quals = new byte[reads.length];
            Arrays.fill(new_quals, this.recalArgs.defaultBaseQualities);
            read.setBaseQualities(new_quals);
        }
        return read;
    }

    private boolean[] calculateSkipArray(GATKRead read, Iterable<? extends Locatable> knownSites) {
        int readLength = read.getLength();
        boolean[] skip = new boolean[readLength];
        boolean[] knownSitesArray = BaseRecalibrationEngine.calculateKnownSites(read, knownSites);
        for (int i = 0; i < readLength; ++i) {
            skip[i] = !BaseUtils.isRegularBase(read.getBase(i)) || read.getBaseQuality(i) < this.recalArgs.PRESERVE_QSCORES_LESS_THAN || knownSitesArray[i];
        }
        return skip;
    }

    protected static boolean[] calculateKnownSites(GATKRead read, Iterable<? extends Locatable> knownSites) {
        int readLength = read.getLength();
        boolean[] knownSitesArray = new boolean[readLength];
        Cigar cigar = read.getCigar();
        int softStart = read.getSoftStart();
        int softEnd = read.getSoftEnd();
        for (Locatable locatable : knownSites) {
            Pair<Integer, CigarOperator> featureEndAndOperatorOnRead;
            int featureEndOnRead;
            int featureStartOnRead;
            if (locatable.getEnd() < softStart || locatable.getStart() > softEnd) continue;
            Pair<Integer, CigarOperator> featureStartAndOperatorOnRead = ReadUtils.getReadIndexForReferenceCoordinate(read, locatable.getStart());
            int n = featureStartOnRead = (Integer)featureStartAndOperatorOnRead.getLeft() == -1 ? 0 : (Integer)featureStartAndOperatorOnRead.getLeft();
            if (featureStartAndOperatorOnRead.getRight() == CigarOperator.DELETION) {
                --featureStartOnRead;
            }
            int n2 = featureEndOnRead = (Integer)(featureEndAndOperatorOnRead = ReadUtils.getReadIndexForReferenceCoordinate(read, locatable.getEnd())).getLeft() == -1 ? readLength : (Integer)featureEndAndOperatorOnRead.getLeft();
            if (featureStartOnRead > readLength) {
                featureStartOnRead = featureEndOnRead = readLength;
            }
            Arrays.fill(knownSitesArray, Math.max(0, featureStartOnRead), Math.min(readLength, featureEndOnRead + 1), true);
        }
        return knownSitesArray;
    }

    protected static int calculateIsSNPOrIndel(GATKRead read, ReferenceDataSource ref, int[] snp, int[] isIns, int[] isDel) {
        byte[] refBases = ref.queryAndPrefetch(read.getContig(), read.getStart(), read.getEnd()).getBases();
        int readPos = 0;
        int refPos = 0;
        int nEvents = 0;
        block8: for (CigarElement ce : read.getCigarElements()) {
            int elementLength = ce.getLength();
            switch (ce.getOperator()) {
                case M: 
                case EQ: 
                case X: {
                    for (int i = 0; i < elementLength; ++i) {
                        int snpInt;
                        snp[readPos] = snpInt = BaseUtils.basesAreEqual(read.getBase(readPos), refBases[refPos]) ? 0 : 1;
                        nEvents += snpInt;
                        ++readPos;
                        ++refPos;
                    }
                    continue block8;
                }
                case D: {
                    int index = read.isReverseStrand() ? readPos : readPos - 1;
                    BaseRecalibrationEngine.updateIndel(isDel, index);
                    refPos += elementLength;
                    break;
                }
                case N: {
                    refPos += elementLength;
                    break;
                }
                case I: {
                    boolean forwardStrandRead;
                    boolean bl = forwardStrandRead = !read.isReverseStrand();
                    if (forwardStrandRead) {
                        BaseRecalibrationEngine.updateIndel(isIns, readPos - 1);
                    }
                    readPos += elementLength;
                    if (forwardStrandRead) continue block8;
                    BaseRecalibrationEngine.updateIndel(isIns, readPos);
                    break;
                }
                case S: {
                    readPos += elementLength;
                    break;
                }
                case H: 
                case P: {
                    break;
                }
                default: {
                    throw new GATKException("Unsupported cigar operator: " + ce.getOperator());
                }
            }
        }
        nEvents = (int)((long)nEvents + (MathUtils.sum(isDel) + MathUtils.sum(isIns)));
        return nEvents;
    }

    private static void updateIndel(int[] indel, int index) {
        if (index >= 0 && index < indel.length) {
            indel[index] = 1;
        }
    }

    public static double[] calculateFractionalErrorArray(int[] errorArray, byte[] baqArray) {
        int i;
        if (errorArray.length != baqArray.length) {
            throw new GATKException("Array length mismatch detected. Malformed read?");
        }
        int BLOCK_START_UNSET = -1;
        double[] fractionalErrors = new double[baqArray.length];
        boolean inBlock = false;
        int blockStartIndex = -1;
        for (i = 0; i < fractionalErrors.length; ++i) {
            if (baqArray[i] == 64) {
                if (!inBlock) {
                    fractionalErrors[i] = errorArray[i];
                    continue;
                }
                BaseRecalibrationEngine.calculateAndStoreErrorsInBlock(i, blockStartIndex, errorArray, fractionalErrors);
                inBlock = false;
                blockStartIndex = -1;
                continue;
            }
            inBlock = true;
            if (blockStartIndex != -1) continue;
            blockStartIndex = i;
        }
        if (inBlock) {
            BaseRecalibrationEngine.calculateAndStoreErrorsInBlock(i - 1, blockStartIndex, errorArray, fractionalErrors);
        }
        if (fractionalErrors.length != errorArray.length) {
            throw new GATKException("Output array length mismatch detected. Malformed read?");
        }
        return fractionalErrors;
    }

    private static void calculateAndStoreErrorsInBlock(int i, int blockStartIndex, int[] errorArray, double[] fractionalErrors) {
        int j;
        int totalErrors = 0;
        for (j = Math.max(0, blockStartIndex - 1); j <= i; ++j) {
            totalErrors += errorArray[j];
        }
        for (j = Math.max(0, blockStartIndex - 1); j <= i; ++j) {
            fractionalErrors[j] = (double)totalErrors / (double)(i - Math.max(0, blockStartIndex - 1) + 1);
        }
    }

    protected static byte[] flatBAQArray(GATKRead read) {
        byte[] baq = new byte[read.getLength()];
        Arrays.fill(baq, (byte)64);
        return baq;
    }

    private byte[] calculateBAQArray(GATKRead read, ReferenceDataSource refDS) {
        this.baq.baqRead(read, refDS, BAQ.CalculationMode.RECALCULATE, BAQ.QualityMode.ADD_TAG);
        return BAQ.getBAQTag(read);
    }

    public static final class BQSRReferenceWindowFunction
    implements SerializableFunction<GATKRead, SimpleInterval> {
        private static final long serialVersionUID = 1L;

        @Override
        public SimpleInterval apply(GATKRead read) {
            return BAQ.getReferenceWindowForRead(read, 7);
        }
    }
}

