/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.recalibration;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.utils.QualityUtils;
import org.broadinstitute.hellbender.utils.Utils;

public final class QualQuantizer {
    private static final Set<QualInterval> MY_EMPTY_SET = Collections.emptySet();
    private static final Logger logger = LogManager.getLogger(QualQuantizer.class);
    final int nLevels;
    final int minInterestingQual;
    final List<Long> nObservationsPerQual;
    final List<Byte> originalToQuantizedMap;
    final TreeSet<QualInterval> quantizedIntervals;

    protected QualQuantizer(int minInterestingQual) {
        this.nObservationsPerQual = Collections.emptyList();
        this.nLevels = 0;
        this.minInterestingQual = minInterestingQual;
        this.quantizedIntervals = null;
        this.originalToQuantizedMap = null;
    }

    public QualQuantizer(List<Long> nObservationsPerQual, int nLevels, int minInterestingQual) {
        this.nObservationsPerQual = nObservationsPerQual;
        this.nLevels = nLevels;
        this.minInterestingQual = minInterestingQual;
        if (Collections.min(nObservationsPerQual) < 0L) {
            throw new GATKException("Quality score histogram has negative values at: " + Utils.join(", ", nObservationsPerQual));
        }
        if (nLevels < 0) {
            throw new GATKException("nLevels must be >= 0");
        }
        if (minInterestingQual < 0) {
            throw new GATKException("minInterestingQual must be >= 0");
        }
        this.quantizedIntervals = this.quantize();
        this.originalToQuantizedMap = this.intervalsToMap(this.quantizedIntervals);
    }

    private TreeSet<QualInterval> quantize() {
        TreeSet<QualInterval> intervals = new TreeSet<QualInterval>();
        for (int qStart = 0; qStart < this.getNQualsInHistogram(); ++qStart) {
            long nObs = this.nObservationsPerQual.get(qStart);
            double errorRate = QualityUtils.qualToErrorProb((byte)qStart);
            double nErrors = (double)nObs * errorRate;
            QualInterval qi = new QualInterval(qStart, qStart, nObs, (long)((int)Math.floor(nErrors)), 0, (byte)qStart);
            intervals.add(qi);
        }
        while (intervals.size() > this.nLevels) {
            this.mergeLowestPenaltyIntervals(intervals);
        }
        return intervals;
    }

    private void mergeLowestPenaltyIntervals(TreeSet<QualInterval> intervals) {
        Iterator<QualInterval> it1 = intervals.iterator();
        Iterator<QualInterval> it1p = intervals.iterator();
        it1p.next();
        QualInterval minMerge = null;
        if (logger.isDebugEnabled()) {
            logger.debug("mergeLowestPenaltyIntervals: " + intervals.size());
        }
        int lastMergeOrder = 0;
        while (it1p.hasNext()) {
            QualInterval left = it1.next();
            QualInterval right = it1p.next();
            QualInterval merged = left.merge(right);
            lastMergeOrder = Math.max(Math.max(lastMergeOrder, left.mergeOrder), right.mergeOrder);
            if (minMerge != null && !(merged.getPenalty() < minMerge.getPenalty())) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("  Updating merge " + minMerge);
            }
            minMerge = merged;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("  => final min merge " + minMerge);
        }
        intervals.removeAll(minMerge.subIntervals);
        intervals.add(minMerge);
        minMerge.mergeOrder = lastMergeOrder + 1;
        if (logger.isDebugEnabled()) {
            logger.debug("updated intervals: " + intervals);
        }
    }

    private List<Byte> intervalsToMap(TreeSet<QualInterval> intervals) {
        ArrayList<Byte> map = new ArrayList<Byte>(this.getNQualsInHistogram());
        map.addAll(Collections.nCopies(this.getNQualsInHistogram(), (byte)-128));
        for (QualInterval interval : intervals) {
            for (int q = interval.qStart; q <= interval.qEnd; ++q) {
                map.set(q, interval.getQual());
            }
        }
        if ((Byte)Collections.min(map) == -128) {
            throw new GATKException("quantized quality score map contains an un-initialized value");
        }
        return map;
    }

    private int getNQualsInHistogram() {
        return this.nObservationsPerQual.size();
    }

    public List<Byte> getOriginalToQuantizedMap() {
        return this.originalToQuantizedMap;
    }

    protected final class QualInterval
    implements Comparable<QualInterval> {
        final int qStart;
        final int qEnd;
        final int fixedQual;
        final int level;
        final long nObservations;
        final long nErrors;
        final Set<QualInterval> subIntervals;
        int mergeOrder;

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level) {
            this(qStart, qEnd, nObservations, nErrors, level, -1, MY_EMPTY_SET);
        }

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, Set<QualInterval> subIntervals) {
            this(qStart, qEnd, nObservations, nErrors, level, -1, subIntervals);
        }

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, int fixedQual) {
            this(qStart, qEnd, nObservations, nErrors, level, fixedQual, MY_EMPTY_SET);
        }

        public QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, int fixedQual, Set<QualInterval> subIntervals) {
            this.qStart = qStart;
            this.qEnd = qEnd;
            this.nObservations = nObservations;
            this.nErrors = nErrors;
            this.fixedQual = fixedQual;
            this.level = level;
            this.mergeOrder = 0;
            this.subIntervals = Collections.unmodifiableSet(subIntervals);
        }

        public String getName() {
            return this.qStart + "-" + this.qEnd;
        }

        public String toString() {
            return "QQ:" + this.getName();
        }

        public double getErrorRate() {
            if (this.hasFixedQual()) {
                return QualityUtils.qualToErrorProb((byte)this.fixedQual);
            }
            if (this.nObservations == 0L) {
                return 0.0;
            }
            return (double)(this.nErrors + 1L) / (1.0 * (double)(this.nObservations + 1L));
        }

        public byte getQual() {
            if (!this.hasFixedQual()) {
                return QualityUtils.errorProbToQual(this.getErrorRate());
            }
            return (byte)this.fixedQual;
        }

        public boolean hasFixedQual() {
            return this.fixedQual != -1;
        }

        @Override
        public int compareTo(QualInterval qualInterval) {
            return Integer.valueOf(this.qStart).compareTo(qualInterval.qStart);
        }

        public QualInterval merge(QualInterval toMerge) {
            QualInterval right;
            QualInterval left = this.compareTo(toMerge) < 0 ? this : toMerge;
            QualInterval qualInterval = right = this.compareTo(toMerge) < 0 ? toMerge : this;
            if (left.qEnd + 1 != right.qStart) {
                throw new GATKException("Attempting to merge non-contiguous intervals: left = " + left + " right = " + right);
            }
            long nCombinedObs = left.nObservations + right.nObservations;
            long nCombinedErr = left.nErrors + right.nErrors;
            int level = Math.max(left.level, right.level) + 1;
            LinkedHashSet<QualInterval> subIntervals = new LinkedHashSet<QualInterval>(Arrays.asList(left, right));
            QualInterval merged = new QualInterval(left.qStart, right.qEnd, nCombinedObs, nCombinedErr, level, subIntervals);
            return merged;
        }

        public double getPenalty() {
            return this.calcPenalty(this.getErrorRate());
        }

        private double calcPenalty(double globalErrorRate) {
            if (globalErrorRate == 0.0) {
                return 0.0;
            }
            if (this.subIntervals.isEmpty()) {
                if (this.qEnd <= QualQuantizer.this.minInterestingQual) {
                    return 0.0;
                }
                return Math.abs(Math.log10(this.getErrorRate()) - Math.log10(globalErrorRate)) * (double)this.nObservations;
            }
            double sum = 0.0;
            for (QualInterval interval : this.subIntervals) {
                sum += interval.calcPenalty(globalErrorRate);
            }
            return sum;
        }
    }
}

