/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.hellbender.utils.report;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.broadinstitute.hellbender.exceptions.GATKException;
import org.broadinstitute.hellbender.utils.report.GATKReportColumn;
import org.broadinstitute.hellbender.utils.report.GATKReportDataType;
import org.broadinstitute.hellbender.utils.report.GATKReportVersion;
import org.broadinstitute.hellbender.utils.text.TextFormattingUtils;

public final class GATKReportTable {
    public static final String INVALID_TABLE_NAME_REGEX = "[^a-zA-Z0-9_\\-\\.]";
    private static final String GATKTABLE_HEADER_PREFIX = "#:GATKTable";
    private static final String SEPARATOR = ":";
    private static final String ENDLINE = ":;";
    private static final Comparator<? super Object[]> ROW_COMPARATOR = (objectArr1, objectArr2) -> {
        boolean EQUAL = false;
        int result = 0;
        for (int x = 0; x < ((Object[])objectArr1).length; ++x) {
            result = !Objects.equals(objectArr1[x].getClass(), objectArr2[x].getClass()) ? objectArr1[x].toString().compareTo(objectArr2[x].toString()) : (objectArr1[x] instanceof Byte ? ((Byte)objectArr1[x]).compareTo((Byte)objectArr2[x]) : (objectArr1[x] instanceof Short ? ((Short)objectArr1[x]).compareTo((Short)objectArr2[x]) : (objectArr1[x] instanceof Integer ? ((Integer)objectArr1[x]).compareTo((Integer)objectArr2[x]) : (objectArr1[x] instanceof Long ? ((Long)objectArr1[x]).compareTo((Long)objectArr2[x]) : (objectArr1[x] instanceof BigInteger ? ((BigInteger)objectArr1[x]).compareTo((BigInteger)objectArr2[x]) : (objectArr1[x] instanceof Float ? ((Float)objectArr1[x]).compareTo((Float)objectArr2[x]) : (objectArr1[x] instanceof Double ? ((Double)objectArr1[x]).compareTo((Double)objectArr2[x]) : (objectArr1[x] instanceof BigDecimal ? ((BigDecimal)objectArr1[x]).compareTo((BigDecimal)objectArr2[x]) : objectArr1[x].toString().compareTo(objectArr2[x].toString())))))))));
            if (result == 0) continue;
            return result;
        }
        return result;
    };
    private final String tableName;
    private final String tableDescription;
    private final Sorting sortingWay;
    private final List<Object[]> underlyingData;
    private final List<GATKReportColumn> columnInfo;
    private final Map<Object, Integer> columnNameToIndex;
    private final Map<Object, Integer> rowIdToIndex;
    private static final String COULD_NOT_READ_HEADER = "Could not read the header of this file -- ";
    private static final String COULD_NOT_READ_COLUMN_NAMES = "Could not read the column names of this file -- ";
    private static final String COULD_NOT_READ_DATA_LINE = "Could not read a data line of this table -- ";
    private static final String COULD_NOT_READ_EMPTY_LINE = "Could not read the last empty line of this table -- ";
    private static final String OLD_GATK_TABLE_VERSION = "We no longer support older versions of the GATK Tables";
    private static final int INITITAL_ARRAY_SIZE = 10000;

    public GATKReportTable(BufferedReader reader, GATKReportVersion version) {
        switch (version) {
            case V1_1: {
                int i;
                String columnLine;
                String[] tableNameData;
                String[] tableData;
                try {
                    tableData = reader.readLine().split(SEPARATOR);
                    tableNameData = reader.readLine().split(SEPARATOR);
                }
                catch (IOException e) {
                    throw new GATKException(COULD_NOT_READ_HEADER + e.getMessage());
                }
                this.tableName = tableNameData[TableNameHeaderFields.NAME.index()];
                this.tableDescription = tableNameData.length <= TableNameHeaderFields.DESCRIPTION.index() ? "" : tableNameData[TableNameHeaderFields.DESCRIPTION.index()];
                this.sortingWay = Sorting.DO_NOT_SORT;
                int nColumns = Integer.parseInt(tableData[TableDataHeaderFields.COLS.index()]);
                int nRows = Integer.parseInt(tableData[TableDataHeaderFields.ROWS.index()]);
                this.underlyingData = new ArrayList<Object[]>(nRows);
                this.columnInfo = new ArrayList<GATKReportColumn>(nColumns);
                this.columnNameToIndex = new LinkedHashMap<Object, Integer>(nColumns);
                this.rowIdToIndex = new LinkedHashMap<Object, Integer>();
                for (int i2 = 0; i2 < nRows; ++i2) {
                    this.rowIdToIndex.put(i2, i2);
                }
                try {
                    columnLine = reader.readLine();
                }
                catch (IOException e) {
                    throw new GATKException(COULD_NOT_READ_COLUMN_NAMES);
                }
                List<Integer> columnStarts = TextFormattingUtils.getWordStarts(columnLine);
                String[] columnNames = TextFormattingUtils.splitFixedWidth(columnLine, columnStarts);
                for (i = 0; i < nColumns; ++i) {
                    String format = tableData[TableDataHeaderFields.FORMAT_START.index() + i];
                    this.addColumn(columnNames[i], format);
                }
                try {
                    for (i = 0; i < nRows; ++i) {
                        String dataLine = reader.readLine();
                        List<String> lineSplits = Arrays.asList(TextFormattingUtils.splitFixedWidth(dataLine, columnStarts));
                        this.underlyingData.add(new Object[nColumns]);
                        for (int columnIndex = 0; columnIndex < nColumns; ++columnIndex) {
                            GATKReportDataType type = this.columnInfo.get(columnIndex).getDataType();
                            String columnName = columnNames[columnIndex];
                            this.set((Object)i, columnName, type.Parse(lineSplits.get(columnIndex)));
                        }
                    }
                }
                catch (IOException e) {
                    throw new GATKException(COULD_NOT_READ_DATA_LINE + e.getMessage());
                }
                try {
                    reader.readLine();
                    break;
                }
                catch (IOException e) {
                    throw new GATKException(COULD_NOT_READ_EMPTY_LINE + e.getMessage());
                }
            }
            default: {
                throw new GATKException(OLD_GATK_TABLE_VERSION);
            }
        }
    }

    public GATKReportTable(String tableName, String tableDescription, int numColumns) {
        this(tableName, tableDescription, numColumns, Sorting.SORT_BY_ROW);
    }

    public GATKReportTable(String tableName, String tableDescription, int numColumns, Sorting sortingWay) {
        if (!this.isValidName(tableName)) {
            throw new GATKException("Attempted to set a GATKReportTable name of '" + tableName + "'.  GATKReportTable names must be purely alphanumeric - no spaces or special characters are allowed.");
        }
        if (!this.isValidDescription(tableDescription)) {
            throw new GATKException("Attempted to set a GATKReportTable description of '" + tableDescription + "'.  GATKReportTable descriptions must not contain newlines.");
        }
        this.tableName = tableName;
        this.tableDescription = tableDescription;
        this.sortingWay = sortingWay;
        this.underlyingData = new ArrayList<Object[]>(10000);
        this.columnInfo = new ArrayList<GATKReportColumn>(numColumns);
        this.columnNameToIndex = new LinkedHashMap<Object, Integer>(numColumns);
        this.rowIdToIndex = new LinkedHashMap<Object, Integer>();
    }

    private boolean isValidName(String name) {
        Pattern p = Pattern.compile(INVALID_TABLE_NAME_REGEX);
        Matcher m = p.matcher(name);
        return !m.find();
    }

    private boolean isValidDescription(String description) {
        Pattern p = Pattern.compile("\\r|\\n");
        Matcher m = p.matcher(description);
        return !m.find();
    }

    public void addRowID(String ID, boolean populateFirstColumn) {
        this.addRowIDMapping(ID, this.underlyingData.size(), populateFirstColumn);
    }

    public void addRowIDMapping(String ID, int index) {
        this.addRowIDMapping(ID, index, false);
    }

    public void addRowIDMapping(Object ID, int index, boolean populateFirstColumn) {
        this.expandTo(index, false);
        this.rowIdToIndex.put(ID, index);
        if (populateFirstColumn) {
            this.set(index, 0, ID);
        }
    }

    public void addColumn(String columnName, String format) {
        this.columnNameToIndex.put(columnName, this.columnInfo.size());
        this.columnInfo.add(new GATKReportColumn(columnName, format));
    }

    private void verifyEntry(int rowIndex, int colIndex) {
        if (rowIndex < 0 || colIndex < 0 || colIndex >= this.getNumColumns()) {
            throw new GATKException("attempted to access a cell that does not exist in table '" + this.tableName + "'");
        }
    }

    private void expandTo(int rowIndex, boolean updateRowIdMap) {
        int currentSize = this.underlyingData.size();
        if (rowIndex >= currentSize) {
            int numNewRows = rowIndex - currentSize + 1;
            for (int i = 0; i < numNewRows; ++i) {
                if (updateRowIdMap) {
                    this.rowIdToIndex.put(currentSize, currentSize);
                }
                this.underlyingData.add(new Object[this.getNumColumns()]);
                ++currentSize;
            }
        }
    }

    public void set(Object rowID, String columnName, Object value) {
        if (!this.rowIdToIndex.containsKey(rowID)) {
            this.rowIdToIndex.put(rowID, this.underlyingData.size());
            this.expandTo(this.underlyingData.size(), false);
        }
        this.set(this.rowIdToIndex.get(rowID), this.columnNameToIndex.get(columnName), value);
    }

    public void set(int rowIndex, int colIndex, Object value) {
        this.expandTo(rowIndex, true);
        this.verifyEntry(rowIndex, colIndex);
        GATKReportColumn column = this.columnInfo.get(colIndex);
        value = value == null ? "null" : this.fixType(value, column);
        if (!column.getDataType().equals((Object)GATKReportDataType.fromObject(value)) && !column.getDataType().equals((Object)GATKReportDataType.Unknown)) {
            throw new GATKException(String.format("Tried to add an object of type: %s to a column of type: %s", GATKReportDataType.fromObject(value).name(), column.getDataType().name()));
        }
        this.underlyingData.get((int)rowIndex)[colIndex] = value;
        column.updateFormatting(value);
    }

    public void increment(Object rowID, String columnName) {
        int prevValue;
        if (!this.rowIdToIndex.containsKey(rowID)) {
            this.rowIdToIndex.put(rowID, this.underlyingData.size());
            this.underlyingData.add(new Object[this.getNumColumns()]);
            prevValue = 0;
        } else {
            Object obj = this.get(rowID, columnName);
            if (!(obj instanceof Integer)) {
                throw new GATKException("Attempting to increment a value in a cell that is not an integer");
            }
            prevValue = (Integer)obj;
        }
        this.set(this.rowIdToIndex.get(rowID), this.columnNameToIndex.get(columnName), (Object)(prevValue + 1));
    }

    private Object fixType(Object value, GATKReportColumn column) {
        Object newValue = null;
        if (value instanceof String && !column.getDataType().equals((Object)GATKReportDataType.String)) {
            if (column.getDataType().equals((Object)GATKReportDataType.Integer)) {
                try {
                    newValue = Long.parseLong((String)value);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (column.getDataType().equals((Object)GATKReportDataType.Decimal)) {
                try {
                    newValue = Double.parseDouble((String)value);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (column.getDataType().equals((Object)GATKReportDataType.Character) && ((String)value).length() == 1) {
                newValue = Character.valueOf(((String)value).charAt(0));
            }
        }
        return newValue != null ? newValue : value;
    }

    public Object get(Object rowID, String columnName) {
        return this.get((int)this.rowIdToIndex.get(rowID), this.columnNameToIndex.get(columnName));
    }

    public Object[] getRow(int rowIndex) {
        return this.underlyingData.get(rowIndex);
    }

    public Object get(int rowIndex, String columnName) {
        return this.get(rowIndex, this.columnNameToIndex.get(columnName));
    }

    public Object get(int rowIndex, int columnIndex) {
        this.verifyEntry(rowIndex, columnIndex);
        return this.underlyingData.get(rowIndex)[columnIndex];
    }

    void write(PrintStream out) {
        this.write(out, this.sortingWay);
    }

    void write(PrintStream out, Sorting sortingWay) {
        out.printf("#:GATKTable:%d:%d", this.getNumColumns(), this.getNumRows());
        for (GATKReportColumn gATKReportColumn : this.columnInfo) {
            out.print(SEPARATOR + gATKReportColumn.getFormat());
        }
        out.println(ENDLINE);
        out.printf("#:GATKTable:%s:%s\n", this.tableName, this.tableDescription);
        boolean needsPadding = false;
        for (GATKReportColumn column : this.columnInfo) {
            if (needsPadding) {
                out.printf("  ", new Object[0]);
            }
            needsPadding = true;
            out.printf(column.getColumnFormat().getNameFormat(), column.getColumnName());
        }
        out.println();
        switch (sortingWay) {
            case SORT_BY_COLUMN: {
                Collections.sort(this.underlyingData, ROW_COMPARATOR);
                for (Object[] row : this.underlyingData) {
                    this.writeRow(out, row);
                }
                break;
            }
            case SORT_BY_ROW: {
                TreeMap<Object, Integer> treeMap;
                if (this.rowIdToIndex.size() != this.underlyingData.size()) {
                    throw new GATKException("There isn't a 1-to-1 mapping from row ID to index; this can happen when rows are not created consistently");
                }
                try {
                    treeMap = new TreeMap<Object, Integer>(this.rowIdToIndex);
                }
                catch (ClassCastException e) {
                    throw new GATKException("Unable to sort the rows based on the row IDs because the ID Objects are of different types");
                }
                for (Map.Entry entry : treeMap.entrySet()) {
                    this.writeRow(out, this.underlyingData.get((Integer)entry.getValue()));
                }
                break;
            }
            case DO_NOT_SORT: {
                for (Object[] objectArray : this.underlyingData) {
                    this.writeRow(out, objectArray);
                }
                break;
            }
        }
        out.println();
    }

    private void writeRow(PrintStream out, Object[] row) {
        boolean needsPadding = false;
        for (int i = 0; i < row.length; ++i) {
            if (needsPadding) {
                out.printf("  ", new Object[0]);
            }
            needsPadding = true;
            Object obj = row[i];
            GATKReportColumn info = this.columnInfo.get(i);
            String value = obj == null ? "null" : (info.getDataType().equals((Object)GATKReportDataType.Unknown) && (obj instanceof Double || obj instanceof Float) ? String.format("%.8f", obj) : (obj instanceof Double || obj instanceof Float ? (Double.isFinite(Double.parseDouble(obj.toString())) ? String.format(info.getFormat(), obj) : obj.toString()) : String.format(info.getFormat(), obj)));
            out.printf(info.getColumnFormat().getValueFormat(), value);
        }
        out.println();
    }

    public int getNumRows() {
        return this.underlyingData.size();
    }

    public int getNumColumns() {
        return this.columnInfo.size();
    }

    public List<GATKReportColumn> getColumnInfo() {
        return this.columnInfo;
    }

    public String getTableName() {
        return this.tableName;
    }

    public String getTableDescription() {
        return this.tableDescription;
    }

    public void concat(GATKReportTable table) {
        if (!this.isSameFormat(table)) {
            throw new GATKException("Error trying to concatenate tables with different formats");
        }
        this.underlyingData.addAll(table.underlyingData);
        int currentNumRows = this.getNumRows();
        for (Map.Entry<Object, Integer> entry : table.rowIdToIndex.entrySet()) {
            this.rowIdToIndex.put(entry.getKey(), entry.getValue() + currentNumRows);
        }
    }

    public boolean isSameFormat(GATKReportTable table) {
        if (!this.tableName.equals(table.tableName) || !this.tableDescription.equals(table.tableDescription) || this.columnInfo.size() != table.columnInfo.size()) {
            return false;
        }
        for (int i = 0; i < this.columnInfo.size(); ++i) {
            if (this.columnInfo.get(i).getFormat().equals(table.columnInfo.get(i).getFormat()) && this.columnInfo.get(i).getColumnName().equals(table.columnInfo.get(i).getColumnName())) continue;
            return false;
        }
        return true;
    }

    public boolean equals(GATKReportTable table) {
        if (!this.isSameFormat(table) || this.underlyingData.size() != table.underlyingData.size()) {
            return false;
        }
        List<Object[]> myOrderedRows = this.getOrderedRows();
        List<Object[]> otherOrderedRows = table.getOrderedRows();
        for (int i = 0; i < this.underlyingData.size(); ++i) {
            Object[] myData = myOrderedRows.get(i);
            Object[] otherData = otherOrderedRows.get(i);
            for (int j = 0; j < myData.length; ++j) {
                if (myData[j].toString().equals(otherData[j].toString())) continue;
                return false;
            }
        }
        return true;
    }

    private List<Object[]> getOrderedRows() {
        switch (this.sortingWay) {
            case SORT_BY_COLUMN: {
                Collections.sort(this.underlyingData, ROW_COMPARATOR);
                return this.underlyingData;
            }
            case SORT_BY_ROW: {
                TreeMap<Object, Integer> sortedMap;
                try {
                    sortedMap = new TreeMap<Object, Integer>(this.rowIdToIndex);
                }
                catch (ClassCastException e) {
                    return this.underlyingData;
                }
                ArrayList<Object[]> orderedData = new ArrayList<Object[]>(this.underlyingData.size());
                for (int rowKey : sortedMap.values()) {
                    orderedData.add(this.underlyingData.get(rowKey));
                }
                return orderedData;
            }
        }
        return this.underlyingData;
    }

    protected static enum TableNameHeaderFields {
        NAME(2),
        DESCRIPTION(3);

        private final int index;

        private TableNameHeaderFields(int index) {
            this.index = index;
        }

        public int index() {
            return this.index;
        }
    }

    public static enum Sorting {
        SORT_BY_ROW,
        SORT_BY_COLUMN,
        DO_NOT_SORT;

    }

    protected static enum TableDataHeaderFields {
        COLS(2),
        ROWS(3),
        FORMAT_START(4);

        private final int index;

        private TableDataHeaderFields(int index) {
            this.index = index;
        }

        public int index() {
            return this.index;
        }
    }
}

