/*
 * Copyright (C) 2014 Samuel Audet
 *
 * Licensed either under the Apache License, Version 2.0, or (at your option)
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation (subject to the "Classpath" exception),
 * either version 2, or any later version (collectively, the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     http://www.gnu.org/licenses/
 *     http://www.gnu.org/software/classpath/license.html
 *
 * or as provided in the LICENSE.txt file that accompanied this code.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.bytedeco.javacpp.indexer;

/**
 * An indexer for a {@code double[]} array.
 *
 * @author Samuel Audet
 */
public class DoubleArrayIndexer extends DoubleIndexer {
    /** The backing array. */
    protected double[] array;

    /** Constructor to set the {@link #array}, {@link #sizes} and {@link #strides}. */
    public DoubleArrayIndexer(double[] array, int[] sizes, int[] strides) {
        super(sizes, strides);
        this.array = array;
    }

    @Override public double[] array() {
        return array;
    }

    @Override public double get(int i) {
        return array[i];
    }
    @Override public DoubleIndexer get(int i, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            d[offset + n] = array[i * strides[0] + n];
        }
        return this;
    }
    @Override public double get(int i, int j) {
        return array[i * strides[0] + j];
    }
    @Override public DoubleIndexer get(int i, int j, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            d[offset + n] = array[i * strides[0] + j * strides[1] + n];
        }
        return this;
    }
    @Override public double get(int i, int j, int k) {
        return array[i * strides[0] + j * strides[1] + k];
    }
    @Override public double get(int ... indices) {
        return array[index(indices)];
    }
    @Override public DoubleIndexer get(int[] indices, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            d[offset + n] = array[index(indices) + n];
        }
        return this;
    }

    @Override public DoubleIndexer put(int i, double d) {
        array[i] = d;
        return this;
    }
    @Override public DoubleIndexer put(int i, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            array[i * strides[0] + n] = d[offset + n];
        }
        return this;
    }
    @Override public DoubleIndexer put(int i, int j, double d) {
        array[i * strides[0] + j] = d;
        return this;
    }
    @Override public DoubleIndexer put(int i, int j, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            array[i * strides[0] + j * strides[1] + n] = d[offset + n];
        }
        return this;
    }
    @Override public DoubleIndexer put(int i, int j, int k, double d) {
        array[i * strides[0] + j * strides[1] + k] = d;
        return this;
    }
    @Override public DoubleIndexer put(int[] indices, double d) {
        array[index(indices)] = d;
        return this;
    }
    @Override public DoubleIndexer put(int[] indices, double[] d, int offset, int length) {
        for (int n = 0; n < length; n++) {
            array[index(indices) + n] = d[offset + n];
        }
        return this;
    }

    @Override public void release() { array = null; }
}
