/*
 * Copyright 2016 camunda services GmbH.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.camunda.bpm.engine.impl.cmd;

import java.io.Serializable;
import org.camunda.bpm.engine.exception.NotFoundException;
import org.camunda.bpm.engine.history.UserOperationLogEntry;
import org.camunda.bpm.engine.impl.cfg.CommandChecker;
import org.camunda.bpm.engine.impl.interceptor.Command;
import org.camunda.bpm.engine.impl.interceptor.CommandContext;
import org.camunda.bpm.engine.impl.persistence.entity.PropertyChange;
import org.camunda.bpm.engine.impl.persistence.entity.UserOperationLogManager;
import org.camunda.bpm.engine.repository.ProcessDefinition;
import static org.camunda.bpm.engine.impl.util.EnsureUtil.ensureNotNull;

/**
 * Command to delete a process definition form a deployment.
 *
 * @author Christopher Zell <christopher.zell@camunda.com>
 */
public class DeleteProcessDefinitionCmd implements Command<Void>, Serializable {

  private final String processDefinitionId;
  private final Boolean cascade;
  private final Boolean skipCustomListeners;

  public DeleteProcessDefinitionCmd(String processDefinitionId, Boolean cascade, Boolean skipCustomListeners) {
    this.processDefinitionId = processDefinitionId;
    this.cascade = cascade;
    this.skipCustomListeners = skipCustomListeners;
  }

  @Override
  public Void execute(CommandContext commandContext) {
    ensureNotNull("processDefinitionId", processDefinitionId);

    ProcessDefinition processDefinition = commandContext.getProcessDefinitionManager()
                                                .findLatestProcessDefinitionById(processDefinitionId);
    ensureNotNull(NotFoundException.class, "No process definition found with id '" + processDefinitionId + "'", "processDefinition", processDefinition);

    for(CommandChecker checker : commandContext.getProcessEngineConfiguration().getCommandCheckers()) {
      checker.checkDeleteProcessDefinitionById(processDefinitionId);
    }

    UserOperationLogManager logManager = commandContext.getOperationLogManager();
    logManager.logProcessDefinitionOperation(UserOperationLogEntry.OPERATION_TYPE_DELETE,
                                             processDefinitionId,
                                             processDefinition.getKey(),
                                             new PropertyChange("cascade", null, cascade));

    commandContext.getProcessDefinitionManager().deleteProcessDefinition(processDefinition, processDefinitionId, cascade, cascade, skipCustomListeners);
    return null;
  }
}
