/*
 * Copyright 2019: Christophe Saint-Marcel
 * This software is part of the Caseine project.
 * This software was developped with the support of the following organizations:
 * Université Grenoble Alpes
 * Institut Polytechnique de Grenoble
 * 
 * 
 * This file is part of the VPL Design Tool.
 * The VPL Design Tool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * The VPL Design Tool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the VPL Design Tool.  If not, see <https://www.gnu.org/licenses/>.
 */
package caseine.project;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.Arrays;

import caseine.Caseine;
import caseine.PythonLauncher;
import caseine.exceptions.PythonFileMissingException;
import caseine.exceptions.TestDirectoryMissingException;
import caseine.exceptions.UnitTestsFileMissingException;

/**
 * The base class for python projects.
 * @author christophe
 *
 */
public class CaseinePythonProject extends CaseineCommonProject implements CaseineProject {


	/**
	 * 
	 * @param projectPath the path
	 * @param vplId the identifier of the VPL
	 * @param url the caseine server URL
	 * @param token the caseine token
	 */
	public CaseinePythonProject(String projectPath, String vplId, String url, String token, boolean settings) {
		super(projectPath, vplId, url, token, settings);
	}

	@Override
	public ProjectType getType() {
		return ProjectType.PYTHON;
	}

	/**
	 * Generates the project template.
	 * 
	 * @param mvn True if we must generate the maven pom.xml
	 * @param template
	 * 
	 * @throws CaseineProjectAlreadyExistingException if an existing project is present
	 * @throws BadIDEException if IDE is not supported
	 * @throws IOException if IO failures occur
	 * @throws TestDirectoryMissingException if the directory of test is not here
	 * @throws UnitTestsFileMissingException if test file is missing
	 * @throws FileMissingException if the root file is missing
	 */
	@Override
	public void generate(boolean mvn, int template) throws CaseineProjectAlreadyExistingException, BadIDEException,
			IOException, TestDirectoryMissingException, FileMissingException, UnitTestsFileMissingException {

		if (isCaseine()) {
			throw new CaseineProjectAlreadyExistingException("A project already exists in " + projectPath);
		}
		File root = new File(projectPath);
		if (!root.exists()) {
			root.mkdir();
		}

		caseine.FileUtils.copyAZipAndUnZipIt(Caseine.class.getResourceAsStream("/python/default/t1.zip"), root,
				".");

		boolean rootWasEmptyFirst = !Arrays.stream(root.listFiles()).filter(f -> !f.isHidden())
				.filter(f -> !"vpl_unittest.py".equals(f.getName())).findFirst().isPresent();
		if (rootWasEmptyFirst) {
			writeATemplate(root);
		} else {
			try {
				checkRoot(root);
			} catch (PythonFileMissingException e) {
				throw new FileMissingException(e);
			}
		}

		// Adds vplId in the project
		fillCaseine(vplId);
		// Fills the .caseine with the project name
		fillCaseineProjectFile();

	}

	private void writeATemplate(File root) throws TestDirectoryMissingException {
		File test = new File(root, "test");
		test.mkdirs();
		if (test == null || !test.exists() || !test.isDirectory()) {
			throw new TestDirectoryMissingException("It should have a test directory");
		}
		caseine.FileUtils.copyAZipAndUnZipIt(PythonLauncher.class.getResourceAsStream("/python-templates/t1.zip"), root,
				".");
		System.out.println("A python project template was generated because the directory was empty");
		System.out.println("You can use it like this");
		System.out.println("You can change existing and/or add new files");
	}

	private void checkRoot(File root)
			throws PythonFileMissingException, UnitTestsFileMissingException, TestDirectoryMissingException {
		if (!Arrays.stream(root.listFiles()).filter(f -> f.getName().endsWith(".py")).filter(f -> !f.isHidden())
				.filter(f -> !"vpl_unittest.py".equals(f.getName())).findAny().isPresent()) {
			throw new PythonFileMissingException("It should have at least one python file in this directory");
		}
		File test = new File(root, "test");
		if (test == null || !test.exists() || !test.isDirectory()) {
			throw new TestDirectoryMissingException("It should have a test directory");
		}

		if (!Arrays.stream(test.listFiles()).filter(f -> f.getName().endsWith(".py")).findAny().isPresent()) {
			throw new UnitTestsFileMissingException("It should have at least one unit test file in test directory");
		}
	}

	/**
	 * Generates the local caseine templates into caseine-output.
	 * 
	 * @param cl an optional classloader to provide dependencies
	 * @throws IOException for IO issues
	 * @throws ClassNotFoundException if the class is not found
	 * @throws MavenProjectException if the 
	 */
	public void local(ClassLoader cl) throws IOException, ClassNotFoundException, MavenProjectException {

		if (vplId.equals(VPLID_0)) {
			vplId = getVplId();
		} else { // Update the caseine project file
			fillCaseine(vplId);
		}

		// Clean to be confident about the result
		clean();

		try {
			PythonLauncher.launch(Paths.get(projectPath).toAbsolutePath().toString());
		} catch (Exception ex) {
			System.err.println(ex.getMessage());
		}
	}
}
