/**
 * <h1>cdk8s+ (cdk8s-plus)</h1>
 * <p>
 * <blockquote>
 * <p>
 * <img alt="Experimental" src="https://img.shields.io/badge/experimental-important.svg?style=for-the-badge"><br><br>
 * This library is in very early stages of development, as such, and in correspondence with a <code>0.x</code> semantic major version line, its <code>API</code> is
 * likely to rapidly change in breaking ways. It is therefore not recommended to use library for production workloads.
 * <p>
 * </blockquote>
 * <p>
 * <strong>cdk8s+</strong> is a software development framework that provides high level abstractions for authoring Kubernetes applications.
 * Built on top of the auto generated building blocks provided by <a href="../cdk8s">cdk8s</a>, this library includes a hand crafted <em>construct</em>
 * for each native kubernetes object, exposing richer API's with reduced complexity.
 * <p>
 * <h2>Kubernetes Spec</h2>
 * <p>
 * <strong>cdk8s+</strong> is currently built on top of version <a href="https://github.com/instrumenta/kubernetes-json-schema/tree/master/v1.17.0">1.17.0</a> of the kubernetes API specifications.
 * If you are deploying manifests produced by <code>cdk8s+</code> onto clusters of a lower version, you might encounter some unsupported spec properties or invalid manifests.
 * <p>
 * <blockquote>
 * <p>
 * See <a href="https://github.com/awslabs/cdk8s/issues/299">Supporting various k8s API specs</a> for more details and progress on this issue.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Letter Of Intent</h2>
 * <p>
 * We strive to develop this library with full transparency and as much community feedback and contributions as possible.
 * To that end, we publish this development version. The lack of features/capabilities is intentional, we look forward to build and expand this framework with the help of the community.
 * <p>
 * <blockquote>
 * <p>
 * If you are interested in contributing, see <a href="./CONTRIBUTING.md">Contribution Guide</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h2>At a glance</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * import path.*;
 * 
 * // our cdk app
 * App app = new App();
 * 
 * // our kuberentes chart
 * Chart chart = new Chart(app, "Chart");
 * 
 * // lets create a volume that contains our app.
 * // we use a trick with a config map!
 * ConfigMap appData = new ConfigMap(chart, "AppData");
 * appData.addDirectory(path.join(__dirname, "app"));
 * 
 * Volume appVolume = kplus.Volume.fromConfigMap(appData);
 * 
 * // now we create a container that runs our app
 * String appPath = "/var/lib/app";
 * int port = 80;
 * Container container = new Container(new ContainerProps()
 *         .image("node:14.4.0-alpine3.12")
 *         .command(asList("node", "index.js", port))
 *         .port(port)
 *         .workingDir(appPath));
 * 
 * // make the app accessible to the container
 * container.mount(appPath, appVolume);
 * 
 * // now lets create a deployment to run a few instances of this container
 * Deployment deployment = new Deployment(chart, "Deployment", new DeploymentProps()
 *         .replicas(3)
 *         .containers(asList(container)));
 * 
 * // finally, we expose the deployment as a load balancer service and make it run
 * deployment.expose(8080, new ExposeOptions().serviceType(kplus.ServiceType.getLOAD_BALANCER()));
 * 
 * // we are done, synth
 * app.synth();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * apiVersion: v1
 * data:
 *   index.js: |-
 *     var http = require('http');
 * 
 *     var port = process.argv[2];
 * 
 *     //create a server object:
 *     http.createServer(function (req, res) {
 *       res.write('Hello World!'); //write a response to the client
 *       res.end(); //end the response
 *     }).listen(port); //the server object listens on port 80
 * kind: ConfigMap
 * metadata:
 *   annotations: {}
 *   labels: {}
 *   name: chart-appdata-configmap-da4c63ab
 * ---
 * apiVersion: apps/v1
 * kind: Deployment
 * metadata:
 *   annotations: {}
 *   labels: {}
 *   name: chart-deployment-pod-d4285cc9
 * spec:
 *   replicas: 3
 *   selector:
 *     matchLabels:
 *       cdk8s.deployment: ChartDeploymentCFC2E30C
 *   template:
 *     metadata:
 *       annotations: {}
 *       labels:
 *         cdk8s.deployment: ChartDeploymentCFC2E30C
 *     spec:
 *       containers:
 *         - command:
 *             - node
 *             - index.js
 *             - "80"
 *           env: []
 *           image: node:14.4.0-alpine3.12
 *           name: main
 *           ports:
 *             - containerPort: 80
 *           volumeMounts:
 *             - mountPath: /var/lib/app
 *               name: configmap-chart-appdata-configmap-da4c63ab
 *           workingDir: /var/lib/app
 *       volumes:
 *         - configMap:
 *             name: chart-appdata-configmap-da4c63ab
 *           name: configmap-chart-appdata-configmap-da4c63ab
 * ---
 * apiVersion: v1
 * kind: Service
 * metadata:
 *   annotations: {}
 *   labels: {}
 *   name: chart-deployment-service-pod-42f50c26
 * spec:
 *   externalIPs: []
 *   ports:
 *     - port: 8080
 *       targetPort: 80
 *   selector:
 *     cdk8s.deployment: ChartDeploymentCFC2E30C
 *   type: LoadBalancer
 * </pre></blockquote>
 * <p>
 * <h2>Installation and Usage</h2>
 * <p>
 * We currently support both Python and TypeScript/JavaScript. More languages are coming soon.
 * <p>
 * <blockquote>
 * <p>
 * We would love to hear which languages you want to see next: <a href="https://github.com/awslabs/cdk8s/issues/134">Languages Support</a>
 * <p>
 * </blockquote>
 * <p>
 * <h3>TypeScript/JavaScript</h3>
 * <p>
 * <code>❯ npm install cdk8s-plus cdk8s</code>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * new Deployment(chart, "Deployment", new DeploymentProps()
 *         .replicas(3)
 *         .containers(asList(new Container(new ContainerProps()
 *                 .image("ubuntu")))));
 * </pre></blockquote>
 * <p>
 * <h3>Python</h3>
 * <p>
 * <code>❯ pip install cdk8s-plus cdk8s</code>
 * <p>
 * <blockquote><pre>
 * import cdk8s_plus as kplus
 * import cdk8s
 * 
 * app = cdk8s.App()
 * chart = cdk8s.Chart(app, 'Chart')
 * 
 * kplus.Deployment(chart, 'Deployment',
 *   replicas=1,
 *   containers=[kplus.Container(image='ubuntu')]
 * )
 * </pre></blockquote>
 * <p>
 * <h2>In Depth</h2>
 * <p>
 * Following are excerpts for the usage of every <em>construct</em> provided by this library. It details the commonly used patterns and configuration properties.
 * In general, every such construct can be configured using two mechanisms:
 * <p>
 * <ul>
 * <li>Spec Constructor Properties</li>
 * <li>Post Instantiation Spec Mutations</li>
 * </ul>
 * <p>
 * The documentation presented here focuses on post instantiation mutations, however, every such mutation can also be pre-configured
 * using constructor properties for the corresponding spec. A complete API reference can be found in <a href="./API.md">here</a>.
 * <p>
 * <h3><code>Container</code></h3>
 * <p>
 * Define containers that run in a pod using the <code>Container</code> class.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-container">Container</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Environment variables</h4>
 * <p>
 * Environment variables can be added to containers using various sources, via semantically explicit API's:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * Container container = new Container(new ContainerProps()
 *         .image("my-app"));
 * 
 * // explicitly use a value.
 * container.addEnv("endpoint", kplus.EnvValue.fromValue("value"));
 * 
 * // use a specific key from a config map.
 * IConfigMap backendsConfig = kplus.ConfigMap.fromConfigMapName("backends");
 * container.addEnv("endpoint", kplus.EnvValue.fromConfigMap(backendsConfig, "endpoint"));
 * 
 * // use a specific key from a secret.
 * ISecret credentials = kplus.Secret.fromSecretName("credentials");
 * container.addEnv("password", kplus.EnvValue.fromSecret(credentials, "password"));
 * </pre></blockquote>
 * <p>
 * <h4>Volume Mounts</h4>
 * <p>
 * A very common capability is to mount a volume with some data onto a container. Using pure kubernetes API, this would require writing something like:
 * <p>
 * <blockquote><pre>
 * kind: Pod
 * apiVersion: v1
 * spec:
 *   containers:
 *     - name: main
 *       volumeMounts:
 *         - mountPath: /path/to/mount
 *           name: 'config-volume'
 *   volumes:
 *     - name: 'config-volume'
 *       configMap:
 *         name: 'config'
 * </pre></blockquote>
 * <p>
 * Notice the apparent redundancy of having to specify the volume name twice. Also, if you happen to need the same mount in other pods,
 * you would need to duplicate this configuration. This can get complex and cluttered very fast.
 * <p>
 * In contrast, here is how to do this with <code>cdk8s+</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * IConfigMap config = kplus.ConfigMap.fromConfigMapName("config");
 * Volume volume = kplus.Volume.fromConfigMap(config);
 * 
 * Container container = new Container(new ContainerProps()
 *         .image("my-app"));
 * 
 * // Cool alert: every pod that will later be configured with this container,
 * // will automatically have access to this volume, so you don't need to explicitly add it to the pod spec!.
 * container.mount("/path/to/mount", volume);
 * </pre></blockquote>
 * <p>
 * <h2>Probes</h2>
 * <p>
 * A <a href="https://kubernetes.io/docs/reference/generated/kubernetes-api/v1.19/#probe-v1-core">Probe</a> is a diagnostic performed periodically by the kubelet on a Container. To
 * perform a diagnostic, the kubelet calls a Handler implemented by the container.
 * <p>
 * A <code>Probe</code> instance can be created through one of the <code>fromXxx</code> static methods:
 * <p>
 * <ul>
 * <li><code>Probe.fromHttpGet()</code></li>
 * <li><code>Probe.fromCommand()</code></li>
 * </ul>
 * <p>
 * Readiness probes can be configured at the container-level through the <code>readiness</code> option:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Container.Builder.create()
 *         // ...
 *         .readiness(kplus.Probe.fromHttpGet("/ping"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the API reference for details.
 * <p>
 * <h3><code>Volume</code></h3>
 * <p>
 * Volume represents a named volume in a pod that may be accessed by any container in the pod.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-volume">Volume</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Create from a ConfigMap</h4>
 * <p>
 * A very useful operation is to create a volume from a <code>ConfigMap</code>. Kubernetes will translate every key in the config map to a file,
 * who's content is the value of the key.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * IConfigMap configMap = kplus.ConfigMap.fromConfigMapName("redis-config");
 * Volume configVolume = kplus.Volume.fromConfigMap(configMap);
 * </pre></blockquote>
 * <p>
 * <h4>Create from an EmptyDir</h4>
 * <p>
 * The easiest way to allocate some persistent storage to your container is to create a volume from an empty directory.
 * This volume, as the name suggests, is initially empty, and can be written to by containers who mount it.
 * The data in the volume is preserved throughout the lifecycle of the pod, but is deleted forever as soon as the pod itself is removed.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * Volume data = kplus.Volume.fromEmptyDir(configMap);
 * 
 * Container redis = new Container(new ContainerProps()
 *         .image("redis"));
 * 
 * // mount to the redis container.
 * redis.mount("/var/lib/redis", data);
 * </pre></blockquote>
 * <p>
 * <h3><code>Job</code></h3>
 * <p>
 * Jobs are a very useful concept in kubernetes deployments.
 * They can be used for add-hoc provisioning tasks, as well as long running processing jobs.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-job">Job</a>
 * <p>
 * </blockquote>
 * <p>
 * In configuration, they don't differ much from regular pods, but offer some additional properties.
 * <p>
 * <h4>Delete a Job after its finished</h4>
 * <p>
 * You can configure a TTL for the job after it finished its execution successfully.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.*;
 * import org.cdk8s.plus.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * // let's define a job spec, and set a 1 second TTL.
 * Job load = new Job(chart, "LoadData", new JobProps()
 *         .ttlAfterFinished(kplus.Duration.seconds(1)));
 * 
 * // now add a container to all the pods created by this job
 * job.addContainer(new Container(new ContainerProps()
 *         .image("loader")));
 * </pre></blockquote>
 * <p>
 * <h3><code>Service</code></h3>
 * <p>
 * Use services when you want to expose a set of pods using a stable network identity. They can also be used for externalizing
 * endpoints to clients outside of the kubernetes cluster.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-service">Service</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Selectors</h4>
 * <p>
 * Services must be configured with selectors that tell it which pods should it serve.
 * The most common selector method is using labels.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.*;
 * import org.cdk8s.plus.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * Service frontends = new Service(chart, "FrontEnds");
 * 
 * // this will cause the service to select all pods with the 'run: frontend' label.
 * frontends.selectByLabel("run", "frontend");
 * </pre></blockquote>
 * <p>
 * <h4>Ports</h4>
 * <p>
 * Ports that the service will listen and redirect to can be configured like so:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.*;
 * import org.cdk8s.plus.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * Service frontends = new Service(chart, "FrontEnds");
 * 
 * // make the service bind to port 9000 and redirect to port 80 on the associated containers.
 * frontends.serve(Map.of("port", 9000, "targetPort", 80));
 * </pre></blockquote>
 * <p>
 * <h3><code>Deployment</code></h3>
 * <p>
 * Create a deployment to govern the lifecycle and orchestration of a set of identical pods.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-deployment">Deployment</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Automatic pod selection</h4>
 * <p>
 * When you specify pods in a deployment, you normally have to configure the appropriate labels and selectors to
 * make the deployment control the relevant pods. This construct does this automatically.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.*;
 * import org.cdk8s.plus.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * new Deployment(chart, "FrontEnds", new DeploymentProps()
 *         .containers(asList(new Container(new ContainerProps().image("node")))));
 * </pre></blockquote>
 * <p>
 * Note the resulting manifest contains a special <code>cdk8s.deployment</code> label that is applied to the pods, and is used as
 * the selector for the deployment.
 * <p>
 * <blockquote><pre>
 * apiVersion: apps/v1
 * kind: Deployment
 * metadata:
 *   annotations: {}
 *   labels: {}
 *   name: chart-frontends-pod-a48e7f2e
 * spec:
 *   replicas: 1
 *   selector:
 *     matchLabels:
 *       cdk8s.deployment: ChartFrontEndsDD8A97CE
 *   template:
 *     metadata:
 *       annotations: {}
 *       labels:
 *         cdk8s.deployment: ChartFrontEndsDD8A97CE
 * </pre></blockquote>
 * <p>
 * <h4>Exposing via a service</h4>
 * <p>
 * Following up on pod selection, you can also easily create a service that will select the pods relevant to the deployment.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // store the deployment to created in a constant
 * var frontends = new Deployment(chart, "FrontEnds");
 * 
 * // create a ClusterIP service that listens on port 9000 and redirects to port 9000 on the containers.
 * frontends.expose(9000);
 * </pre></blockquote>
 * <p>
 * Notice the resulting manifest, will have the same <code>cdk8s.deployment</code> magic label as the selector.
 * This will cause the service to attach to the pods that were configured as part of the said deployment.
 * <p>
 * <blockquote><pre>
 * apiVersion: v1
 * kind: Service
 * metadata:
 *   annotations: {}
 *   labels: {}
 *   name: chart-frontends-service-pod-1f70150b
 * spec:
 *   externalIPs: []
 *   ports:
 *     - port: 9000
 *   selector:
 *     cdk8s.deployment: ChartFrontEndsDD8A97CE
 *   type: ClusterIP
 * </pre></blockquote>
 * <p>
 * <h3><code>ConfigMap</code></h3>
 * <p>
 * ConfigMap are used to store configuration data. They provide a dictionary based data structure that can be consumed in
 * various shapes and forms.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-configmap">ConfigMap</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Use an existing <code>ConfigMap</code></h4>
 * <p>
 * You can reference to an existing <code>ConfigMap</code> like so. Note that this does not create a new object,
 * and will therefore not be included in the resulting manifest.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * IConfigMap config = kplus.ConfigMap.fromConfigMapName("config");
 * 
 * // the 'config' constant can later be used by API's that require an IConfigMap.
 * // for example when creating a volume.
 * Volume volume = kplus.Volume.fromConfigMap(config);
 * </pre></blockquote>
 * <p>
 * <h4>Adding data</h4>
 * <p>
 * You can create config maps and add some data to them like so:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * var config = new new ConfigMap(chart, "Config")();
 * config.addData("url", "https://my-endpoint:8080");
 * </pre></blockquote>
 * <p>
 * <h4>Creating a volume from a directory</h4>
 * <p>
 * Here is a nifty little trick you can use to create a volume that contains a directory on the client machine (machine that runs <code>cdk8s synth</code>):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * import path.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * var appMap = new new ConfigMap(chart, "Config")();
 * 
 * // add the files in the directory to the config map.
 * // this will create a key for each file.
 * // note that only top level files will be included, sub-directories are not yet supported.
 * appMap.addDirectory(path.join(__dirname, "app"));
 * 
 * Volume appVolume = kplus.Volume.fromConfigMap(appMap);
 * 
 * // for here, just mount the volume to a container, and run your app!
 * String mountPath = "/var/app";
 * Container container = new Container(new ContainerProps()
 *         .image("node")
 *         .command(asList("node", "app.js"))
 *         .workingDir(mountPath));
 * 
 * container.mount(mountPath, appVolume);
 * </pre></blockquote>
 * <p>
 * <h3><code>Pod</code></h3>
 * <p>
 * A pod is essentially a collection of containers. It is the most fundamental computation unit that can be provisioned.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-pod">Pod</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Adding Containers/Volumes</h4>
 * <p>
 * Containers and volumes can be added to pod definition like so:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * Container container = new Container(new ContainerProps()
 *         .image("node"));
 * 
 * Volume storage = kplus.Volume.fromEmptyDir("storage");
 * 
 * container.mount("/data", storage);
 * 
 * var app = new App();
 * var chart = new Chart(app, "Chart");
 * 
 * var pod = new new Pod(chart, "Pod")();
 * 
 * // this will automatically add the volume as well.
 * pod.addContainer(container);
 * 
 * // but if you want to explicitly add it, simply use:
 * pod.addVolume(storage);
 * </pre></blockquote>
 * <p>
 * <h4>Applying a restart policy</h4>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * var app = new App();
 * var chart = new Chart(app, "Chart");
 * 
 * var pod = new new Pod(chart, "Pod", new PodProps()
 *         .restartPolicy(NEVER))();
 * </pre></blockquote>
 * <p>
 * <h4>Assigning a ServiceAccount</h4>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * var app = new App();
 * var chart = new Chart(app, "Chart");
 * 
 * var pod = new new Pod(chart, "Pod", new PodProps()
 *         .serviceAccount(fromServiceAccountName("aws")))();
 * </pre></blockquote>
 * <p>
 * <h3><code>Secret</code></h3>
 * <p>
 * Secrets are used to store confidential information. Never store such information on the definition of the pod itself.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-secret">Secret</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Use an existing <code>Secret</code></h4>
 * <p>
 * To reference a secret created outside of your deployment definition, use the following. Note that this does not create a new object,
 * and will therefore not be included in the resulting manifest.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * ISecret secret = kplus.Secret.fromSecretName("aws-creds");
 * </pre></blockquote>
 * <p>
 * <h4>Adding data</h4>
 * <p>
 * To create a new secret with some data, use:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * Secret secret = new Secret(chart, "Secret");
 * secret.addStringData("password", "some-encrypted-data");
 * </pre></blockquote>
 * <p>
 * <h3><code>ServiceAccount</code></h3>
 * <p>
 * Use service accounts to provide an identity for pods.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-serviceaccount">ServiceAccount</a>
 * <p>
 * </blockquote>
 * <p>
 * <h4>Use an existing <code>ServiceAccount</code></h4>
 * <p>
 * To reference a service account created outside of your deployment definition, use the following. Note that this does not create a new object,
 * and will therefore not be included in the resulting manifest.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * 
 * IServiceAccount serviceAccount = kplus.ServiceAccount.fromServiceAccountName("aws-service");
 * </pre></blockquote>
 * <p>
 * <h4>Allowing access to secrets</h4>
 * <p>
 * To create a new service account, and give it access to some secrets, use the following:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import org.cdk8s.plus.*;
 * import org.cdk8s.*;
 * 
 * App app = new App();
 * Chart chart = new Chart(app, "Chart");
 * 
 * ISecret awsCreds = kplus.Secret.fromSecretName("aws-creds");
 * ServiceAccount awsService = new ServiceAccount(chart, "AWS");
 * 
 * // give access to the aws creds secret.
 * awsService.addSecret(awsCreds);
 * </pre></blockquote>
 * <p>
 * <h3><code>Ingress</code></h3>
 * <p>
 * <a href="https://kubernetes.io/docs/concepts/services-networking/ingress/">Ingress</a> manages external access to services in a cluster, typically through
 * HTTP. Ingress may provide load balancing, SSL termination and name-based virtual
 * hosting.
 * <p>
 * You must have an <a href="https://kubernetes.io/docs/concepts/services-networking/ingress-controllers">Ingress controller</a> to satisfy an Ingress. Only creating an
 * Ingress resource has no effect.
 * <p>
 * <blockquote>
 * <p>
 * API Reference: <a href="./API.md#cdk8s-plus-ingress">Ingress</a>
 * <p>
 * </blockquote>
 * <p>
 * The following example will route HTTP requests sent to the <code>/hello</code> url prefix
 * to a service associated with a deployment of the
 * <a href="https://github.com/hashicorp/http-echo">hashicorp/http-echo</a> image.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var helloDeployment = Deployment.Builder.create(this, text)
 *         .containers(asList(
 *             Container.Builder.create()
 *                     .image("hashicorp/http-echo")
 *                     .args(asList("-text", "hello ingress"))
 *                     .build()))
 *         .build();
 * 
 * var helloService = helloDeployment.expose(5678);
 * 
 * var ingress = new Ingress(this, "ingress");
 * ingress.addRule("/hello", kplus.IngressBackend.fromService(helloService));
 * </pre></blockquote>
 * <p>
 * You can use <code>addHostRule(host, path, backend)</code> to define a route that will only
 * apply to requests with this <code>Host</code> header. This can be used to implement virtual
 * hosts.
 * <p>
 * The <code>addDefaultBackend(backend)</code> and <code>addHostDefaultBackend(host, backend)</code>
 * methods can be used to define backends that will accept all requests that do not
 * match any other rules.
 * <p>
 * The TCP port used to route requests to services will be determined based on
 * which ports are exposed by the service (e.g. through <code>serve()</code>). If the service
 * exposes multiple ports, then a port must be specified via
 * <code>IngressBackend.fromService(service, { port }</code>.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package org.cdk8s.plus;
