/*
 * Copyright Red Hat Inc. and/or its affiliates and other contributors
 * as indicated by the authors tag. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License version 2.
 * 
 * This particular file is subject to the "Classpath" exception as provided in the 
 * LICENSE file that accompanied this code.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

package com.redhat.ceylon.compiler.java.codegen;

import com.redhat.ceylon.compiler.typechecker.tree.Node;
import com.redhat.ceylon.compiler.typechecker.tree.Tree;
import com.redhat.ceylon.compiler.typechecker.tree.Visitor;
import com.redhat.ceylon.model.typechecker.model.Class;
import com.redhat.ceylon.model.typechecker.model.ClassOrInterface;
import com.redhat.ceylon.model.typechecker.model.Value;

/**
 * Visits everything and return true if it finds a non-alias ClassOrInterface or object definition.
 * 
 * @author Stéphane Épardaud <stef@epardaud.fr>
 */
public class HasTypeVisitor extends Visitor {
    
    @SuppressWarnings("serial")
    private static class HasTypeException extends RuntimeException {}
    
    public boolean hasType(Node target) {
        try{
            visitAny(target);
            return false;
        }catch(HasTypeException x){
            return true;
        }
    }

    @Override
    public void handleException(Exception e, Node that) {
        // rethrow
        throw (RuntimeException)e;
    }

    @Override
    public void visit(Tree.ClassOrInterface that){
        ClassOrInterface model = that.getDeclarationModel();
        // stop at aliases, do not collect them since we can never create any instance of them
        // and they are useless at runtime
        if(!model.isAlias())
            throw new HasTypeException();
    }

    @Override
    public void visit(Tree.ObjectDefinition that){
        Value model = that.getDeclarationModel();
        if(model != null)
            throw new HasTypeException();
    }

    @Override
    public void visit(Tree.ObjectArgument that){
        Value model = that.getDeclarationModel();
        if(model != null)
            throw new HasTypeException();
    }

    @Override
    public void visit(Tree.ObjectExpression that){
        Class model = that.getAnonymousClass();
        if(model != null)
            throw new HasTypeException();
    }
}
