/*
 * Copyright Red Hat Inc. and/or its affiliates and other contributors
 * as indicated by the authors tag. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License version 2.
 * 
 * This particular file is subject to the "Classpath" exception as provided in the 
 * LICENSE file that accompanied this code.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package com.redhat.ceylon.compiler.java.loader;

import static com.redhat.ceylon.compiler.typechecker.tree.TreeUtil.isForBackend;

import com.redhat.ceylon.common.Backend;
import com.redhat.ceylon.compiler.typechecker.tree.Tree;
import com.redhat.ceylon.compiler.typechecker.tree.Tree.Declaration;
import com.redhat.ceylon.compiler.typechecker.tree.Tree.ModuleDescriptor;
import com.redhat.ceylon.compiler.typechecker.tree.Tree.PackageDescriptor;
import com.redhat.ceylon.compiler.typechecker.tree.Visitor;

public abstract class SourceDeclarationVisitor extends Visitor {
    
    public abstract void loadFromSource(ModuleDescriptor that);
    public abstract void loadFromSource(PackageDescriptor that);
    public abstract void loadFromSource(Tree.Declaration decl);
    
    // Returns `true` if the declaration is not marked `native` or if
    // it has a "jvm" argument. In the case it is a native header
    // it will also return `true` for methods and attributes that
    // have an implementation. In all other cases returns `false`.
    protected boolean checkNative(Tree.Declaration decl){
        return isForBackend(decl.getAnnotationList(), Backend.Java, decl.getUnit())
                || (isForBackend(decl.getAnnotationList(), Backend.Header, decl.getUnit())
                        && hasImplementation(decl));
    }
    
    private boolean hasImplementation(Declaration decl) {
        if (decl instanceof Tree.AttributeDeclaration) {
            return ((Tree.AttributeDeclaration)decl).getSpecifierOrInitializerExpression() != null;
        }
        else if (decl instanceof Tree.AttributeGetterDefinition) {
            return ((Tree.AttributeGetterDefinition)decl).getBlock() != null;
        }
        else if (decl instanceof Tree.AttributeSetterDefinition) {
            return ((Tree.AttributeSetterDefinition)decl).getBlock() != null;
        }
        else if (decl instanceof Tree.MethodDeclaration) {
            return ((Tree.MethodDeclaration)decl).getSpecifierExpression() != null;
        }
        else if (decl instanceof Tree.MethodDefinition) {
            return ((Tree.MethodDefinition)decl).getBlock() != null;
        }
        return false;
    }
    
    @Override
    public void visit(Tree.ModuleDescriptor that) {
        loadFromSource(that);
    }
    @Override
    public void visit(Tree.PackageDescriptor that) {
        loadFromSource(that);
    }
    
    @Override
    public void visit(Tree.TypeAliasDeclaration that) {
        loadFromSource(that);
    }

    @Override
    public void visit(Tree.AnyClass that) {
        loadFromSource(that);
    }
    
    @Override
    public void visit(Tree.AnyInterface that) {
        loadFromSource(that);
    }
    
    @Override
    public void visit(Tree.ObjectDefinition that) {
        loadFromSource(that);
    }

    @Override
    public void visit(Tree.AnyMethod that) {
        loadFromSource(that);
    }

    @Override
    public void visit(Tree.AttributeDeclaration that) {
        loadFromSource(that);
    }

    @Override
    public void visit(Tree.AttributeGetterDefinition that) {
        loadFromSource(that);
    }
}