package com.redhat.ceylon.compiler.typechecker.tree;

import java.util.List;
import static com.redhat.ceylon.compiler.typechecker.tree.Tree.*;
import static com.redhat.ceylon.compiler.typechecker.tree.Tree.Package;

public class Walker {

    public static void walkCompilationUnit(Visitor visitor, CompilationUnit node) {
        List<CompilerAnnotation> CompilerAnnotations = node.getCompilerAnnotations();
        for (int i=0,l=CompilerAnnotations.size();i<l;i++){
            CompilerAnnotation subnode = CompilerAnnotations.get(i);
            subnode.visit(visitor);
        }
        if (node.getImportList()!=null)
            node.getImportList().visit(visitor);
        List<ModuleDescriptor> ModuleDescriptors = node.getModuleDescriptors();
        for (int i=0,l=ModuleDescriptors.size();i<l;i++){
            ModuleDescriptor subnode = ModuleDescriptors.get(i);
            subnode.visit(visitor);
        }
        List<PackageDescriptor> PackageDescriptors = node.getPackageDescriptors();
        for (int i=0,l=PackageDescriptors.size();i<l;i++){
            PackageDescriptor subnode = PackageDescriptors.get(i);
            subnode.visit(visitor);
        }
        List<Declaration> Declarations = node.getDeclarations();
        for (int i=0,l=Declarations.size();i<l;i++){
            Declaration subnode = Declarations.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkModuleDescriptor(Visitor visitor, ModuleDescriptor node) {
        walkStatementOrArgument(visitor, node);
        if (node.getAnnotationList()!=null)
            node.getAnnotationList().visit(visitor);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
        if (node.getNamespace()!=null)
            node.getNamespace().visit(visitor);
        if (node.getGroupImportPath()!=null)
            node.getGroupImportPath().visit(visitor);
        if (node.getGroupQuotedLiteral()!=null)
            node.getGroupQuotedLiteral().visit(visitor);
        if (node.getArtifact()!=null)
            node.getArtifact().visit(visitor);
        if (node.getClassifier()!=null)
            node.getClassifier().visit(visitor);
        if (node.getVersion()!=null)
            node.getVersion().visit(visitor);
        if (node.getImportModuleList()!=null)
            node.getImportModuleList().visit(visitor);
    }

    public static void walkPackageDescriptor(Visitor visitor, PackageDescriptor node) {
        walkStatementOrArgument(visitor, node);
        if (node.getAnnotationList()!=null)
            node.getAnnotationList().visit(visitor);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
    }

    public static void walkImportModuleList(Visitor visitor, ImportModuleList node) {
        List<AnyAttribute> AnyAttributes = node.getConstants();
        for (int i=0,l=AnyAttributes.size();i<l;i++){
            AnyAttribute subnode = AnyAttributes.get(i);
            subnode.visit(visitor);
        }
        List<ImportModule> ImportModules = node.getImportModules();
        for (int i=0,l=ImportModules.size();i<l;i++){
            ImportModule subnode = ImportModules.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkImportModule(Visitor visitor, ImportModule node) {
        walkStatementOrArgument(visitor, node);
        if (node.getAnnotationList()!=null)
            node.getAnnotationList().visit(visitor);
        if (node.getNamespace()!=null)
            node.getNamespace().visit(visitor);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
        if (node.getQuotedLiteral()!=null)
            node.getQuotedLiteral().visit(visitor);
        if (node.getArtifact()!=null)
            node.getArtifact().visit(visitor);
        if (node.getClassifier()!=null)
            node.getClassifier().visit(visitor);
        if (node.getVersion()!=null)
            node.getVersion().visit(visitor);
        if (node.getConstantVersion()!=null)
            node.getConstantVersion().visit(visitor);
    }

    public static void walkImportList(Visitor visitor, ImportList node) {
        List<Import> Imports = node.getImports();
        for (int i=0,l=Imports.size();i<l;i++){
            Import subnode = Imports.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkImport(Visitor visitor, Import node) {
        walkStatementOrArgument(visitor, node);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
        if (node.getImportMemberOrTypeList()!=null)
            node.getImportMemberOrTypeList().visit(visitor);
    }

    public static void walkImportPath(Visitor visitor, ImportPath node) {
        List<Identifier> Identifiers = node.getIdentifiers();
        for (int i=0,l=Identifiers.size();i<l;i++){
            Identifier subnode = Identifiers.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkImportMemberOrTypeList(Visitor visitor, ImportMemberOrTypeList node) {
        List<ImportMemberOrType> ImportMemberOrTypes = node.getImportMemberOrTypes();
        for (int i=0,l=ImportMemberOrTypes.size();i<l;i++){
            ImportMemberOrType subnode = ImportMemberOrTypes.get(i);
            subnode.visit(visitor);
        }
        if (node.getImportWildcard()!=null)
            node.getImportWildcard().visit(visitor);
    }

    public static void walkImportMemberOrType(Visitor visitor, ImportMemberOrType node) {
        walkStatementOrArgument(visitor, node);
        if (node.getAlias()!=null)
            node.getAlias().visit(visitor);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getImportMemberOrTypeList()!=null)
            node.getImportMemberOrTypeList().visit(visitor);
    }

    public static void walkImportMember(Visitor visitor, ImportMember node) {
        walkImportMemberOrType(visitor, node);
    }

    public static void walkImportType(Visitor visitor, ImportType node) {
        walkImportMemberOrType(visitor, node);
    }

    public static void walkAlias(Visitor visitor, Alias node) {
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
    }

    public static void walkImportWildcard(Visitor visitor, ImportWildcard node) {
    }

    public static void walkDeclaration(Visitor visitor, Declaration node) {
        walkStatement(visitor, node);
        if (node.getAnnotationList()!=null)
            node.getAnnotationList().visit(visitor);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
    }

    public static void walkMissingDeclaration(Visitor visitor, MissingDeclaration node) {
        walkDeclaration(visitor, node);
    }

    public static void walkTypeDeclaration(Visitor visitor, TypeDeclaration node) {
        walkDeclaration(visitor, node);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
    }

    public static void walkClassOrInterface(Visitor visitor, ClassOrInterface node) {
        walkTypeDeclaration(visitor, node);
    }

    public static void walkTypeAliasDeclaration(Visitor visitor, TypeAliasDeclaration node) {
        walkTypeDeclaration(visitor, node);
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
        if (node.getTypeSpecifier()!=null)
            node.getTypeSpecifier().visit(visitor);
    }

    public static void walkSatisfiedTypes(Visitor visitor, SatisfiedTypes node) {
        List<StaticType> StaticTypes = node.getTypes();
        for (int i=0,l=StaticTypes.size();i<l;i++){
            StaticType subnode = StaticTypes.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkAbstractedType(Visitor visitor, AbstractedType node) {
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkCaseTypes(Visitor visitor, CaseTypes node) {
        List<StaticType> StaticTypes = node.getTypes();
        for (int i=0,l=StaticTypes.size();i<l;i++){
            StaticType subnode = StaticTypes.get(i);
            subnode.visit(visitor);
        }
        List<StaticMemberOrTypeExpression> StaticMemberOrTypeExpressions = node.getBaseMemberExpressions();
        for (int i=0,l=StaticMemberOrTypeExpressions.size();i<l;i++){
            StaticMemberOrTypeExpression subnode = StaticMemberOrTypeExpressions.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkExtendedType(Visitor visitor, ExtendedType node) {
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getInvocationExpression()!=null)
            node.getInvocationExpression().visit(visitor);
    }

    public static void walkTypeConstraintList(Visitor visitor, TypeConstraintList node) {
        List<TypeConstraint> TypeConstraints = node.getTypeConstraints();
        for (int i=0,l=TypeConstraints.size();i<l;i++){
            TypeConstraint subnode = TypeConstraints.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkTypeConstraint(Visitor visitor, TypeConstraint node) {
        walkTypeDeclaration(visitor, node);
        if (node.getCaseTypes()!=null)
            node.getCaseTypes().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getAbstractedType()!=null)
            node.getAbstractedType().visit(visitor);
    }

    public static void walkTypeSpecifier(Visitor visitor, TypeSpecifier node) {
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkDefaultTypeArgument(Visitor visitor, DefaultTypeArgument node) {
        walkTypeSpecifier(visitor, node);
    }

    public static void walkClassSpecifier(Visitor visitor, ClassSpecifier node) {
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getInvocationExpression()!=null)
            node.getInvocationExpression().visit(visitor);
    }

    public static void walkAnyClass(Visitor visitor, AnyClass node) {
        walkClassOrInterface(visitor, node);
        if (node.getParameterList()!=null)
            node.getParameterList().visit(visitor);
        if (node.getCaseTypes()!=null)
            node.getCaseTypes().visit(visitor);
        if (node.getExtendedType()!=null)
            node.getExtendedType().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
    }

    public static void walkClassDefinition(Visitor visitor, ClassDefinition node) {
        walkAnyClass(visitor, node);
        if (node.getClassBody()!=null)
            node.getClassBody().visit(visitor);
    }

    public static void walkEnumerated(Visitor visitor, Enumerated node) {
        walkDeclaration(visitor, node);
        if (node.getDelegatedConstructor()!=null)
            node.getDelegatedConstructor().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkConstructor(Visitor visitor, Constructor node) {
        walkDeclaration(visitor, node);
        if (node.getParameterList()!=null)
            node.getParameterList().visit(visitor);
        if (node.getDelegatedConstructor()!=null)
            node.getDelegatedConstructor().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkDelegatedConstructor(Visitor visitor, DelegatedConstructor node) {
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getInvocationExpression()!=null)
            node.getInvocationExpression().visit(visitor);
    }

    public static void walkClassDeclaration(Visitor visitor, ClassDeclaration node) {
        walkAnyClass(visitor, node);
        if (node.getClassSpecifier()!=null)
            node.getClassSpecifier().visit(visitor);
    }

    public static void walkAnyInterface(Visitor visitor, AnyInterface node) {
        walkClassOrInterface(visitor, node);
        if (node.getCaseTypes()!=null)
            node.getCaseTypes().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
    }

    public static void walkInterfaceDefinition(Visitor visitor, InterfaceDefinition node) {
        walkAnyInterface(visitor, node);
        if (node.getInterfaceBody()!=null)
            node.getInterfaceBody().visit(visitor);
    }

    public static void walkInterfaceDeclaration(Visitor visitor, InterfaceDeclaration node) {
        walkAnyInterface(visitor, node);
        if (node.getTypeSpecifier()!=null)
            node.getTypeSpecifier().visit(visitor);
    }

    public static void walkTypedDeclaration(Visitor visitor, TypedDeclaration node) {
        walkDeclaration(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkAnyAttribute(Visitor visitor, AnyAttribute node) {
        walkTypedDeclaration(visitor, node);
    }

    public static void walkAttributeDeclaration(Visitor visitor, AttributeDeclaration node) {
        walkAnyAttribute(visitor, node);
        if (node.getSpecifierOrInitializerExpression()!=null)
            node.getSpecifierOrInitializerExpression().visit(visitor);
    }

    public static void walkAttributeGetterDefinition(Visitor visitor, AttributeGetterDefinition node) {
        walkAnyAttribute(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkAttributeSetterDefinition(Visitor visitor, AttributeSetterDefinition node) {
        walkTypedDeclaration(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkAnyMethod(Visitor visitor, AnyMethod node) {
        walkTypedDeclaration(visitor, node);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
        List<ParameterList> ParameterLists = node.getParameterLists();
        for (int i=0,l=ParameterLists.size();i<l;i++){
            ParameterList subnode = ParameterLists.get(i);
            subnode.visit(visitor);
        }
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
    }

    public static void walkMethodDefinition(Visitor visitor, MethodDefinition node) {
        walkAnyMethod(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkMethodDeclaration(Visitor visitor, MethodDeclaration node) {
        walkAnyMethod(visitor, node);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkVoidModifier(Visitor visitor, VoidModifier node) {
        walkType(visitor, node);
    }

    public static void walkObjectDefinition(Visitor visitor, ObjectDefinition node) {
        walkTypedDeclaration(visitor, node);
        if (node.getExtendedType()!=null)
            node.getExtendedType().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getClassBody()!=null)
            node.getClassBody().visit(visitor);
    }

    public static void walkParameterList(Visitor visitor, ParameterList node) {
        List<Parameter> Parameters = node.getParameters();
        for (int i=0,l=Parameters.size();i<l;i++){
            Parameter subnode = Parameters.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkParameter(Visitor visitor, Parameter node) {
    }

    public static void walkParameterDeclaration(Visitor visitor, ParameterDeclaration node) {
        walkParameter(visitor, node);
        if (node.getTypedDeclaration()!=null)
            node.getTypedDeclaration().visit(visitor);
    }

    public static void walkValueParameterDeclaration(Visitor visitor, ValueParameterDeclaration node) {
        walkParameterDeclaration(visitor, node);
    }

    public static void walkFunctionalParameterDeclaration(Visitor visitor, FunctionalParameterDeclaration node) {
        walkParameterDeclaration(visitor, node);
    }

    public static void walkInitializerParameter(Visitor visitor, InitializerParameter node) {
        walkParameter(visitor, node);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkPatternParameter(Visitor visitor, PatternParameter node) {
        walkParameter(visitor, node);
        if (node.getPattern()!=null)
            node.getPattern().visit(visitor);
    }

    public static void walkTypeParameterList(Visitor visitor, TypeParameterList node) {
        List<TypeParameterDeclaration> TypeParameterDeclarations = node.getTypeParameterDeclarations();
        for (int i=0,l=TypeParameterDeclarations.size();i<l;i++){
            TypeParameterDeclaration subnode = TypeParameterDeclarations.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkTypeParameterDeclaration(Visitor visitor, TypeParameterDeclaration node) {
        walkDeclaration(visitor, node);
        if (node.getTypeVariance()!=null)
            node.getTypeVariance().visit(visitor);
        if (node.getTypeSpecifier()!=null)
            node.getTypeSpecifier().visit(visitor);
    }

    public static void walkTypeVariance(Visitor visitor, TypeVariance node) {
    }

    public static void walkBody(Visitor visitor, Body node) {
        if (node.getImportList()!=null)
            node.getImportList().visit(visitor);
        List<Statement> Statements = node.getStatements();
        for (int i=0,l=Statements.size();i<l;i++){
            Statement subnode = Statements.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkBlock(Visitor visitor, Block node) {
        walkBody(visitor, node);
    }

    public static void walkClassBody(Visitor visitor, ClassBody node) {
        walkBody(visitor, node);
    }

    public static void walkInterfaceBody(Visitor visitor, InterfaceBody node) {
        walkBody(visitor, node);
    }

    public static void walkType(Visitor visitor, Type node) {
    }

    public static void walkStaticType(Visitor visitor, StaticType node) {
        walkType(visitor, node);
        if (node.getTypeVariance()!=null)
            node.getTypeVariance().visit(visitor);
    }

    public static void walkGroupedType(Visitor visitor, GroupedType node) {
        walkStaticType(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkSimpleType(Visitor visitor, SimpleType node) {
        walkStaticType(visitor, node);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getTypeArgumentList()!=null)
            node.getTypeArgumentList().visit(visitor);
    }

    public static void walkBaseType(Visitor visitor, BaseType node) {
        walkSimpleType(visitor, node);
    }

    public static void walkQualifiedType(Visitor visitor, QualifiedType node) {
        walkSimpleType(visitor, node);
        if (node.getOuterType()!=null)
            node.getOuterType().visit(visitor);
    }

    public static void walkUnionType(Visitor visitor, UnionType node) {
        walkStaticType(visitor, node);
        List<StaticType> StaticTypes = node.getStaticTypes();
        for (int i=0,l=StaticTypes.size();i<l;i++){
            StaticType subnode = StaticTypes.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkIntersectionType(Visitor visitor, IntersectionType node) {
        walkStaticType(visitor, node);
        List<StaticType> StaticTypes = node.getStaticTypes();
        for (int i=0,l=StaticTypes.size();i<l;i++){
            StaticType subnode = StaticTypes.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkSequenceType(Visitor visitor, SequenceType node) {
        walkStaticType(visitor, node);
        if (node.getElementType()!=null)
            node.getElementType().visit(visitor);
        if (node.getLength()!=null)
            node.getLength().visit(visitor);
    }

    public static void walkIterableType(Visitor visitor, IterableType node) {
        walkStaticType(visitor, node);
        if (node.getElementType()!=null)
            node.getElementType().visit(visitor);
    }

    public static void walkOptionalType(Visitor visitor, OptionalType node) {
        walkStaticType(visitor, node);
        if (node.getDefiniteType()!=null)
            node.getDefiniteType().visit(visitor);
    }

    public static void walkTupleType(Visitor visitor, TupleType node) {
        walkStaticType(visitor, node);
        List<Type> Types = node.getElementTypes();
        for (int i=0,l=Types.size();i<l;i++){
            Type subnode = Types.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkFunctionType(Visitor visitor, FunctionType node) {
        walkStaticType(visitor, node);
        if (node.getReturnType()!=null)
            node.getReturnType().visit(visitor);
        List<Type> Types = node.getArgumentTypes();
        for (int i=0,l=Types.size();i<l;i++){
            Type subnode = Types.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkEntryType(Visitor visitor, EntryType node) {
        walkStaticType(visitor, node);
        if (node.getKeyType()!=null)
            node.getKeyType().visit(visitor);
        if (node.getValueType()!=null)
            node.getValueType().visit(visitor);
    }

    public static void walkTypeConstructor(Visitor visitor, TypeConstructor node) {
        walkStaticType(visitor, node);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
    }

    public static void walkSuperType(Visitor visitor, SuperType node) {
        walkStaticType(visitor, node);
    }

    public static void walkMetaLiteral(Visitor visitor, MetaLiteral node) {
        walkPrimary(visitor, node);
    }

    public static void walkTypeLiteral(Visitor visitor, TypeLiteral node) {
        walkMetaLiteral(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkMemberLiteral(Visitor visitor, MemberLiteral node) {
        walkMetaLiteral(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getTypeArgumentList()!=null)
            node.getTypeArgumentList().visit(visitor);
    }

    public static void walkClassLiteral(Visitor visitor, ClassLiteral node) {
        walkTypeLiteral(visitor, node);
    }

    public static void walkNewLiteral(Visitor visitor, NewLiteral node) {
        walkTypeLiteral(visitor, node);
    }

    public static void walkInterfaceLiteral(Visitor visitor, InterfaceLiteral node) {
        walkTypeLiteral(visitor, node);
    }

    public static void walkAliasLiteral(Visitor visitor, AliasLiteral node) {
        walkTypeLiteral(visitor, node);
    }

    public static void walkTypeParameterLiteral(Visitor visitor, TypeParameterLiteral node) {
        walkTypeLiteral(visitor, node);
    }

    public static void walkValueLiteral(Visitor visitor, ValueLiteral node) {
        walkMemberLiteral(visitor, node);
    }

    public static void walkFunctionLiteral(Visitor visitor, FunctionLiteral node) {
        walkMemberLiteral(visitor, node);
    }

    public static void walkModuleLiteral(Visitor visitor, ModuleLiteral node) {
        walkMetaLiteral(visitor, node);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
    }

    public static void walkPackageLiteral(Visitor visitor, PackageLiteral node) {
        walkMetaLiteral(visitor, node);
        if (node.getImportPath()!=null)
            node.getImportPath().visit(visitor);
    }

    public static void walkDynamicModifier(Visitor visitor, DynamicModifier node) {
        walkType(visitor, node);
    }

    public static void walkLocalModifier(Visitor visitor, LocalModifier node) {
        walkType(visitor, node);
    }

    public static void walkValueModifier(Visitor visitor, ValueModifier node) {
        walkLocalModifier(visitor, node);
    }

    public static void walkFunctionModifier(Visitor visitor, FunctionModifier node) {
        walkLocalModifier(visitor, node);
    }

    public static void walkSyntheticVariable(Visitor visitor, SyntheticVariable node) {
        walkValueModifier(visitor, node);
    }

    public static void walkTypeArguments(Visitor visitor, TypeArguments node) {
    }

    public static void walkTypeArgumentList(Visitor visitor, TypeArgumentList node) {
        walkTypeArguments(visitor, node);
        List<Type> Types = node.getTypes();
        for (int i=0,l=Types.size();i<l;i++){
            Type subnode = Types.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkInferredTypeArguments(Visitor visitor, InferredTypeArguments node) {
        walkTypeArguments(visitor, node);
    }

    public static void walkSequencedType(Visitor visitor, SequencedType node) {
        walkType(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkDefaultedType(Visitor visitor, DefaultedType node) {
        walkType(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkSpreadType(Visitor visitor, SpreadType node) {
        walkType(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkDirective(Visitor visitor, Directive node) {
        walkExecutableStatement(visitor, node);
    }

    public static void walkReturn(Visitor visitor, Return node) {
        walkDirective(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkThrow(Visitor visitor, Throw node) {
        walkDirective(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkContinue(Visitor visitor, Continue node) {
        walkDirective(visitor, node);
    }

    public static void walkBreak(Visitor visitor, Break node) {
        walkDirective(visitor, node);
    }

    public static void walkStatementOrArgument(Visitor visitor, StatementOrArgument node) {
        List<CompilerAnnotation> CompilerAnnotations = node.getCompilerAnnotations();
        for (int i=0,l=CompilerAnnotations.size();i<l;i++){
            CompilerAnnotation subnode = CompilerAnnotations.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkStatement(Visitor visitor, Statement node) {
        walkStatementOrArgument(visitor, node);
    }

    public static void walkCompilerAnnotation(Visitor visitor, CompilerAnnotation node) {
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getStringLiteral()!=null)
            node.getStringLiteral().visit(visitor);
    }

    public static void walkExecutableStatement(Visitor visitor, ExecutableStatement node) {
        walkStatement(visitor, node);
    }

    public static void walkAssertion(Visitor visitor, Assertion node) {
        walkExecutableStatement(visitor, node);
        if (node.getAnnotationList()!=null)
            node.getAnnotationList().visit(visitor);
        if (node.getConditionList()!=null)
            node.getConditionList().visit(visitor);
    }

    public static void walkSpecifierStatement(Visitor visitor, SpecifierStatement node) {
        walkExecutableStatement(visitor, node);
        if (node.getBaseMemberExpression()!=null)
            node.getBaseMemberExpression().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkExpressionStatement(Visitor visitor, ExpressionStatement node) {
        walkExecutableStatement(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkPattern(Visitor visitor, Pattern node) {
    }

    public static void walkVariablePattern(Visitor visitor, VariablePattern node) {
        walkPattern(visitor, node);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
    }

    public static void walkTuplePattern(Visitor visitor, TuplePattern node) {
        walkPattern(visitor, node);
        List<Pattern> Patterns = node.getPatterns();
        for (int i=0,l=Patterns.size();i<l;i++){
            Pattern subnode = Patterns.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkKeyValuePattern(Visitor visitor, KeyValuePattern node) {
        walkPattern(visitor, node);
        if (node.getKey()!=null)
            node.getKey().visit(visitor);
        if (node.getValue()!=null)
            node.getValue().visit(visitor);
    }

    public static void walkDestructure(Visitor visitor, Destructure node) {
        walkExecutableStatement(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getPattern()!=null)
            node.getPattern().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkControlStatement(Visitor visitor, ControlStatement node) {
        walkExecutableStatement(visitor, node);
    }

    public static void walkControlClause(Visitor visitor, ControlClause node) {
    }

    public static void walkDynamicStatement(Visitor visitor, DynamicStatement node) {
        walkControlStatement(visitor, node);
        if (node.getDynamicClause()!=null)
            node.getDynamicClause().visit(visitor);
    }

    public static void walkDynamicClause(Visitor visitor, DynamicClause node) {
        walkControlClause(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkLetExpression(Visitor visitor, LetExpression node) {
        walkTerm(visitor, node);
        if (node.getLetClause()!=null)
            node.getLetClause().visit(visitor);
    }

    public static void walkLetClause(Visitor visitor, LetClause node) {
        walkControlClause(visitor, node);
        List<Statement> Statements = node.getVariables();
        for (int i=0,l=Statements.size();i<l;i++){
            Statement subnode = Statements.get(i);
            subnode.visit(visitor);
        }
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkIfStatement(Visitor visitor, IfStatement node) {
        walkControlStatement(visitor, node);
        if (node.getIfClause()!=null)
            node.getIfClause().visit(visitor);
        if (node.getElseClause()!=null)
            node.getElseClause().visit(visitor);
    }

    public static void walkIfClause(Visitor visitor, IfClause node) {
        walkControlClause(visitor, node);
        if (node.getConditionList()!=null)
            node.getConditionList().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkElseClause(Visitor visitor, ElseClause node) {
        walkControlClause(visitor, node);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkSwitchStatement(Visitor visitor, SwitchStatement node) {
        walkControlStatement(visitor, node);
        if (node.getSwitchClause()!=null)
            node.getSwitchClause().visit(visitor);
        if (node.getSwitchCaseList()!=null)
            node.getSwitchCaseList().visit(visitor);
    }

    public static void walkSwitchClause(Visitor visitor, SwitchClause node) {
        if (node.getSwitched()!=null)
            node.getSwitched().visit(visitor);
    }

    public static void walkSwitched(Visitor visitor, Switched node) {
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkSwitchCaseList(Visitor visitor, SwitchCaseList node) {
        List<CaseClause> CaseClauses = node.getCaseClauses();
        for (int i=0,l=CaseClauses.size();i<l;i++){
            CaseClause subnode = CaseClauses.get(i);
            subnode.visit(visitor);
        }
        if (node.getElseClause()!=null)
            node.getElseClause().visit(visitor);
    }

    public static void walkCaseClause(Visitor visitor, CaseClause node) {
        walkControlClause(visitor, node);
        if (node.getCaseItem()!=null)
            node.getCaseItem().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkCaseItem(Visitor visitor, CaseItem node) {
    }

    public static void walkMatchCase(Visitor visitor, MatchCase node) {
        walkCaseItem(visitor, node);
        if (node.getExpressionList()!=null)
            node.getExpressionList().visit(visitor);
    }

    public static void walkIsCase(Visitor visitor, IsCase node) {
        walkCaseItem(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
    }

    public static void walkPatternCase(Visitor visitor, PatternCase node) {
        walkCaseItem(visitor, node);
        if (node.getPattern()!=null)
            node.getPattern().visit(visitor);
    }

    public static void walkSatisfiesCase(Visitor visitor, SatisfiesCase node) {
        walkCaseItem(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkTryCatchStatement(Visitor visitor, TryCatchStatement node) {
        walkControlStatement(visitor, node);
        if (node.getTryClause()!=null)
            node.getTryClause().visit(visitor);
        List<CatchClause> CatchClauses = node.getCatchClauses();
        for (int i=0,l=CatchClauses.size();i<l;i++){
            CatchClause subnode = CatchClauses.get(i);
            subnode.visit(visitor);
        }
        if (node.getFinallyClause()!=null)
            node.getFinallyClause().visit(visitor);
    }

    public static void walkTryClause(Visitor visitor, TryClause node) {
        walkControlClause(visitor, node);
        if (node.getResourceList()!=null)
            node.getResourceList().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkCatchClause(Visitor visitor, CatchClause node) {
        walkControlClause(visitor, node);
        if (node.getCatchVariable()!=null)
            node.getCatchVariable().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkFinallyClause(Visitor visitor, FinallyClause node) {
        walkControlClause(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkResourceList(Visitor visitor, ResourceList node) {
        List<Resource> Resources = node.getResources();
        for (int i=0,l=Resources.size();i<l;i++){
            Resource subnode = Resources.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkResource(Visitor visitor, Resource node) {
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkCatchVariable(Visitor visitor, CatchVariable node) {
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
    }

    public static void walkForStatement(Visitor visitor, ForStatement node) {
        walkControlStatement(visitor, node);
        if (node.getForClause()!=null)
            node.getForClause().visit(visitor);
        if (node.getElseClause()!=null)
            node.getElseClause().visit(visitor);
    }

    public static void walkForClause(Visitor visitor, ForClause node) {
        walkControlClause(visitor, node);
        if (node.getForIterator()!=null)
            node.getForIterator().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkForIterator(Visitor visitor, ForIterator node) {
        walkStatementOrArgument(visitor, node);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkValueIterator(Visitor visitor, ValueIterator node) {
        walkForIterator(visitor, node);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
    }

    public static void walkPatternIterator(Visitor visitor, PatternIterator node) {
        walkForIterator(visitor, node);
        if (node.getPattern()!=null)
            node.getPattern().visit(visitor);
    }

    public static void walkWhileStatement(Visitor visitor, WhileStatement node) {
        walkControlStatement(visitor, node);
        if (node.getWhileClause()!=null)
            node.getWhileClause().visit(visitor);
    }

    public static void walkWhileClause(Visitor visitor, WhileClause node) {
        walkControlClause(visitor, node);
        if (node.getConditionList()!=null)
            node.getConditionList().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkConditionList(Visitor visitor, ConditionList node) {
        List<Condition> Conditions = node.getConditions();
        for (int i=0,l=Conditions.size();i<l;i++){
            Condition subnode = Conditions.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkCondition(Visitor visitor, Condition node) {
    }

    public static void walkBooleanCondition(Visitor visitor, BooleanCondition node) {
        walkCondition(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkExistsOrNonemptyCondition(Visitor visitor, ExistsOrNonemptyCondition node) {
        walkCondition(visitor, node);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
        if (node.getBrokenExpression()!=null)
            node.getBrokenExpression().visit(visitor);
    }

    public static void walkExistsCondition(Visitor visitor, ExistsCondition node) {
        walkExistsOrNonemptyCondition(visitor, node);
    }

    public static void walkNonemptyCondition(Visitor visitor, NonemptyCondition node) {
        walkExistsOrNonemptyCondition(visitor, node);
    }

    public static void walkIsCondition(Visitor visitor, IsCondition node) {
        walkCondition(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getVariable()!=null)
            node.getVariable().visit(visitor);
    }

    public static void walkSatisfiesCondition(Visitor visitor, SatisfiesCondition node) {
        walkCondition(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
    }

    public static void walkVariable(Visitor visitor, Variable node) {
        walkTypedDeclaration(visitor, node);
        List<ParameterList> ParameterLists = node.getParameterLists();
        for (int i=0,l=ParameterLists.size();i<l;i++){
            ParameterList subnode = ParameterLists.get(i);
            subnode.visit(visitor);
        }
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkTerm(Visitor visitor, Term node) {
    }

    public static void walkOperatorExpression(Visitor visitor, OperatorExpression node) {
        walkTerm(visitor, node);
    }

    public static void walkBinaryOperatorExpression(Visitor visitor, BinaryOperatorExpression node) {
        walkOperatorExpression(visitor, node);
        if (node.getLeftTerm()!=null)
            node.getLeftTerm().visit(visitor);
        if (node.getRightTerm()!=null)
            node.getRightTerm().visit(visitor);
    }

    public static void walkArithmeticOp(Visitor visitor, ArithmeticOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkSumOp(Visitor visitor, SumOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkDifferenceOp(Visitor visitor, DifferenceOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkProductOp(Visitor visitor, ProductOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkQuotientOp(Visitor visitor, QuotientOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkPowerOp(Visitor visitor, PowerOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkRemainderOp(Visitor visitor, RemainderOp node) {
        walkArithmeticOp(visitor, node);
    }

    public static void walkAssignmentOp(Visitor visitor, AssignmentOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkAssignOp(Visitor visitor, AssignOp node) {
        walkAssignmentOp(visitor, node);
    }

    public static void walkArithmeticAssignmentOp(Visitor visitor, ArithmeticAssignmentOp node) {
        walkAssignmentOp(visitor, node);
    }

    public static void walkAddAssignOp(Visitor visitor, AddAssignOp node) {
        walkArithmeticAssignmentOp(visitor, node);
    }

    public static void walkSubtractAssignOp(Visitor visitor, SubtractAssignOp node) {
        walkArithmeticAssignmentOp(visitor, node);
    }

    public static void walkMultiplyAssignOp(Visitor visitor, MultiplyAssignOp node) {
        walkArithmeticAssignmentOp(visitor, node);
    }

    public static void walkDivideAssignOp(Visitor visitor, DivideAssignOp node) {
        walkArithmeticAssignmentOp(visitor, node);
    }

    public static void walkRemainderAssignOp(Visitor visitor, RemainderAssignOp node) {
        walkArithmeticAssignmentOp(visitor, node);
    }

    public static void walkBitwiseAssignmentOp(Visitor visitor, BitwiseAssignmentOp node) {
        walkAssignmentOp(visitor, node);
    }

    public static void walkIntersectAssignOp(Visitor visitor, IntersectAssignOp node) {
        walkBitwiseAssignmentOp(visitor, node);
    }

    public static void walkUnionAssignOp(Visitor visitor, UnionAssignOp node) {
        walkBitwiseAssignmentOp(visitor, node);
    }

    public static void walkComplementAssignOp(Visitor visitor, ComplementAssignOp node) {
        walkBitwiseAssignmentOp(visitor, node);
    }

    public static void walkLogicalAssignmentOp(Visitor visitor, LogicalAssignmentOp node) {
        walkAssignmentOp(visitor, node);
    }

    public static void walkAndAssignOp(Visitor visitor, AndAssignOp node) {
        walkLogicalAssignmentOp(visitor, node);
    }

    public static void walkOrAssignOp(Visitor visitor, OrAssignOp node) {
        walkLogicalAssignmentOp(visitor, node);
    }

    public static void walkLogicalOp(Visitor visitor, LogicalOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkAndOp(Visitor visitor, AndOp node) {
        walkLogicalOp(visitor, node);
    }

    public static void walkOrOp(Visitor visitor, OrOp node) {
        walkLogicalOp(visitor, node);
    }

    public static void walkBitwiseOp(Visitor visitor, BitwiseOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkIntersectionOp(Visitor visitor, IntersectionOp node) {
        walkBitwiseOp(visitor, node);
    }

    public static void walkUnionOp(Visitor visitor, UnionOp node) {
        walkBitwiseOp(visitor, node);
    }

    public static void walkComplementOp(Visitor visitor, ComplementOp node) {
        walkBitwiseOp(visitor, node);
    }

    public static void walkEqualityOp(Visitor visitor, EqualityOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkEqualOp(Visitor visitor, EqualOp node) {
        walkEqualityOp(visitor, node);
    }

    public static void walkNotEqualOp(Visitor visitor, NotEqualOp node) {
        walkEqualityOp(visitor, node);
    }

    public static void walkComparisonOp(Visitor visitor, ComparisonOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkLargerOp(Visitor visitor, LargerOp node) {
        walkComparisonOp(visitor, node);
    }

    public static void walkSmallerOp(Visitor visitor, SmallerOp node) {
        walkComparisonOp(visitor, node);
    }

    public static void walkLargeAsOp(Visitor visitor, LargeAsOp node) {
        walkComparisonOp(visitor, node);
    }

    public static void walkSmallAsOp(Visitor visitor, SmallAsOp node) {
        walkComparisonOp(visitor, node);
    }

    public static void walkScaleOp(Visitor visitor, ScaleOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkBound(Visitor visitor, Bound node) {
        walkTerm(visitor, node);
        if (node.getTerm()!=null)
            node.getTerm().visit(visitor);
    }

    public static void walkOpenBound(Visitor visitor, OpenBound node) {
        walkBound(visitor, node);
    }

    public static void walkClosedBound(Visitor visitor, ClosedBound node) {
        walkBound(visitor, node);
    }

    public static void walkWithinOp(Visitor visitor, WithinOp node) {
        walkOperatorExpression(visitor, node);
        if (node.getUpperBound()!=null)
            node.getUpperBound().visit(visitor);
        if (node.getLowerBound()!=null)
            node.getLowerBound().visit(visitor);
        if (node.getTerm()!=null)
            node.getTerm().visit(visitor);
    }

    public static void walkDefaultOp(Visitor visitor, DefaultOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkThenOp(Visitor visitor, ThenOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkIdenticalOp(Visitor visitor, IdenticalOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkEntryOp(Visitor visitor, EntryOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkRangeOp(Visitor visitor, RangeOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkSegmentOp(Visitor visitor, SegmentOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkCompareOp(Visitor visitor, CompareOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkInOp(Visitor visitor, InOp node) {
        walkBinaryOperatorExpression(visitor, node);
    }

    public static void walkUnaryOperatorExpression(Visitor visitor, UnaryOperatorExpression node) {
        walkOperatorExpression(visitor, node);
        if (node.getTerm()!=null)
            node.getTerm().visit(visitor);
    }

    public static void walkNotOp(Visitor visitor, NotOp node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkExists(Visitor visitor, Exists node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkNonempty(Visitor visitor, Nonempty node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkNegativeOp(Visitor visitor, NegativeOp node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkPositiveOp(Visitor visitor, PositiveOp node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkTypeOperatorExpression(Visitor visitor, TypeOperatorExpression node) {
        walkUnaryOperatorExpression(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkIsOp(Visitor visitor, IsOp node) {
        walkTypeOperatorExpression(visitor, node);
    }

    public static void walkSatisfies(Visitor visitor, Satisfies node) {
        walkTypeOperatorExpression(visitor, node);
    }

    public static void walkExtends(Visitor visitor, Extends node) {
        walkTypeOperatorExpression(visitor, node);
    }

    public static void walkOfOp(Visitor visitor, OfOp node) {
        walkTypeOperatorExpression(visitor, node);
    }

    public static void walkPrefixOperatorExpression(Visitor visitor, PrefixOperatorExpression node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkIncrementOp(Visitor visitor, IncrementOp node) {
        walkPrefixOperatorExpression(visitor, node);
    }

    public static void walkDecrementOp(Visitor visitor, DecrementOp node) {
        walkPrefixOperatorExpression(visitor, node);
    }

    public static void walkPostfixOperatorExpression(Visitor visitor, PostfixOperatorExpression node) {
        walkUnaryOperatorExpression(visitor, node);
    }

    public static void walkPostfixIncrementOp(Visitor visitor, PostfixIncrementOp node) {
        walkPostfixOperatorExpression(visitor, node);
    }

    public static void walkPostfixDecrementOp(Visitor visitor, PostfixDecrementOp node) {
        walkPostfixOperatorExpression(visitor, node);
    }

    public static void walkExpressionList(Visitor visitor, ExpressionList node) {
        List<Expression> Expressions = node.getExpressions();
        for (int i=0,l=Expressions.size();i<l;i++){
            Expression subnode = Expressions.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkExpression(Visitor visitor, Expression node) {
        walkAtom(visitor, node);
        if (node.getTerm()!=null)
            node.getTerm().visit(visitor);
    }

    public static void walkParExpression(Visitor visitor, ParExpression node) {
        walkExpression(visitor, node);
    }

    public static void walkPrimary(Visitor visitor, Primary node) {
        walkTerm(visitor, node);
    }

    public static void walkPostfixExpression(Visitor visitor, PostfixExpression node) {
        walkPrimary(visitor, node);
        if (node.getPrimary()!=null)
            node.getPrimary().visit(visitor);
    }

    public static void walkInvocationExpression(Visitor visitor, InvocationExpression node) {
        walkPostfixExpression(visitor, node);
        if (node.getPositionalArgumentList()!=null)
            node.getPositionalArgumentList().visit(visitor);
        if (node.getNamedArgumentList()!=null)
            node.getNamedArgumentList().visit(visitor);
    }

    public static void walkParameterizedExpression(Visitor visitor, ParameterizedExpression node) {
        walkPrimary(visitor, node);
        if (node.getPrimary()!=null)
            node.getPrimary().visit(visitor);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
        List<ParameterList> ParameterLists = node.getParameterLists();
        for (int i=0,l=ParameterLists.size();i<l;i++){
            ParameterList subnode = ParameterLists.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkMemberOrTypeExpression(Visitor visitor, MemberOrTypeExpression node) {
        walkPrimary(visitor, node);
    }

    public static void walkExtendedTypeExpression(Visitor visitor, ExtendedTypeExpression node) {
        walkMemberOrTypeExpression(visitor, node);
    }

    public static void walkStaticMemberOrTypeExpression(Visitor visitor, StaticMemberOrTypeExpression node) {
        walkMemberOrTypeExpression(visitor, node);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
        if (node.getTypeArguments()!=null)
            node.getTypeArguments().visit(visitor);
    }

    public static void walkBaseMemberOrTypeExpression(Visitor visitor, BaseMemberOrTypeExpression node) {
        walkStaticMemberOrTypeExpression(visitor, node);
    }

    public static void walkBaseMemberExpression(Visitor visitor, BaseMemberExpression node) {
        walkBaseMemberOrTypeExpression(visitor, node);
    }

    public static void walkBaseTypeExpression(Visitor visitor, BaseTypeExpression node) {
        walkBaseMemberOrTypeExpression(visitor, node);
    }

    public static void walkQualifiedMemberOrTypeExpression(Visitor visitor, QualifiedMemberOrTypeExpression node) {
        walkStaticMemberOrTypeExpression(visitor, node);
        if (node.getPrimary()!=null)
            node.getPrimary().visit(visitor);
        if (node.getMemberOperator()!=null)
            node.getMemberOperator().visit(visitor);
    }

    public static void walkQualifiedMemberExpression(Visitor visitor, QualifiedMemberExpression node) {
        walkQualifiedMemberOrTypeExpression(visitor, node);
    }

    public static void walkQualifiedTypeExpression(Visitor visitor, QualifiedTypeExpression node) {
        walkQualifiedMemberOrTypeExpression(visitor, node);
    }

    public static void walkMemberOperator(Visitor visitor, MemberOperator node) {
    }

    public static void walkMemberOp(Visitor visitor, MemberOp node) {
        walkMemberOperator(visitor, node);
    }

    public static void walkSafeMemberOp(Visitor visitor, SafeMemberOp node) {
        walkMemberOperator(visitor, node);
    }

    public static void walkSpreadOp(Visitor visitor, SpreadOp node) {
        walkMemberOperator(visitor, node);
    }

    public static void walkIndexExpression(Visitor visitor, IndexExpression node) {
        walkPostfixExpression(visitor, node);
        if (node.getElementOrRange()!=null)
            node.getElementOrRange().visit(visitor);
    }

    public static void walkElementOrRange(Visitor visitor, ElementOrRange node) {
    }

    public static void walkElement(Visitor visitor, Element node) {
        walkElementOrRange(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkElementRange(Visitor visitor, ElementRange node) {
        walkElementOrRange(visitor, node);
        if (node.getLowerBound()!=null)
            node.getLowerBound().visit(visitor);
        if (node.getUpperBound()!=null)
            node.getUpperBound().visit(visitor);
        if (node.getLength()!=null)
            node.getLength().visit(visitor);
    }

    public static void walkOuter(Visitor visitor, Outer node) {
        walkAtom(visitor, node);
    }

    public static void walkPackage(Visitor visitor, Package node) {
        walkAtom(visitor, node);
    }

    public static void walkArgumentList(Visitor visitor, ArgumentList node) {
    }

    public static void walkNamedArgumentList(Visitor visitor, NamedArgumentList node) {
        walkArgumentList(visitor, node);
        List<NamedArgument> NamedArguments = node.getNamedArguments();
        for (int i=0,l=NamedArguments.size();i<l;i++){
            NamedArgument subnode = NamedArguments.get(i);
            subnode.visit(visitor);
        }
        if (node.getSequencedArgument()!=null)
            node.getSequencedArgument().visit(visitor);
    }

    public static void walkSequencedArgument(Visitor visitor, SequencedArgument node) {
        walkStatementOrArgument(visitor, node);
        List<PositionalArgument> PositionalArguments = node.getPositionalArguments();
        for (int i=0,l=PositionalArguments.size();i<l;i++){
            PositionalArgument subnode = PositionalArguments.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkPositionalArgumentList(Visitor visitor, PositionalArgumentList node) {
        walkArgumentList(visitor, node);
        List<PositionalArgument> PositionalArguments = node.getPositionalArguments();
        for (int i=0,l=PositionalArguments.size();i<l;i++){
            PositionalArgument subnode = PositionalArguments.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkPositionalArgument(Visitor visitor, PositionalArgument node) {
    }

    public static void walkListedArgument(Visitor visitor, ListedArgument node) {
        walkPositionalArgument(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkSpreadArgument(Visitor visitor, SpreadArgument node) {
        walkPositionalArgument(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkFunctionArgument(Visitor visitor, FunctionArgument node) {
        walkTerm(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
        List<ParameterList> ParameterLists = node.getParameterLists();
        for (int i=0,l=ParameterLists.size();i<l;i++){
            ParameterList subnode = ParameterLists.get(i);
            subnode.visit(visitor);
        }
        if (node.getTypeConstraintList()!=null)
            node.getTypeConstraintList().visit(visitor);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
    }

    public static void walkObjectExpression(Visitor visitor, ObjectExpression node) {
        walkPrimary(visitor, node);
        if (node.getExtendedType()!=null)
            node.getExtendedType().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getClassBody()!=null)
            node.getClassBody().visit(visitor);
    }

    public static void walkIfExpression(Visitor visitor, IfExpression node) {
        walkTerm(visitor, node);
        if (node.getIfClause()!=null)
            node.getIfClause().visit(visitor);
        if (node.getElseClause()!=null)
            node.getElseClause().visit(visitor);
    }

    public static void walkSwitchExpression(Visitor visitor, SwitchExpression node) {
        walkTerm(visitor, node);
        if (node.getSwitchClause()!=null)
            node.getSwitchClause().visit(visitor);
        if (node.getSwitchCaseList()!=null)
            node.getSwitchCaseList().visit(visitor);
    }

    public static void walkNamedArgument(Visitor visitor, NamedArgument node) {
        walkStatementOrArgument(visitor, node);
        if (node.getIdentifier()!=null)
            node.getIdentifier().visit(visitor);
    }

    public static void walkSpecifiedArgument(Visitor visitor, SpecifiedArgument node) {
        walkNamedArgument(visitor, node);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkTypedArgument(Visitor visitor, TypedArgument node) {
        walkNamedArgument(visitor, node);
        if (node.getType()!=null)
            node.getType().visit(visitor);
    }

    public static void walkMethodArgument(Visitor visitor, MethodArgument node) {
        walkTypedArgument(visitor, node);
        if (node.getTypeParameterList()!=null)
            node.getTypeParameterList().visit(visitor);
        List<ParameterList> ParameterLists = node.getParameterLists();
        for (int i=0,l=ParameterLists.size();i<l;i++){
            ParameterList subnode = ParameterLists.get(i);
            subnode.visit(visitor);
        }
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkAttributeArgument(Visitor visitor, AttributeArgument node) {
        walkTypedArgument(visitor, node);
        if (node.getBlock()!=null)
            node.getBlock().visit(visitor);
        if (node.getSpecifierExpression()!=null)
            node.getSpecifierExpression().visit(visitor);
    }

    public static void walkObjectArgument(Visitor visitor, ObjectArgument node) {
        walkTypedArgument(visitor, node);
        if (node.getExtendedType()!=null)
            node.getExtendedType().visit(visitor);
        if (node.getSatisfiedTypes()!=null)
            node.getSatisfiedTypes().visit(visitor);
        if (node.getClassBody()!=null)
            node.getClassBody().visit(visitor);
    }

    public static void walkSpecifierOrInitializerExpression(Visitor visitor, SpecifierOrInitializerExpression node) {
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkSpecifierExpression(Visitor visitor, SpecifierExpression node) {
        walkSpecifierOrInitializerExpression(visitor, node);
    }

    public static void walkLazySpecifierExpression(Visitor visitor, LazySpecifierExpression node) {
        walkSpecifierExpression(visitor, node);
    }

    public static void walkInitializerExpression(Visitor visitor, InitializerExpression node) {
        walkSpecifierOrInitializerExpression(visitor, node);
    }

    public static void walkAtom(Visitor visitor, Atom node) {
        walkPrimary(visitor, node);
    }

    public static void walkLiteral(Visitor visitor, Literal node) {
        walkAtom(visitor, node);
    }

    public static void walkNaturalLiteral(Visitor visitor, NaturalLiteral node) {
        walkLiteral(visitor, node);
    }

    public static void walkFloatLiteral(Visitor visitor, FloatLiteral node) {
        walkLiteral(visitor, node);
    }

    public static void walkCharLiteral(Visitor visitor, CharLiteral node) {
        walkLiteral(visitor, node);
    }

    public static void walkStringLiteral(Visitor visitor, StringLiteral node) {
        walkLiteral(visitor, node);
        List<DocLink> DocLinks = node.getDocLinks();
        for (int i=0,l=DocLinks.size();i<l;i++){
            DocLink subnode = DocLinks.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkQuotedLiteral(Visitor visitor, QuotedLiteral node) {
        walkLiteral(visitor, node);
    }

    public static void walkDocLink(Visitor visitor, DocLink node) {
    }

    public static void walkSelfExpression(Visitor visitor, SelfExpression node) {
        walkAtom(visitor, node);
    }

    public static void walkThis(Visitor visitor, This node) {
        walkSelfExpression(visitor, node);
    }

    public static void walkSuper(Visitor visitor, Super node) {
        walkSelfExpression(visitor, node);
    }

    public static void walkSequenceEnumeration(Visitor visitor, SequenceEnumeration node) {
        walkAtom(visitor, node);
        if (node.getSequencedArgument()!=null)
            node.getSequencedArgument().visit(visitor);
        List<Statement> Statements = node.getStatements();
        for (int i=0,l=Statements.size();i<l;i++){
            Statement subnode = Statements.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkTuple(Visitor visitor, Tuple node) {
        walkAtom(visitor, node);
        if (node.getSequencedArgument()!=null)
            node.getSequencedArgument().visit(visitor);
    }

    public static void walkDynamic(Visitor visitor, Dynamic node) {
        walkAtom(visitor, node);
        if (node.getNamedArgumentList()!=null)
            node.getNamedArgumentList().visit(visitor);
    }

    public static void walkStringTemplate(Visitor visitor, StringTemplate node) {
        walkAtom(visitor, node);
        List<StringLiteral> StringLiterals = node.getStringLiterals();
        for (int i=0,l=StringLiterals.size();i<l;i++){
            StringLiteral subnode = StringLiterals.get(i);
            subnode.visit(visitor);
        }
        List<Expression> Expressions = node.getExpressions();
        for (int i=0,l=Expressions.size();i<l;i++){
            Expression subnode = Expressions.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkAnnotation(Visitor visitor, Annotation node) {
        walkInvocationExpression(visitor, node);
    }

    public static void walkAnonymousAnnotation(Visitor visitor, AnonymousAnnotation node) {
        if (node.getStringLiteral()!=null)
            node.getStringLiteral().visit(visitor);
        if (node.getStringTemplate()!=null)
            node.getStringTemplate().visit(visitor);
    }

    public static void walkAnnotationList(Visitor visitor, AnnotationList node) {
        if (node.getAnonymousAnnotation()!=null)
            node.getAnonymousAnnotation().visit(visitor);
        List<Annotation> Annotations = node.getAnnotations();
        for (int i=0,l=Annotations.size();i<l;i++){
            Annotation subnode = Annotations.get(i);
            subnode.visit(visitor);
        }
    }

    public static void walkIdentifier(Visitor visitor, Identifier node) {
    }

    public static void walkComprehension(Visitor visitor, Comprehension node) {
        walkPositionalArgument(visitor, node);
        if (node.getInitialComprehensionClause()!=null)
            node.getInitialComprehensionClause().visit(visitor);
    }

    public static void walkComprehensionClause(Visitor visitor, ComprehensionClause node) {
        walkControlClause(visitor, node);
    }

    public static void walkInitialComprehensionClause(Visitor visitor, InitialComprehensionClause node) {
        walkComprehensionClause(visitor, node);
    }

    public static void walkExpressionComprehensionClause(Visitor visitor, ExpressionComprehensionClause node) {
        walkComprehensionClause(visitor, node);
        if (node.getExpression()!=null)
            node.getExpression().visit(visitor);
    }

    public static void walkForComprehensionClause(Visitor visitor, ForComprehensionClause node) {
        walkInitialComprehensionClause(visitor, node);
        if (node.getForIterator()!=null)
            node.getForIterator().visit(visitor);
        if (node.getComprehensionClause()!=null)
            node.getComprehensionClause().visit(visitor);
    }

    public static void walkIfComprehensionClause(Visitor visitor, IfComprehensionClause node) {
        walkInitialComprehensionClause(visitor, node);
        if (node.getConditionList()!=null)
            node.getConditionList().visit(visitor);
        if (node.getComprehensionClause()!=null)
            node.getComprehensionClause().visit(visitor);
    }

}
