package org.checkerframework.framework.qual;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.checkerframework.framework.util.QualifierPolymorphism;

/**
 * A polymorphic type qualifier that varies over all type hierarchies.
 * Writing <tt>@PolyAll</tt> is equivalent to writing a polymorphic
 * qualifier for every type system.
 * <p>
 *
 * The <tt>@PolyAll</tt> annotation applies to every type qualifier hierarchy for
 * which no explicit qualifier is written.  For example, a declaration like
 * <tt>@PolyAll @NonNull String s</tt> is polymorphic over every type system
 * <em>except</em> the nullness type system, for which the type is fixed at
 * <tt>@NonNull</tt>.
 * <p>
 *
 * <!-- TODO: uncomment when this is implemented
 * The optional argument creates conceptually distinct polymorphic
 * qualifiers, such as <tt>@PolyAll(1)</tt> and <tt>@PolyAll(2)</tt>.
 * These two qualifierrs can vary independently.  When a method has
 * multiple occurrences of a single polymorphic qualifier, all of the
 * occurrences with the same argument (or with no argument) vary together.
 * <p>
 * -->
 *
 * <tt>@PolyAll</tt> only works for a given type system if that type system
 * already has its own polymorphic qualifier, such as
 * {@code @PolyNull} or {@code @PolyRegex}.
 * Therefore, every type system should define a polymorphic qualifier.
 * Then, to support <tt>@PolyAll</tt> in a type system, simply add it to the
 * list of supported type qualifiers.
 *
 * @see org.checkerframework.checker.nullness.qual.PolyNull
 * @see org.checkerframework.checker.interning.qual.PolyInterned
 * @see PolymorphicQualifier
 * @see QualifierPolymorphism
 *
 * @checker_framework.manual #polyall The @PolyAll qualifier applies to every type system
 */
@Documented
@Target({ElementType.TYPE_USE, ElementType.TYPE_PARAMETER})
@Retention(RetentionPolicy.RUNTIME)
@PolymorphicQualifier
public @interface PolyAll {
    // TODO: support multiple variables using an id, then uncomment some Javadoc
    //int value() default 0;
}
