/*
 * Decompiled with CFR 0.152.
 */
package org.citrusframework.spi;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Paths;
import org.citrusframework.exceptions.CitrusRuntimeException;
import org.citrusframework.spi.Resource;
import org.citrusframework.util.ReflectionHelper;

public class Resources {
    public static final String CLASSPATH_RESOURCE_PREFIX = "classpath:";
    public static final String FILESYSTEM_RESOURCE_PREFIX = "file:";
    public static final String JAR_RESOURCE_PREFIX = "jar:";
    public static final String HTTP_RESOURCE_PREFIX = "http:";

    public static Resource create(String filePath) {
        if (filePath.startsWith(CLASSPATH_RESOURCE_PREFIX)) {
            return Resources.fromClasspath(filePath);
        }
        if (filePath.startsWith(FILESYSTEM_RESOURCE_PREFIX)) {
            return Resources.fromFileSystem(filePath);
        }
        if (filePath.startsWith(HTTP_RESOURCE_PREFIX) || filePath.startsWith(JAR_RESOURCE_PREFIX)) {
            try {
                return Resources.create(new URL(filePath));
            }
            catch (MalformedURLException e) {
                throw new CitrusRuntimeException(e);
            }
        }
        Resource file = Resources.fromFileSystem(filePath);
        if (file.exists()) {
            return file;
        }
        return Resources.fromClasspath(filePath);
    }

    public static Resource create(String filePath, Class<?> contextClass) {
        return Resources.fromClasspath(filePath, contextClass);
    }

    public static Resource create(byte[] content) {
        return new ByteArrayResource(content);
    }

    public static Resource create(File file) {
        return new FileSystemResource(file);
    }

    public static Resource create(URL url) {
        return new UrlResource(url);
    }

    public static Resource fromClasspath(String filePath) {
        return new ClasspathResource(filePath);
    }

    public static Resource fromClasspath(String filePath, Class<?> contextClass) {
        return Resources.fromClasspath(contextClass.getPackageName().replace(".", "/") + "/" + filePath);
    }

    public static Resource fromFileSystem(String filePath) {
        return new FileSystemResource(filePath);
    }

    private static String getRawPath(String filePath) {
        if (filePath.startsWith(CLASSPATH_RESOURCE_PREFIX)) {
            return filePath.substring(CLASSPATH_RESOURCE_PREFIX.length());
        }
        if (filePath.startsWith(FILESYSTEM_RESOURCE_PREFIX)) {
            return filePath.substring(FILESYSTEM_RESOURCE_PREFIX.length());
        }
        return filePath;
    }

    public static class ByteArrayResource
    implements Resource {
        private final byte[] content;

        public ByteArrayResource(byte[] content) {
            this.content = content;
        }

        @Override
        public String getLocation() {
            return "";
        }

        @Override
        public boolean exists() {
            return true;
        }

        @Override
        public InputStream getInputStream() {
            return new ByteArrayInputStream(this.content);
        }

        @Override
        public File getFile() {
            throw new UnsupportedOperationException("ByteArrayResource does not provide access to a file");
        }
    }

    public static class FileSystemResource
    implements Resource {
        private final File file;

        public FileSystemResource(String path) {
            this.file = new File(Resources.getRawPath(path));
        }

        public FileSystemResource(File file) {
            this.file = file;
        }

        @Override
        public String getLocation() {
            return this.file.getPath();
        }

        @Override
        public URI getURI() {
            return this.file.toURI();
        }

        @Override
        public boolean exists() {
            return this.file.exists();
        }

        @Override
        public InputStream getInputStream() {
            if (!this.exists()) {
                throw new CitrusRuntimeException(this.file + " does not exists");
            }
            if (this.file.isDirectory()) {
                throw new UnsupportedOperationException(this.file + " is a directory");
            }
            try {
                return new FileInputStream(this.file);
            }
            catch (FileNotFoundException e) {
                throw new CitrusRuntimeException(this.file + " does not exists", e);
            }
        }

        @Override
        public File getFile() {
            return this.file;
        }
    }

    public static class UrlResource
    implements Resource {
        private final URL url;

        public UrlResource(URL url) {
            this.url = url;
        }

        @Override
        public String getLocation() {
            return this.url.toString();
        }

        @Override
        public boolean exists() {
            URLConnection connection = null;
            try {
                connection = this.url.openConnection();
                if (connection instanceof HttpURLConnection) {
                    boolean bl = ((HttpURLConnection)connection).getResponseCode() == 200;
                    return bl;
                }
                boolean bl = connection.getContentLengthLong() > 0L;
                return bl;
            }
            catch (IOException e) {
                throw new CitrusRuntimeException(e);
            }
            finally {
                if (connection instanceof HttpURLConnection) {
                    ((HttpURLConnection)connection).disconnect();
                }
            }
        }

        @Override
        public InputStream getInputStream() {
            URLConnection connection = null;
            try {
                connection = this.url.openConnection();
                connection.setUseCaches(false);
                InputStream inputStream = connection.getInputStream();
                return inputStream;
            }
            catch (IOException e) {
                throw new CitrusRuntimeException(e);
            }
            finally {
                if (connection instanceof HttpURLConnection) {
                    ((HttpURLConnection)connection).disconnect();
                }
            }
        }

        @Override
        public File getFile() {
            if (!"file".equals(this.url.getProtocol())) {
                throw new CitrusRuntimeException("Failed to resolve to absolute file path because it does not reside in the file system: " + this.url);
            }
            try {
                return new File(this.url.toURI().getSchemeSpecificPart());
            }
            catch (URISyntaxException ex) {
                return new File(this.url.getFile());
            }
        }
    }

    public static class ClasspathResource
    implements Resource {
        private final String location;

        public ClasspathResource(String location) {
            String raw = Resources.getRawPath(location);
            this.location = raw.startsWith("/") ? raw.substring(1) : raw;
        }

        @Override
        public String getLocation() {
            return this.location;
        }

        @Override
        public boolean exists() {
            return this.getURI() != null;
        }

        @Override
        public InputStream getInputStream() {
            return ReflectionHelper.class.getClassLoader().getResourceAsStream(this.location.replace("\\", "/"));
        }

        @Override
        public File getFile() {
            if (!this.exists()) {
                throw new CitrusRuntimeException(String.format("Failed to load classpath resource %s - does not exist", this.getLocation()));
            }
            return Paths.get(this.getURI()).toFile();
        }

        @Override
        public URI getURI() {
            URL url = ReflectionHelper.class.getClassLoader().getResource(this.location);
            try {
                return url != null ? url.toURI() : null;
            }
            catch (URISyntaxException e) {
                throw new CitrusRuntimeException("Failed to load classpath resource", e);
            }
        }
    }
}

