/*
 * Decompiled with CFR 0.152.
 */
package org.citrusframework.jbang;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.citrusframework.exceptions.CitrusRuntimeException;
import org.citrusframework.jbang.JBangSettings;
import org.citrusframework.jbang.ProcessAndOutput;
import org.citrusframework.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JBangSupport {
    private static final Logger LOG = LoggerFactory.getLogger(JBangSupport.class);
    private static final boolean IS_OS_WINDOWS = System.getProperty("os.name").toLowerCase(Locale.ENGLISH).contains("windows");
    public static final int OK_EXIT_CODE = 0;
    private static Path installDir;
    private static final AtomicBoolean initialized;
    private static final Set<String> trustUrls;
    private final Map<String, String> systemProperties = new HashMap<String, String>();
    private final Map<String, String> envVars = new HashMap<String, String>();
    private String app;
    private Path workingDir;

    private JBangSupport() {
    }

    public static JBangSupport jbang() {
        if (!initialized.getAndSet(true)) {
            JBangSupport.detectJBang();
            Arrays.stream(JBangSettings.getTrustUrls()).forEach(JBangSupport::addTrust);
        }
        return new JBangSupport();
    }

    public String version() {
        ProcessAndOutput p = JBangSupport.execute(JBangSupport.jBang("version"), null, null);
        return p.getOutput();
    }

    public JBangSupport trust(String url) {
        JBangSupport.addTrust(url);
        return this;
    }

    public JBangSupport withSystemProperty(String name, String value) {
        this.systemProperties.put(name, value);
        return this;
    }

    public JBangSupport withSystemProperties(Map<String, String> systemProperties) {
        this.systemProperties.putAll(systemProperties);
        return this;
    }

    public JBangSupport withEnv(String name, String value) {
        this.systemProperties.put(name, value);
        return this;
    }

    public JBangSupport withEnvs(Map<String, String> envVars) {
        this.envVars.putAll(envVars);
        return this;
    }

    public JBangSupport app(String name) {
        this.app = name;
        return this;
    }

    public JBangSupport workingDir(Path workingDir) {
        this.workingDir = workingDir;
        return this;
    }

    public ProcessAndOutput run(String command, String ... args) {
        return this.run(command, Arrays.asList(args));
    }

    public ProcessAndOutput run(String command, List<String> args) {
        return JBangSupport.execute(JBangSupport.jBang(this.systemProperties, this.constructAllArgs(command, args)), this.workingDir, this.envVars);
    }

    public ProcessAndOutput runAsync(String command, String ... args) {
        return this.runAsync(command, Arrays.asList(args));
    }

    public ProcessAndOutput runAsync(String command, List<String> args) {
        return JBangSupport.executeAsync(JBangSupport.jBang(this.systemProperties, this.constructAllArgs(command, args)), this.workingDir, this.envVars);
    }

    public ProcessAndOutput runAsync(String command, File output, String ... args) {
        return this.runAsync(command, output, Arrays.asList(args));
    }

    public ProcessAndOutput runAsync(String command, File output, List<String> args) {
        return JBangSupport.executeAsync(JBangSupport.jBang(this.systemProperties, this.constructAllArgs(command, args)), this.workingDir, output, this.envVars);
    }

    private List<String> constructAllArgs(String command, List<String> args) {
        ArrayList<String> allArgs = new ArrayList<String>();
        if (this.app != null) {
            allArgs.add(this.app);
        }
        allArgs.add(command);
        allArgs.addAll(args);
        return allArgs;
    }

    private static void detectJBang() {
        ProcessAndOutput result = JBangSupport.getVersion();
        if (result.getProcess().exitValue() == 0) {
            LOG.info("Found JBang v" + result.getOutput());
        } else if (JBangSettings.isAutoDownload()) {
            LOG.warn("JBang not found. Downloading ...");
            JBangSupport.download();
            result = JBangSupport.getVersion();
            if (result.getProcess().exitValue() == 0) {
                LOG.info("Using JBang v" + result.getOutput());
            }
        } else {
            throw new CitrusRuntimeException("Missing JBang installation on host - make sure to install JBang");
        }
    }

    private static void download() {
        String homePath = "jbang";
        Path installPath = Paths.get(System.getProperty("user.home"), new String[0]).toAbsolutePath().resolve(".jbang").toAbsolutePath();
        if (installPath.resolve(homePath).toFile().exists()) {
            LOG.info("Using local JBang in " + installPath);
            installDir = installPath.resolve(homePath);
            return;
        }
        LOG.info("Downloading JBang from " + JBangSettings.getJBangDownloadUrl() + " and installing in " + installPath);
        try {
            Files.createDirectories(installPath, new FileAttribute[0]);
            HttpRequest request = HttpRequest.newBuilder().uri(new URI(JBangSettings.getJBangDownloadUrl())).GET().build();
            HttpClient client = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.ALWAYS).build();
            HttpResponse<Path> response = client.send(request, HttpResponse.BodyHandlers.ofFileDownload(installPath, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));
            if (response.statusCode() != 200) {
                throw new CitrusRuntimeException(String.format("Failed to download JBang - response code %d", response.statusCode()));
            }
            JBangSupport.unzip(response.body(), installPath);
        }
        catch (IOException | InterruptedException | URISyntaxException e) {
            throw new CitrusRuntimeException("Failed to download JBang", (Throwable)e);
        }
        installDir = installPath.resolve(homePath);
    }

    private static ProcessAndOutput getVersion() {
        return JBangSupport.execute(JBangSupport.jBang("version"), null, null);
    }

    private static void addTrust(String url) {
        ProcessAndOutput result;
        int exitValue;
        if (trustUrls.add(url) && (exitValue = (result = JBangSupport.execute(JBangSupport.jBang("trust", "add", url), null, null)).getProcess().exitValue()) != 0 && exitValue != 1) {
            throw new CitrusRuntimeException("Error while trusting JBang URLs. Exit code: " + exitValue);
        }
    }

    private static List<String> jBang(String ... args) {
        return JBangSupport.jBang(List.of(args));
    }

    private static List<String> jBang(List<String> args) {
        return JBangSupport.jBang(Collections.emptyMap(), args);
    }

    private static List<String> jBang(Map<String, String> systemProperties, List<String> args) {
        ArrayList<String> command = new ArrayList<String>();
        if (IS_OS_WINDOWS) {
            command.add("cmd.exe");
            command.add("/c");
        } else {
            command.add("sh");
            command.add("-c");
        }
        String jBangCommand = JBangSupport.getJBangExecutable() + " " + JBangSupport.getSystemPropertyArgs(systemProperties) + String.join((CharSequence)" ", args);
        command.add(jBangCommand);
        return command;
    }

    private static String getSystemPropertyArgs(Map<String, String> systemProperties) {
        if (systemProperties.isEmpty()) {
            return "";
        }
        return systemProperties.entrySet().stream().map(entry -> "-D%s=\"%s\"".formatted(entry.getKey(), entry.getValue())).collect(Collectors.joining(" ")) + " ";
    }

    private static ProcessAndOutput execute(List<String> command, Path workingDir, Map<String, String> envVars) {
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Executing JBang command: %s".formatted(String.join((CharSequence)" ", command)));
            }
            ProcessBuilder pBuilder = new ProcessBuilder(command).redirectErrorStream(true);
            if (envVars != null) {
                pBuilder.environment().putAll(envVars);
            }
            if (workingDir != null) {
                pBuilder.directory(workingDir.toFile());
            }
            Process p = pBuilder.start();
            String output = FileUtils.readToString((InputStream)p.getInputStream(), (Charset)StandardCharsets.UTF_8);
            p.waitFor();
            if (JBangSettings.isDumpProcessOutput()) {
                Path workDir = JBangSettings.getWorkDir();
                FileUtils.writeToFile((String)output, (File)workDir.resolve(String.format("%s-output.txt", p.pid())).toFile());
            }
            if (LOG.isDebugEnabled() && p.exitValue() != 0) {
                LOG.debug("Command failed: " + String.join((CharSequence)" ", command));
                LOG.debug(output);
            }
            return new ProcessAndOutput(p, output);
        }
        catch (IOException | InterruptedException e) {
            throw new CitrusRuntimeException("Error while executing JBang", (Throwable)e);
        }
    }

    private static ProcessAndOutput executeAsync(List<String> command, Path workingDir, Map<String, String> envVars) {
        try {
            ProcessBuilder pBuilder = new ProcessBuilder(command).redirectErrorStream(true);
            if (envVars != null) {
                pBuilder.environment().putAll(envVars);
            }
            if (workingDir != null) {
                pBuilder.directory(workingDir.toFile());
            }
            Process p = pBuilder.start();
            return new ProcessAndOutput(p);
        }
        catch (IOException e) {
            throw new CitrusRuntimeException("Error while executing JBang", (Throwable)e);
        }
    }

    private static ProcessAndOutput executeAsync(List<String> command, Path workingDir, File outputFile, Map<String, String> envVars) {
        try {
            if (!outputFile.getParentFile().exists() && !outputFile.getParentFile().mkdirs()) {
                throw new CitrusRuntimeException("Unable to create process output directory: " + outputFile.getParent());
            }
            ProcessBuilder pBuilder = new ProcessBuilder(command).redirectErrorStream(true).redirectOutput(outputFile);
            if (envVars != null) {
                pBuilder.environment().putAll(envVars);
            }
            if (workingDir != null) {
                pBuilder.directory(workingDir.toFile());
            }
            Process p = pBuilder.start();
            return new ProcessAndOutput(p, outputFile);
        }
        catch (IOException e) {
            throw new CitrusRuntimeException("Error while executing JBang", (Throwable)e);
        }
    }

    private static String getJBangExecutable() {
        if (installDir != null) {
            if (IS_OS_WINDOWS) {
                return installDir.resolve("bin/jbang.cmd").toString();
            }
            return installDir.resolve("bin/jbang").toString();
        }
        if (IS_OS_WINDOWS) {
            return "jbang.cmd";
        }
        return "jbang";
    }

    private static void unzip(Path downloadZip, Path installPath) throws IOException {
        ZipInputStream zis = new ZipInputStream(new FileInputStream(downloadZip.toFile()));
        ZipEntry zipEntry = zis.getNextEntry();
        while (zipEntry != null) {
            Path filePath = JBangSupport.newFile(installPath, zipEntry);
            File newFile = filePath.toFile();
            if (zipEntry.isDirectory()) {
                if (!newFile.isDirectory() && !newFile.mkdirs()) {
                    throw new IOException("Failed to create directory " + newFile);
                }
            } else {
                File parent = newFile.getParentFile();
                if (!parent.isDirectory() && !parent.mkdirs()) {
                    throw new IOException("Failed to create directory " + parent);
                }
                Files.copy(zis, filePath, StandardCopyOption.REPLACE_EXISTING);
                if ("jbang".equals(filePath.getFileName().toString())) {
                    Files.setPosixFilePermissions(filePath, PosixFilePermissions.fromString("rwxr--r--"));
                }
            }
            zipEntry = zis.getNextEntry();
        }
        zis.closeEntry();
        zis.close();
    }

    private static Path newFile(Path destinationDir, ZipEntry zipEntry) throws IOException {
        Path destFile = destinationDir.resolve(zipEntry.getName());
        String destDirPath = destinationDir.toFile().getCanonicalPath();
        String destFilePath = destFile.toFile().getCanonicalPath();
        if (!destFilePath.startsWith(destDirPath + File.separator)) {
            throw new IOException("Entry is outside of the target dir: " + zipEntry.getName());
        }
        return destFile;
    }

    static {
        initialized = new AtomicBoolean(false);
        trustUrls = new HashSet<String>();
    }
}

