/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */

package org.sonar.commons;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.URL;


public class ServerHttpClient {

  private String url;

  protected static final String SERVER_API_PATH = "/api/server";
  private static final String KEY_PATH = SERVER_API_PATH + "/key";
  private static final String VERSION_PATH = SERVER_API_PATH + "/version";
  protected static final String BATCH_PATH = SERVER_API_PATH + "/consolidate_snapshot";

  protected static final String MAVEN_PATH = "/deploy/maven";

  private static final int CONNECT_TIMEOUT_MILLISECONDS = 30000;
  private static final int READ_TIMEOUT_MILLISECONDS = 60000;


  public ServerHttpClient(String remoteServerUrl) {
    this.url = StringUtils.chomp(remoteServerUrl, "/");
  }

  public ServerHttpClient(Configuration configuration) {
    this(configuration.getString("sonar.host.url", "http://localhost:9000"));
  }

  /**
   * Throws a runtime ServerConnectionException if it fails to connect Sonar server
   */
  public void checkUp() {
    String exceptionLabel = "Sonar server at " + url +
        " is unreacheable. Either start it or setup the sonar.host.url maven setting if the URL is incorrect";
    try {
      if (getId() == null) {
        throw new ServerConnectionException(exceptionLabel);
      }

    } catch (ServerConnectionException e) {
      throw e;

    } catch (IOException e) {
      throw new ServerConnectionException(exceptionLabel, e);
    }
  }

  public String getId() throws IOException {
    return executeAction(KEY_PATH);
  }

  public String getVersion() throws IOException {
    return executeAction(VERSION_PATH);
  }

  public void launchBatch(Integer snapshotId) {
    String batchUrl = url + BATCH_PATH + "/" + snapshotId;
    try {
      postRemoteContent(batchUrl);
    } catch (IOException e) {
      throw new RuntimeException("Can't access to url : " + batchUrl);
    }
  }

  public String getMavenRepositoryUrl() {
    return this.url + MAVEN_PATH;
  }


  protected String executeAction(String action) throws ServerApiEmptyContentException, IOException {
    String result = getRemoteContent(url + action);
    if (result.trim().length() == 0) {
      throw new ServerApiEmptyContentException("Empty " + action + " returned from server");
    }
    return result;
  }

  protected String getRemoteContent(String url) throws IOException {
    HttpURLConnection conn = getConnection(url, "GET");
    Reader reader = new InputStreamReader((InputStream) conn.getContent());
    try {
      int statusCode = conn.getResponseCode();
      if (statusCode != HttpURLConnection.HTTP_OK) {
        throw new IOException("Status returned by url : '" + url + "' is invalid : " + statusCode);
      }

      return IOUtils.toString(reader);
    } finally {
      IOUtils.closeQuietly(reader);
      conn.disconnect();
    }
  }

  protected void postRemoteContent(String url) throws IOException {
    HttpURLConnection conn = getConnection(url, "POST");
    try {

      int statusCode = conn.getResponseCode();
      if (statusCode != HttpURLConnection.HTTP_OK) {
        throw new ServerApiLaunchBatchFailureException("Fail to launch batch on : " + url);
      }
    } finally {
      conn.disconnect();
    }
  }


  protected String getUrl() {
    return url;
  }

  private HttpURLConnection getConnection(String url, String method) throws IOException {
    URL page = new URL(url);
    HttpURLConnection conn = (HttpURLConnection) page.openConnection();
    conn.setConnectTimeout(CONNECT_TIMEOUT_MILLISECONDS);
    conn.setReadTimeout(READ_TIMEOUT_MILLISECONDS);
    conn.setRequestMethod(method);
    conn.connect();
    return conn;
  }

}
