/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */

package org.sonar.commons;

import org.apache.commons.lang.StringUtils;

import java.util.regex.Pattern;

public class WildcardPattern {

  private Pattern pattern;

  public WildcardPattern(String pattern) {
    this(pattern, "/");
  }

  public WildcardPattern(String pattern, String directorySeparator) {
    this.pattern = Pattern.compile(toRegexp(pattern, directorySeparator));
  }

  public boolean match(String value) {
    return pattern.matcher(removeSlahesToIgnore(value)).matches();
  }

  private String toRegexp(String wildcardPattern, String directorySeparator) {
    String pattern = removeSlahesToIgnore(wildcardPattern);
    pattern = StringUtils.replace(pattern, "**/**", "**");
    pattern = StringUtils.replace(pattern, "**/", "|");
    pattern = StringUtils.replace(pattern, "/**", "|");
    pattern = StringUtils.replace(pattern, "**", "|");
    StringBuilder sb = new StringBuilder();

    for (char c : pattern.toCharArray()) {
      switch (c) {
        case '|':
          sb.append(".*");
          break;
        case '*':
          sb.append("[^\\").append(directorySeparator).append("]*");
          break;
        case '?':
          sb.append("[^\\").append(directorySeparator).append("]");
          break;
        case '.':
          sb.append("\\.");
          break;
        case '/':
          sb.append("\\").append(directorySeparator);
          break;
        default:
          sb.append(c);
      }
    }

    return sb.toString();
  }

  private String removeSlahesToIgnore(String wildcardPattern) {
    String pattern = StringUtils.removeStart(wildcardPattern, "/");
    pattern = StringUtils.removeEnd(pattern, "/");
    return pattern;
  }


}
