/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */

package org.sonar.commons.alerts;

import org.sonar.commons.Metric;
import org.sonar.commons.Metric.ValueType;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.lang.StringUtils;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.sonar.commons.resources.Measure;
import org.sonar.commons.rules.RulesProfile;

import javax.persistence.*;

@Entity
@Table(name = "alerts")
public class Alert implements Cloneable {

  public static String OPERATOR_GREATER = ">";
  public static String OPERATOR_SMALLER = "<";
  public static String OPERATOR_EQUALS = "=";
  public static String OPERATOR_NOT_EQUALS = "!=";

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "ALERTS_SEQ", sequenceName = "ALERTS_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "ALERTS_SEQ")
  private Integer id;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "profile_id")
  @Cache(usage = CacheConcurrencyStrategy.READ_ONLY)
  private RulesProfile rulesProfile;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "metric_id")
  @Cache(usage = CacheConcurrencyStrategy.READ_ONLY)
  private Metric metric;

  @Column(name = "operator", updatable = false, nullable = true, length = 3)
  private String operator;

  @Column(name = "value_error", updatable = false, nullable = true, length = 64)
  private String valueError;

  @Column(name = "value_warning", updatable = false, nullable = true, length = 64)
  private String valueWarning;

  public Alert() {
  }

  public Alert(RulesProfile rulesProfile, Metric metric, String operator, String valueError, String valueWarning) {
    super();
    this.rulesProfile = rulesProfile;
    this.metric = metric;
    this.operator = operator;
    this.valueError = valueError;
    this.valueWarning = valueWarning;
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public RulesProfile getRulesProfile() {
    return rulesProfile;
  }

  public void setRulesProfile(RulesProfile rulesProfile) {
    this.rulesProfile = rulesProfile;
  }

  public Metric getMetric() {
    return metric;
  }

  public void setMetric(Metric metric) {
    this.metric = metric;
  }

  public String getOperator() {
    return operator;
  }

  public void setOperator(String operator) {
    this.operator = operator;
  }

  public String getValueError() {
    return valueError;
  }

  public void setValueError(String valueError) {
    this.valueError = valueError;
  }

  public String getValueWarning() {
    return valueWarning;
  }

  public void setValueWarning(String valueWarning) {
    this.valueWarning = valueWarning;
  }

  public boolean isGreaterOperator() {
    return operator.equals(OPERATOR_GREATER);
  }

  public boolean isSmallerOperator() {
    return operator.equals(OPERATOR_SMALLER);
  }

  public boolean isEqualsOperator() {
    return operator.equals(OPERATOR_EQUALS);
  }

  public boolean isNotEqualsOperator() {
    return operator.equals(OPERATOR_NOT_EQUALS);
  }

  /**
   * Get the matching alert level for the given measure
   */
  public Metric.Level getAlertLevel(Measure measure) {
    if ( evaluateAlert(measure, Metric.Level.ERROR) ) {
      return Metric.Level.ERROR;
    }
    if ( evaluateAlert(measure, Metric.Level.WARN) ) {
      return Metric.Level.WARN;
    }
    return Metric.Level.OK;
  }

  private boolean evaluateAlert(Measure measure, Metric.Level alertLevel) {
    String valueToEval;
    if (alertLevel.equals(Metric.Level.ERROR)) {
      valueToEval = getValueError();

    } else if (alertLevel.equals(Metric.Level.WARN)) {
      valueToEval = getValueWarning();
    } else {
      throw new IllegalStateException(alertLevel.toString());
    }
    if (StringUtils.isEmpty(valueToEval)) return false;

    Comparable criteriaValue = getValueForComparison(getMetric(), valueToEval);
    Comparable metricValue = getMeasureValue(getMetric(), measure);

    int comparison = metricValue.compareTo(criteriaValue);
    if (isNotEqualsOperator() && comparison == 0 ||
      isGreaterOperator() && comparison != 1 ||
      isSmallerOperator() && comparison != -1 ||
      isEqualsOperator() && comparison != 0) {
      return false;
    }
    return true;
  }

  public String getAlertLabel(Metric.Level level) {
    return new StringBuilder()
      .append(getMetric().getName())
      .append(" ").append(getOperator())
      .append(" ")
      .append(level.equals(Metric.Level.ERROR) ? getValueError() : getValueWarning()).toString();
  }

  private Comparable<?> getValueForComparison(Metric metric, String value) {
    if (metric.getType() == ValueType.FLOAT ||
      metric.getType() == ValueType.PERCENT) {
      return Double.parseDouble(value);
    } else if (metric.getType() == ValueType.INT ||
      metric.getType() == ValueType.MILLISEC) {
      return value.contains(".") ? Integer.parseInt(value.substring(0, value.indexOf('.'))) : Integer.parseInt(value);
    } else if (metric.getType() == ValueType.STRING ||
      metric.getType() == ValueType.LEVEL) {
      return value;
    } else if (metric.getType() == ValueType.BOOL) {
      return Boolean.valueOf(value);
    }
    throw new NotImplementedException(metric.getType().toString());
  }

  private Comparable<?> getMeasureValue(Metric metric, Measure measure) {
    if (metric.getType() == ValueType.FLOAT ||
      metric.getType() == ValueType.PERCENT) {
      return measure.getValue();
    } else if (metric.getType() == ValueType.INT ||
      metric.getType() == ValueType.MILLISEC) {
      return measure.getValue().intValue();
    } else if (metric.getType() == ValueType.STRING ||
      metric.getType() == ValueType.LEVEL) {
      return measure.getTextValue();
    } else if (metric.getType() == ValueType.BOOL) {
      return measure.getValue() == 0d ? Boolean.FALSE : Boolean.TRUE;
    }
    throw new NotImplementedException(metric.getType().toString());
  }

  @Override
  public Object clone() {
    return new Alert(getRulesProfile(), getMetric(), getOperator(), getValueError(), getValueWarning());
  }

}
