/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.database;

import org.apache.commons.configuration.Configuration;
import org.hibernate.dialect.DB2Dialect;
import org.hibernate.dialect.HSQLDialect;
import org.hibernate.dialect.PostgreSQLDialect;
import org.hibernate.ejb.EntityManagerFactoryImpl;
import org.hibernate.ejb.HibernateEntityManagerFactory;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.Statistics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.commons.database.dialect.DerbyWithDecimalDialect;
import org.sonar.commons.database.dialect.MsSqlDialect;
import org.sonar.commons.database.dialect.MySqlWithDecimalDialect;
import org.sonar.commons.database.dialect.Oracle10gWithDecimalDialect;

import javax.persistence.PersistenceException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

public abstract class AbstractDatabaseConnector implements DatabaseConnector {
  protected static final Logger LOG_SQL = LoggerFactory.getLogger("org.hibernate.SQL");
  protected static final Logger LOG_STATISTICS = LoggerFactory.getLogger("org.sonar.DBSTATISTICS");

  private Configuration configuration = null;
  private int databaseVersion = -1;

  protected AbstractDatabaseConnector(Configuration configuration) {
    this.configuration = configuration;
  }

  protected AbstractDatabaseConnector() {
  }

  public Configuration getConfiguration() {
    return configuration;
  }

  public void setConfiguration(Configuration configuration) {
    this.configuration = configuration;
  }

  public void dumpStatistics() {
    if (LOG_STATISTICS.isInfoEnabled() &&
        getEntityManagerFactory() instanceof EntityManagerFactoryImpl) {

      HibernateEntityManagerFactory hibernateFactory = (HibernateEntityManagerFactory) getEntityManagerFactory();
      Statistics stats = hibernateFactory.getSessionFactory().getStatistics();
      for (String query : stats.getQueries()) {
        QueryStatistics stat = stats.getQueryStatistics(query);
        StringBuilder sb = new StringBuilder();
        sb.append(stats.getQueryExecutionMaxTime());
        sb.append("ms, count=");
        sb.append(stat.getExecutionCount());
        sb.append(", sql=");
        sb.append(stat.getCategoryName());
        LOG_STATISTICS.info(sb.toString());
      }
      stats.clear();
    }
  }

  private int loadVersion() {
    Connection connection = null;
    Statement stmt = null;
    ResultSet rs = null;
    ResultSet tablesRs = null;
    try {
      connection = getConnection();
      tablesRs= connection.getMetaData().getTables(null, null, "%", null);
      boolean tableExists=false;
      while(tablesRs.next()) {
        // the 3rd column is the table name :
        // http://java.sun.com/j2se/1.4.2/docs/api/java/sql/DatabaseMetaData.html#getTables(java.lang.String,%20java.lang.String,%20java.lang.String,%20java.lang.String[])
        if (SchemaInfo.TABLE_NAME.equalsIgnoreCase(tablesRs.getString(3))) {
          tableExists=true;
          break;
        }
      }
      if (tableExists) {
        stmt = connection.createStatement();
        try {
          rs = stmt.executeQuery("SELECT version FROM " + SchemaInfo.TABLE_NAME);
          if (rs.next()) {
            return rs.getInt(1);
          }
        } catch (SQLException e) {
          // can occur with a table not found if multiple sonar schemas 
          // are installed into the db, since the connection.getMetaData().getTables
          // look for tables in all schemas !
        }
      }

    } catch (SQLException e) {
      throw new PersistenceException("Can not check the database version", e);

    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (SQLException e) {

        }
      }
      if (tablesRs != null) {
        try {
          tablesRs.close();
        } catch (SQLException e) {

        }
      }
      if (stmt != null) {
        try {
          stmt.close();
        } catch (SQLException e) {
        }
      }
      try {
        if (connection != null) {
          connection.close();
        }
      } catch (SQLException e) {

      }
    }

    return -1;
  }

  protected boolean checkSchemaVersion() {
    if (databaseVersion == SchemaInfo.VERSION) {
      return true;
    }
    databaseVersion = loadVersion();
    return (databaseVersion == SchemaInfo.VERSION);
  }

  public void checkVersion() throws WrongDatabaseVersionException {
    if (!checkSchemaVersion()) {
      throw new WrongDatabaseVersionException(databaseVersion, SchemaInfo.VERSION);
    }
  }

  public String getDialect() {
    String dialect = configuration.getString("sonar.jdbc.dialect");
    if (dialect==null) {
      Connection connection = null;
      try {
        connection = getConnection();
        dialect = getDialectFromJdbcUrl(connection.getMetaData().getURL());

      } catch (SQLException e) {
        throw new PersistenceException("can not autodetect the dialect", e);

      } finally {
        if (connection !=null) {
          try {
            connection.close();
          } catch (SQLException e) {

          }
        }
      }
    }
    return dialect;
  }

  public String getDialectClass(String dialect) {
    String dialectClass = configuration.getString(DatabaseProperties.PROP_DIALECT_CLASS);
    if (dialectClass==null) {
      dialectClass = getHibernateDialectClassName(dialect);
    }
    return dialectClass;
  }

  private String getDialectFromJdbcUrl(String url) {
    if (url.toLowerCase().startsWith("jdbc:db2:")) {
      return DatabaseProperties.DIALECT_DB2;
    }
    if (url.toLowerCase().startsWith("jdbc:derby:")) {
      return DatabaseProperties.DIALECT_DERBY;
    }
    if (url.toLowerCase().startsWith("jdbc:hsqldb:")) {
      return DatabaseProperties.DIALECT_HSQLDB;
    }
    if (url.toLowerCase().startsWith("jdbc:microsoft:sqlserver:") ||
        url.toLowerCase().startsWith("jdbc:jtds:sqlserver:")) {
      return DatabaseProperties.DIALECT_MSSQL;
    }
    if (url.toLowerCase().startsWith("jdbc:mysql:")) {
      return DatabaseProperties.DIALECT_MYSQL;
    }
    if (url.toLowerCase().startsWith("jdbc:oracle:")) {
      return DatabaseProperties.DIALECT_ORACLE;
    }
    if (url.toLowerCase().startsWith("jdbc:postgresql:")) {
      return DatabaseProperties.DIALECT_POSTGRE;
    }
    return null;
  }

  private String getHibernateDialectClassName(String dialect) {
    if (DatabaseProperties.DIALECT_DB2.equals(dialect)) {
      return DB2Dialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_DERBY.equals(dialect)) {
      return DerbyWithDecimalDialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_HSQLDB.equals(dialect)) {
      return HSQLDialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_MSSQL.equals(dialect)) {
      return MsSqlDialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_MYSQL.equals(dialect)) {
      return MySqlWithDecimalDialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_ORACLE.equals(dialect)) {
      return Oracle10gWithDecimalDialect.class.getName();
    }
    if (DatabaseProperties.DIALECT_POSTGRE.equals(dialect)) {
      return PostgreSQLDialect.class.getName();
    }
    return null;
  }
}
