/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.database;

import org.apache.commons.configuration.Configuration;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;

public class DriverDatabaseConnector extends AbstractDatabaseConnector {
  private EntityManagerFactory factory = null;
  private boolean operational = false;


  public DriverDatabaseConnector(Configuration configuration) {
    super(configuration);
  }

  public boolean isOperational() {
    return operational;
  }

  public void start() {
    if (!operational) {
      checkVersion();
      getEntityManagerFactory();
      operational = true;
    }
  }


  public void stop() {
    if (factory != null && factory.isOpen()) {
      factory.close();
      factory = null;
    }
    operational = false;
  }

  public String getDriver() {
    return getConfiguration().getString(DatabaseProperties.PROP_DRIVER);
  }

  public String getUrl() {
    return getConfiguration().getString(DatabaseProperties.PROP_URL);
  }

  public String getUsername() {
    return getConfiguration().getString(DatabaseProperties.PROP_USER);
  }

  public String getPassword() {
    return getConfiguration().getString(DatabaseProperties.PROP_PASSWORD);
  }

  public int getIsolationLevel() {
    return getConfiguration().getInt(DatabaseProperties.PROP_ISOLATION, Connection.TRANSACTION_READ_UNCOMMITTED);
  }

  public Connection getConnection() throws SQLException {
    try {
      Class.forName(getDriver());

    } catch (ClassNotFoundException e) {
      throw new SQLException("SQL driver not found " + getDriver());
    }
    return DriverManager.getConnection(getUrl(), getUsername(), getPassword());
  }

  protected String hbm2ddl = "validate";

  public void setHbm2ddl(String s) {
    this.hbm2ddl = s;
  }

  public synchronized EntityManagerFactory getEntityManagerFactory() {
    if (factory != null) {
      return factory;
    }

    // other settings are stored into /META-INF/persistence.xml
    Properties props = new Properties();
    props.put("hibernate.connection.isolation", getIsolationLevel());
    props.put("hibernate.hbm2ddl.auto", hbm2ddl);
    props.put("hibernate.connection.url", getUrl());
    props.put("hibernate.connection.driver_class", getDriver());
    props.put("hibernate.connection.username", getUsername());
    props.put("hibernate.dialect", getDialectClass(getDialect()));
    if (getPassword()!=null) {
      props.put("hibernate.connection.password", getPassword());
    }

    if (LOG_STATISTICS.isInfoEnabled()) {
      props.put("hibernate.generate_statistics", "true");
    }
    if (LOG_SQL.isInfoEnabled()) {
      props.put("hibernate.show_sql", "true");
    }

    if (getConfiguration().getBoolean(DatabaseProperties.PROP_HIBERNATE_SECOND_LEVEL_CACHE, true)) {
      props.put("hibernate.cache.provider_class", "org.hibernate.cache.EhCacheProvider");
      props.put("hibernate.cache.use_second_level_cache", "true");
      props.put("hibernate.cache.use_query_cache", "false");

    } else {
      props.put("hibernate.cache.provider_class", "org.hibernate.cache.NoCacheProvider");
      props.put("hibernate.cache.use_second_level_cache", "false");
      props.put("hibernate.cache.use_query_cache", "false");
    }

    factory = Persistence.createEntityManagerFactory("sonar", props);
    return factory;
  }

  public EntityManager createEntityManager() {
    EntityManager manager = getEntityManagerFactory().createEntityManager();
    if (LOG_STATISTICS.isInfoEnabled()) {
      manager = new StatisticsEntityManager(manager, LOG_STATISTICS);
    }
    return manager;
  }
}