/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.events;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.sonar.commons.resources.Resource;
import org.sonar.commons.resources.Snapshot;

import javax.persistence.*;
import java.util.Date;

@Entity
@Table(name = "events")
public class Event {
  public static final String CATEGORY_VERSION = "Version";
  public static final String CATEGORY_ALERT = "Alert";

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "EVENTS_SEQ", sequenceName = "EVENTS_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "EVENTS_SEQ")
  private Integer id;

  @Column(name = "name", updatable = true, nullable = true, length = 50)
  private String name;

  @Column(name = "description", updatable = true, nullable = true, length = 3072)
  private String description;

  @Column(name = "category", updatable = true, nullable = true, length = 50)
  private String category;

  @Column(name = "event_date", updatable = true, nullable = false)
  private Date date;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "snapshot_id", updatable = true, nullable = true)
  private Snapshot snapshot;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "resource_id", updatable = false, nullable = true)
  private Resource resource;


  public Event() {
  }

  public Event(String name, String description, String category, Date date, Resource resource) {
    this.name = name;
    this.description = description;
    this.category = category;
    this.date = date;
    this.resource = resource;
  }

  public Event(String name, String description, String category, Snapshot snapshot) {
    this.name = name;
    this.description = description;
    this.category = category;
    setSnapshot(snapshot);
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public String getCategory() {
    return category;
  }

  public void setCategory(String category) {
    this.category = category;
  }

  public boolean isVersionCategory() {
    return CATEGORY_VERSION.equalsIgnoreCase(category);
  }

  public Date getDate() {
    return date;
  }

  public void setDate(Date date) {
    this.date = date;
  }

  public Snapshot getSnapshot() {
    return snapshot;
  }

  public void setSnapshot(Snapshot snapshot) {
    this.snapshot = snapshot;
    if (snapshot != null) {
      this.date = snapshot.getCreatedAt();
      this.resource = snapshot.getResource();
    }
  }

  public Resource getResource() {
    return resource;
  }

  public void setResource(Resource resource) {
    this.resource = resource;
  }

  public String toString() {
    return new ToStringBuilder(this)
        .append("name", name)
        .append("categ", category)
        .append("date", date)
        .append("snap", snapshot)
        .append("res", resource)
        .toString();
  }

  public boolean isLinkedToSnapshot() {
    return snapshot!=null;
  }
}
