/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.resources;

import org.sonar.commons.Metric;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.sonar.commons.rules.Rule;

import javax.persistence.*;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;

@Entity
@Table(name = "project_measures")
public class Measure {
  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "PROJECT_MEASURES_SEQ", sequenceName = "PROJECT_MEASURES_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "PROJECT_MEASURES_SEQ")
  private Integer id;

  @Column(name = "value", updatable = true, nullable = true, precision = 30, scale = 20)
  private Double value = 0.0;

  @Column(name = "text_value", updatable = true, nullable = true, length = 96)
  private String textValue;

  @Column(name = "tendency", updatable = true, nullable = true)
  private Integer tendency;

  @ManyToOne(fetch = FetchType.EAGER)
  @JoinColumn(name = "metric_id")
  @Cache(usage = CacheConcurrencyStrategy.READ_ONLY)
  private Metric metric;

  @Column(name = "snapshot_id", updatable = true, nullable = true)
  private Integer snapshotId;

  @Column(name = "project_id", updatable = true, nullable = true)
  private Integer projectId;

  @Temporal(TemporalType.TIMESTAMP)
  @Column(name = "measure_date", updatable = true, nullable = true)
  private Date measureDate;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "rule_id")
  @Cache(usage = CacheConcurrencyStrategy.READ_ONLY)
  private Rule rule;

  @Column(name = "rules_category_id")
  private Integer rulesCategoryId;

  @Column(name = "alert_status", updatable = true, nullable = true, length = 5)
  private String alertStatus;

  @Column(name = "alert_text", updatable = true, nullable = true, length = 4000)
  private String alertText;

  @OneToMany(mappedBy = "measure", fetch = FetchType.LAZY, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REMOVE})
  private List<MeasureData> measureData = new ArrayList<MeasureData>();

  public Measure(Metric metric, Double val) {
    if (val.isNaN() || val.isInfinite()) {
      throw new IllegalArgumentException("Measure value is NaN. Metric=" + metric);
    }
    this.metric = metric;
    this.value = val;
  }

  public Measure(Metric metric, Metric.Level level) {
    this.metric = metric;
    if (level != null) {
      this.textValue = level.toString();
    }
  }

  public Measure(Metric metric, String val) {
    this.metric = metric;
    if (metric.isDataType()) {
      measureData.add(new MeasureData(this, val));
    } else {
      this.textValue = val;
    }
  }

  public Measure(MeasureKey key, Double val) {
    if (val == Double.NaN) {
      throw new IllegalArgumentException("Measure value is NaN. Key=" + key);
    }
    rule = key.getRule();
    rulesCategoryId = key.getRulesCategoryId();
    this.metric = key.getMetric();

    this.value = val;
  }

  public Measure(MeasureKey key, Metric.Level level) {
    rule = key.getRule();
    rulesCategoryId = key.getRulesCategoryId();
    this.metric = key.getMetric();
    if (level != null) {
      this.textValue = level.toString();
    }
  }

  public Measure(MeasureKey key, String val) {
    rule = key.getRule();
    rulesCategoryId = key.getRulesCategoryId();
    this.metric = key.getMetric();

    if (key.getMetric().isDataType()) {
      measureData.add(new MeasureData(this, val));

    } else {
      this.textValue = val;
    }
  }

  public Measure() {
  }

  public Double getValue() {
    return value;
  }

  public Measure setValue(Double value) throws IllegalArgumentException {
    if (value != null && (value.isNaN() || value.isInfinite())) {
      throw new IllegalArgumentException();
    }
    this.value = value;
    return this;
  }

  public String getTextValue() {
    return textValue;
  }

  public Metric.Level getLevelValue() {
    if (textValue != null) {
      return Metric.Level.valueOf(textValue);
    }
    return null;
  }

  public Measure setTextValue(String textValue) {
    this.textValue = textValue;
    return this;
  }

  public Integer getTendency() {
    return tendency;
  }

  public Measure setTendency(Integer tendency) {
    this.tendency = tendency;
    return this;
  }

  public Metric getMetric() {
    return metric;
  }

  public void setMetric(Metric metric) {
    this.metric = metric;
  }

  public Integer getSnapshotId() {
    return snapshotId;
  }

  public Measure setSnapshotId(Integer snapshotId) {
    this.snapshotId = snapshotId;
    return this;
  }

  public Rule getRule() {
    return rule;
  }

  public Measure setRule(Rule rule) {
    this.rule = rule;
    return this;
  }

  public Integer getRulesCategoryId() {
    return rulesCategoryId;
  }

  public Measure setRulesCategoryId(Integer id) {
    this.rulesCategoryId = id;
    return this;
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public Integer getProjectId() {
    return projectId;
  }

  public void setProjectId(Integer projectId) {
    this.projectId = projectId;
  }

  public Date getMeasureDate() {
    return measureDate;
  }

  public Measure setMeasureDate(Date measureDate) {
    this.measureDate = measureDate;
    return this;
  }

  public Metric.Level getAlertStatus() {
    if (alertStatus == null) {
      return null;
    }
    return Metric.Level.valueOf(alertStatus);
  }

  public Measure setAlertStatus(Metric.Level level) {
    if (level != null) {
      this.alertStatus = level.toString();
    } else {
      this.alertStatus = null;
    }
    return this;
  }

  public String getData() {
    if (metric.isDataType()) {
      if (measureData != null && !measureData.isEmpty()) {
        return measureData.get(0).getText();
      }
    }
    return null;
  }

  public void setData(String data) {
    measureData.clear();
    measureData.add(new MeasureData(this, data));
  }

  public MeasureData getMeasureData() {
    if (measureData!=null && !measureData.isEmpty()) {
      return measureData.get(0);
    }
    return null;
  }

  public String getAlertText() {
    return alertText;
  }

  public void setAlertText(String alertText) {
    this.alertText = alertText;
  }

  @Override
  public String toString() {
    return new ToStringBuilder(this).
        append("value", value).
        append("metric", metric).
        toString();
  }

  public Integer getRuleId() {
    if (getRule() != null) {
      return getRule().getId();
    }
    return null;
  }

  public void copyFrom(Measure measure) {
    setAlertStatus(measure.getAlertStatus());
    setAlertText(measure.getAlertText());
    setMeasureDate(measure.getMeasureDate());
    setMetric(measure.getMetric());
    setProjectId(measure.getProjectId());
    setRule(measure.getRule());
    setRulesCategoryId(measure.getRulesCategoryId());
    setSnapshotId(measure.getSnapshotId());
    setTendency(measure.getTendency());
    setTextValue(measure.getTextValue());
    setValue(measure.getValue());
  }

  public void setMeasureData(MeasureData data) {
    if (measureData==null) {
      measureData = new ArrayList<MeasureData>();
    } else if (!measureData.isEmpty()) {
      measureData.clear();
    }

    this.measureData.add(data);
  }
}
