/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.resources;

import org.sonar.commons.Metric;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.sonar.commons.BaseDao;
import org.sonar.commons.database.DatabaseSession;

import javax.persistence.Query;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class MeasuresDao extends BaseDao {

  private Map<String, Metric> metricsByName = new HashMap<String, Metric>();

  public MeasuresDao(DatabaseSession session) {
    super(session);
  }

  public Metric getMetric(Metric metric) {
    return getMetricsByName().get(metric.getKey());
  }

  public Metric getMetric(String metricName) {
    return getMetricsByName().get(metricName);
  }

  public Collection<Metric> getMetrics() {
    return getMetricsByName().values();
  }

  public Collection<Metric> getEnabledMetrics() {
    return CollectionUtils.select(getMetricsByName().values(), new Predicate() {
      public boolean evaluate(Object o) {
        return ((Metric) o).getEnabled();
      }
    });
  }

  public Collection<Metric> getUserDefinedMetrics() {
    return CollectionUtils.select(getMetricsByName().values(), new Predicate() {
      public boolean evaluate(Object o) {
        Metric m = (Metric) o;
        return (m.getEnabled() && m.getOrigin() != Metric.Origin.JAV);
      }
    });
  }

  public void disableAutomaticMetrics() {
    getSession().createQuery("update " + Metric.class.getSimpleName() + " m set m.enabled=false where m.userManaged=false").executeUpdate();
    getSession().commit();
    metricsByName.clear();
  }

  public void registerMetrics(Collection<Metric> metrics) {
    if (metrics != null) {
      for (Metric metric : metrics) {
        metric.setEnabled(Boolean.TRUE);
        persistMetric(metric);
        getSession().commit();
      }
    }
  }

  public void persistMetric(Metric metric) {
    Metric dbMetric = getMetric(metric);
    if (dbMetric != null) {
      dbMetric.setDescription(metric.getDescription());
      dbMetric.setDirection(metric.getDirection());
      dbMetric.setDomain(metric.getDomain());
      dbMetric.setEnabled(metric.getEnabled());
      dbMetric.setQualitative(metric.getQualitative());
      dbMetric.setName(metric.getName());
      dbMetric.setType(metric.getType());
      dbMetric.setUserManaged(metric.getUserManaged());
      dbMetric.setOrigin(metric.getOrigin());
      getSession().getEntityManager().merge(dbMetric);

    } else {
      getSession().getEntityManager().persist(metric);
    }

    metricsByName.clear();
  }

  public void disabledMetrics(Collection<Metric> metrics) {
    for (Metric metric : metrics) {
      metric.setEnabled(Boolean.FALSE);
      getSession().getEntityManager().persist(metric);
      metricsByName.put(metric.getName(), metric);
    }
  }

  private Map<String, Metric> getMetricsByName() {
    if (metricsByName.isEmpty()) {
      List<Metric> metrics = getSession().getResults(Metric.class);
      for (Metric metric : metrics) {
        metricsByName.put(metric.getKey(), metric);
      }
    }
    return metricsByName;
  }

  public List<Measure> getMeasures(Integer snapshotId) {
    return getSession().getResults(Measure.class, "snapshotId", snapshotId);
  }

  public Snapshot getSnapshot(int snapshotId) {
    return getSession().getEntity(Snapshot.class, snapshotId);
  }

  public String getSnapshotProjectName(int snapshotId) {
    Query query = getSession().createQuery(
        "SELECT p.name FROM " + Resource.class.getSimpleName() + " p, Snapshot s WHERE s.resource=p AND s.id=:id");
    query.setParameter("id", snapshotId);
    return (String) query.getSingleResult();
  }

  public List<Snapshot> getChildSnapshots(Snapshot snapshot) {
    return getSession().getResults(Snapshot.class, "parent", snapshot);
  }
}
