/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.resources;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.hibernate.annotations.BatchSize;
import org.sonar.commons.rules.RulesProfile;

import javax.persistence.*;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@Entity
@Table(name = "projects")
public class Resource implements Serializable, Cloneable {

  public static final String SCOPE_PROJECT = "PRJ";
  public static final String SCOPE_DIRECTORY = "DIR";
  public static final String SCOPE_FILE = "FIL";

  public static final String QUALIFIER_PROJECT_TRUNK = "TRK";
  public static final String QUALIFIER_PROJECT_BRANCH = "BRC";
  public static final String QUALIFIER_PACKAGE = "PAC";
  public static final String QUALIFIER_DIRECTORY = "DIR";
  public static final String QUALIFIER_FILE = "FIL";
  public static final String QUALIFIER_CLASS = "CLA";
  public static final String QUALIFIER_UNIT_TEST = "UTS";

  private static final long serialVersionUID = -7747333646654989328L;

  public static final int DESCRIPTION_COLUMN_SIZE = 2000;
  public static final int NAME_COLUMN_SIZE = 256;
  public static final int KEY_SIZE = 230;

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "PROJECTS_SEQ", sequenceName = "PROJECTS_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "PROJECTS_SEQ")
  private Integer id;

  @Column(name = "name", updatable = true, nullable = true, length = NAME_COLUMN_SIZE)
  private String name;

  @Column(name = "description", updatable = true, nullable = true, length = DESCRIPTION_COLUMN_SIZE)
  private String description;

  @Column(name = "enabled", updatable = true, nullable = false)
  private Boolean enabled = Boolean.TRUE;

  @Column(name = "scope", updatable = true, nullable = false, length = 3)
  private String scope;

  @Column(name = "qualifier", updatable = true, nullable = false, length = 3)
  private String qualifier;

  @Column(name = "kee", updatable = false, nullable = false, length = KEY_SIZE)
  private String key;

  @Column(name = "language", updatable = true, nullable = true, length = 5)
  private String languageKey;

  @Column(name = "root_id", updatable = true, nullable = true)
  private Integer rootId;

  @OneToMany(mappedBy = "resource", fetch = FetchType.LAZY, cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REMOVE})
  @BatchSize(size = 8)
  private List<ProjectLink> projectLinks = new ArrayList<ProjectLink>();

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "profile_id", updatable = true, nullable = true)
  private RulesProfile rulesProfile;

  public Resource() {
  }

  public Resource(String scope, String key, String qualifier, Integer rootId, String name) {
    this.scope = scope;
    this.key = key;
    this.rootId = rootId;
    this.name = name;
    this.qualifier = qualifier;
  }

  public List<ProjectLink> getProjectLinks() {
    return projectLinks;
  }

  public void setProjectLinks(List<ProjectLink> projectLinks) {
    this.projectLinks = projectLinks;
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(String description) {
    this.description = StringUtils.abbreviate(description, DESCRIPTION_COLUMN_SIZE);
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = StringUtils.abbreviate(name, NAME_COLUMN_SIZE);
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public Boolean getEnabled() {
    return enabled;
  }

  public void setEnabled(Boolean enabled) {
    this.enabled = enabled;
  }

  public String getScope() {
    return scope;
  }

  public void setScope(String scope) {
    this.scope = scope;
  }

  public String getKey() {
    return key;
  }

  public String getLanguageKey() {
    return languageKey;
  }

  public void setLanguageKey(String lang) {
    this.languageKey = lang;
  }

  public void setKey(String key) {
    if (key.length() > KEY_SIZE) {
      throw new IllegalArgumentException("Resource key is too long, max is " + KEY_SIZE + " characters. Got : " + key);
    }
    this.key = key;
  }

  public Integer getRootId() {
    return rootId;
  }

  public void setRootId(Integer rootId) {
    this.rootId = rootId;
  }

  public RulesProfile getRulesProfile() {
    return rulesProfile;
  }

  public void setRulesProfile(RulesProfile rulesProfile) {
    this.rulesProfile = rulesProfile;
  }

  public String getQualifier() {
    return qualifier;
  }

  public void setQualifier(String qualifier) {
    this.qualifier = qualifier;
  }

  public ProjectLink getProjectLinkByType(String type) {
    if (type == null) {
      return null;
    }
    for (ProjectLink projectLink : projectLinks) {
      if (type.equals(projectLink.getType())) {
        return projectLink;
      }
    }
    return null;
  }

  public boolean isProject() {
    return SCOPE_PROJECT.equals(scope);
  }

  public boolean isTrunkProject() {
    return isProject() && QUALIFIER_PROJECT_TRUNK.equals(qualifier);
  }

  public boolean isBranchProject() {
    return isProject() && QUALIFIER_PROJECT_BRANCH.equals(qualifier);
  }

  public boolean isDirectory() {
    return SCOPE_DIRECTORY.equals(scope);
  }

  /**
   * A file can be a java class, a unit test class, a plsql file...
   */
  public boolean isFile() {
    return SCOPE_FILE.equals(scope);
  }

  public boolean isPackage() {
    return QUALIFIER_PACKAGE.equals(qualifier);
  }

  public boolean isClass() {
    return QUALIFIER_CLASS.equals(qualifier);
  }

  public boolean isUnitTest() {
    return QUALIFIER_UNIT_TEST.equals(qualifier);
  }

  public static Resource newDirectory(String key, String qualifier, String language) {
    Resource dir = new Resource();
    dir.setKey(key);
    dir.setName(key);
    dir.setLanguageKey(language);
    dir.setQualifier(qualifier);
    dir.setScope(SCOPE_DIRECTORY);
    return dir;
  }

  public static Resource newFile(String key, String qualifier, String language) {
    Resource file = new Resource();
    file.setKey(key);
    file.setName(key);
    file.setLanguageKey(language);
    file.setQualifier(qualifier);
    file.setScope(SCOPE_FILE);
    return file;
  }

  public static Resource newProject(String key, String language, boolean isTrunk) {
    Resource project = new Resource();
    project.setKey(key);
    project.setLanguageKey(language);
    project.setQualifier(isTrunk ? QUALIFIER_PROJECT_TRUNK : QUALIFIER_PROJECT_BRANCH);
    project.setScope(SCOPE_PROJECT);
    return project;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Resource)) {
      return false;
    }
    if (this == obj) {
      return true;
    }
    Resource other = (Resource) obj;
    return new EqualsBuilder()
        .append(key, other.key)
        .append(enabled, other.enabled)
        .append(rootId, other.rootId)
        .isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37)
        .append(key)
        .append(enabled)
        .append(rootId)
        .toHashCode();
  }

  @Override
  public String toString() {
    return new ToStringBuilder(this)
        .append("id", id)
        .append("key", key)
        .append("scope", scope)
        .append("qualifier", qualifier)
        .append("name", name)
        .append("lang", languageKey)
        .append("enabled", enabled)
        .append("rootId", rootId)
        .toString();
  }

  @Override
  public Object clone() {
    Resource clone = new Resource(getScope(), getKey(), getQualifier(), getRootId(), getName());
    clone.setDescription(getDescription());
    clone.setEnabled(getEnabled());
    clone.setProjectLinks(getProjectLinks());
    clone.setRulesProfile(getRulesProfile());
    clone.setLanguageKey(getLanguageKey());
    return clone;
  }
}