/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.rules;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

import javax.persistence.*;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@Entity
@Table(name = "active_rules")
public class ActiveRule implements Serializable, Cloneable {

  private static final long serialVersionUID = -6161808961786732538L;

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "ACTIVE_RULES_SEQ", sequenceName = "ACTIVE_RULES_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "ACTIVE_RULES_SEQ")
  private Integer id;

  @ManyToOne(fetch = FetchType.EAGER)
  @JoinColumn(name = "rule_id", updatable = true, nullable = false)
  private Rule rule;

  @Column(name = "failure_level", updatable = true, nullable = false)
  @Enumerated(EnumType.ORDINAL)
  private RuleFailureLevel level;

  @ManyToOne(fetch = FetchType.LAZY)
  @JoinColumn(name = "profile_id", updatable = true, nullable = false)
  private RulesProfile rulesProfile;

  @OneToMany(mappedBy = "activeRule", cascade = { CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REMOVE })
  private List<ActiveRuleParam> activeRuleParams = new ArrayList<ActiveRuleParam>();

  public ActiveRule() {
  }

  public ActiveRule(RulesProfile profile, Rule rule, RuleFailureLevel level) {
    this.rule = rule;
    this.level = level;
    this.rulesProfile = profile;
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public Rule getRule() {
    return rule;
  }

  public void setRule(Rule rule) {
    this.rule = rule;
  }

  public RuleFailureLevel getLevel() {
    return level;
  }

  public void setLevel(RuleFailureLevel level) {
    this.level = level;
  }

  public RulesProfile getRulesProfile() {
    return rulesProfile;
  }

  public void setRulesProfile(RulesProfile rulesProfile) {
    this.rulesProfile = rulesProfile;
  }

  public List<ActiveRuleParam> getActiveRuleParams() {
    return activeRuleParams;
  }

  public void setActiveRuleParams(List<ActiveRuleParam> params) {
    this.activeRuleParams = params;
  }

  public String getPluginName() {
    return rule.getPluginName();
  }

  public String getConfigKey() {
    return rule.getConfigKey();
  }

  public String getRuleKey() {
    return rule.getKey();
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof ActiveRule)) {
      return false;
    }
    if (this == obj) {
      return true;
    }
    ActiveRule other = (ActiveRule) obj;
    return new EqualsBuilder().append(id, other.getId()).isEquals();
  }

  @Override
  public int hashCode() {
    return id;
  }

  @Override
  public String toString() {
    return new ToStringBuilder(this).append("id", id).append("rule", rule).append("level", level).append("params", activeRuleParams).toString();
  }

  @Override
  public Object clone() {
    ActiveRule clone = new ActiveRule(getRulesProfile(), getRule(), getLevel());
    if (CollectionUtils.isNotEmpty(getActiveRuleParams())) {
      clone.setActiveRuleParams(new ArrayList<ActiveRuleParam>(CollectionUtils.collect(getActiveRuleParams(), new Transformer() {
        public Object transform(Object input) {
          return ((ActiveRuleParam) input).clone();
        }
      })));
    }
    return clone;
  }

}
