/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.rules;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.sonar.commons.database.DatabaseProperties;

import javax.persistence.*;
import java.util.ArrayList;
import java.util.List;

@Cache(usage = CacheConcurrencyStrategy.READ_WRITE)
@Entity
@Table(name = "rules")
public class Rule {

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "RULES_SEQ", sequenceName = "RULES_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "RULES_SEQ")
  private Integer id;

  @Column(name = "name", updatable = true, nullable = false)
  private String name;

  @Column(name = "plugin_rule_key", updatable = false, nullable = true)
  private String key;

  @Column(name = "plugin_config_key", updatable = true, nullable = true)
  private String configKey;

  @ManyToOne(fetch = FetchType.EAGER)
  @JoinColumn(name = "rules_category_id", updatable = true, nullable = true)
  private RulesCategory rulesCategory;

  @Column(name = "description", updatable = true, nullable = true, length = DatabaseProperties.MAX_TEXT_SIZE)
  private String description;

  @Column(name = "plugin_name", updatable = true, nullable = true)
  private String pluginName;

  @OneToMany(mappedBy = "rule", fetch = FetchType.LAZY, cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REMOVE})
  private List<RuleParam> params = new ArrayList<RuleParam>();

  public Rule() {
  }

  public Rule(String name, String key, RulesCategory rulesCategory, String pluginName, String description) {
    this();
    setName(name);
    this.key = key;
    this.rulesCategory = rulesCategory;
    this.pluginName = pluginName;
    this.description = description;
  }

  public Rule(String name, String key, String configKey, RulesCategory rulesCategory, String pluginName, String description) {
    this();
    setName(name);
    this.key = key;
    this.configKey = configKey;
    this.rulesCategory = rulesCategory;
    this.pluginName = pluginName;
    this.description = description;
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = removeNewLineCharacters(name);
  }

  public String getKey() {
    return key;
  }

  public void setKey(String key) {
    this.key = key;
  }

  public RulesCategory getRulesCategory() {
    return rulesCategory;
  }

  public void setRulesCategory(RulesCategory rulesCategory) {
    this.rulesCategory = rulesCategory;
  }

  public String getPluginName() {
    return pluginName;
  }

  public void setPluginName(String pluginName) {
    this.pluginName = pluginName;
  }

  public String getConfigKey() {
    return configKey;
  }

  public void setConfigKey(String configKey) {
    this.configKey = configKey;
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(String description) {
    this.description = StringUtils.strip(description);
  }

  public List<RuleParam> getParams() {
    return params;
  }

  public void setParams(List<RuleParam> params) {
    this.params = params;
  }

  public Integer getCategoryId() {
    if (rulesCategory!=null) {
      return rulesCategory.getId();
    }
    return null;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Rule)) {
      return false;
    }
    if (this == obj) {
      return true;
    }
    Rule other = (Rule) obj;
    return new EqualsBuilder()
        .append(pluginName, other.getPluginName())
        .append(key, other.getKey())
        .isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37).
        append(pluginName).append(key).toHashCode();
  }

  @Override
  public String toString() {
    return new ToStringBuilder(this)
        .append("id", id)
        .append("name", name)
        .append("key", key)
        .append("configKey", configKey)
        .append("categ", rulesCategory)
        .append("plugin", pluginName)
        .append("params", params)
        .toString();
  }

  protected String removeNewLineCharacters(String text){
    text = StringUtils.remove(text, "\n");
    text = StringUtils.remove(text, "\r");
    text = StringUtils.remove(text, "\n\r");
    text = StringUtils.remove(text, "\r\n");
    return text;
  }

}
