/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.rules;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ReflectionToStringBuilder;
import org.sonar.commons.resources.Snapshot;

import javax.persistence.*;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@Entity
@Table(name = "rule_failures")
public class RuleFailure implements Serializable {

  private static final long serialVersionUID = 6573631005994829171L;

  public static final int MESSAGE_COLUMN_SIZE = 500;

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "RULE_FAILURES_SEQ", sequenceName = "RULE_FAILURES_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "RULE_FAILURES_SEQ")
  private Integer id;

  @Column(name = "snapshot_id")
  protected Integer snapshotId;

  @ManyToOne(fetch = FetchType.EAGER)
  @JoinColumn(name = "rule_id")
  private Rule rule;

  @OneToMany(mappedBy = "ruleFailure", fetch = FetchType.LAZY, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REMOVE})
  private List<RuleFailureParam> params;

  @Column(name = "failure_level", updatable = false, nullable = false)
  @Enumerated(EnumType.ORDINAL)
  private RuleFailureLevel failureLevel;

  @Column(name = "message", updatable = false, nullable = true, length = MESSAGE_COLUMN_SIZE)
  private String message;

  public RuleFailure() {
  }

  public RuleFailure(Rule rule, RuleFailureLevel level) {
    this.rule = rule;
    this.failureLevel = level;
    this.params = new ArrayList<RuleFailureParam>();
  }

  public RuleFailure(Rule rule, RuleFailureLevel level, Snapshot snapshot) {
    this.rule = rule;
    this.failureLevel = level;
    this.params = new ArrayList<RuleFailureParam>();
    this.snapshotId = snapshot.getId();
  }

  public String getMessage() {
    return message;
  }

  public void setMessage(String message) {
    String mes = StringUtils.trim(message);

    // SONAR-475 strange error on oracle
    this.message = StringUtils.abbreviate(mes, MESSAGE_COLUMN_SIZE - 1);
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public void setSnapshot(Snapshot snapshot) {
    this.snapshotId = snapshot.getId();
    for (RuleFailureParam param : params) {
      param.setSnapshot(snapshot);
    }
  }

  public RuleFailureLevel getLevel() {
    return failureLevel;
  }

  public void setLevel(RuleFailureLevel level) {
    this.failureLevel = level;
  }

  public Rule getRule() {
    return rule;
  }

  public void setRule(Rule rule) {
    this.rule = rule;
  }

  public List<RuleFailureParam> getParameters() {
    return params;
  }

  public void setParameters(List<RuleFailureParam> ruleFailureParams) {
    this.params = ruleFailureParams;
  }

  public RuleFailureParam addParameter(String key, Double value) {
    return addParameter(key, value, null);
  }

  public RuleFailureParam addParameter(String key, Double value, Double value2) {
    if (params == null) {
      params = new ArrayList<RuleFailureParam>();
    }
    RuleFailureParam param = new RuleFailureParam(key, value, value2);
    param.setRuleFailure(this);
    params.add(param);
    return param;
  }

  public RuleFailureParam getParameter(String key) {
    for (RuleFailureParam param : params) {
      if (param!=null && param.getKey().equals(key)) {
        return param;
      }
    }
    return null;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof RuleFailure)) {
      return false;
    }
    if (this == obj) {
      return true;
    }
    RuleFailure other = (RuleFailure) obj;
    return new EqualsBuilder()
        .append(id, other.getId()).isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37).
        append(id).toHashCode();
  }

  @Override
  public String toString() {
    return ReflectionToStringBuilder.toString(this);
  }
}
