/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.commons.rules;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.sonar.commons.alerts.Alert;
import org.sonar.commons.resources.Resource;

import javax.persistence.*;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@Entity
@Table(name = "rules_profiles")
public class RulesProfile implements Serializable, Cloneable {

  private static final long serialVersionUID = 4024876470769619760L;

  public static final String SONAR_WAY_NAME = "Sonar way";
  public static final String SUN_CONVENTIONS_NAME = "Sun checks";

  @Id
  @Column(name = "id")
  @SequenceGenerator(name = "RULES_PROFILES_SEQ", sequenceName = "RULES_PROFILES_SEQ")
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "RULES_PROFILES_SEQ")
  private Integer id;

  @Column(name = "name", updatable = true, nullable = false)
  private String name;

  @Column(name = "default_profile", updatable = true, nullable = false)
  private Boolean defaultProfile = Boolean.FALSE;

  @Column(name = "provided", updatable = true, nullable = false)
  private Boolean provided = Boolean.FALSE;

  @Column(name = "language", updatable = true, nullable = false)
  private String language;

  @OneToMany(mappedBy = "rulesProfile", fetch = FetchType.LAZY, cascade = { CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REMOVE })
  private List<ActiveRule> activeRules;

  @OneToMany(mappedBy = "rulesProfile", fetch = FetchType.LAZY, cascade = { CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REMOVE })
  private List<Alert> alerts;
  
  @OneToMany(mappedBy = "rulesProfile", fetch = FetchType.LAZY)
  private List<Resource> projects;

  public RulesProfile() {
  }

  public RulesProfile(String name) {
    this.name = name;
    this.activeRules = new ArrayList<ActiveRule>();
    this.alerts = new ArrayList<Alert>();
    this.projects = new ArrayList<Resource>();
  }

  public RulesProfile(String name, String language, boolean defaultProfile, boolean provided) {
    this(name);
    this.language = language;
    this.defaultProfile = defaultProfile;
    this.provided = provided;
  }

  public Integer getId() {
    return id;
  }

  public void setId(Integer id) {
    this.id = id;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  public List<ActiveRule> getActiveRules() {
    return activeRules;
  }

  public void setActiveRules(List<ActiveRule> activeRules) {
    this.activeRules = activeRules;
  }

  public Boolean getDefaultProfile() {
    return defaultProfile;
  }

  public void setDefaultProfile(Boolean defaultProfile) {
    this.defaultProfile = defaultProfile;
  }

  public Boolean getProvided() {
    return provided;
  }

  public void setProvided(Boolean provided) {
    this.provided = provided;
  }

  public String getLanguage() {
    return language;
  }

  public void setLanguage(String language) {
    this.language = language;
  }

  public List<Alert> getAlerts() {
    return alerts;
  }

  public void setAlerts(List<Alert> alerts) {
    this.alerts = alerts;
  }
  
  public List<Resource> getProjects() {
    return projects;
  }

  public void setProjects(List<Resource> projects) {
    this.projects = projects;
  }

  public List<ActiveRule> getActiveRules(RuleFailureLevel level) {
    List<ActiveRule> result = new ArrayList<ActiveRule>();
    for (ActiveRule activeRule : getActiveRules()) {
      if (activeRule.getLevel().equals(level)) {
        result.add(activeRule);
      }
    }
    return result;
  }

  public List<ActiveRule> getActiveRulesByPlugin(String pluginKey) {
    List<ActiveRule> result = new ArrayList<ActiveRule>();
    for (ActiveRule activeRule : getActiveRules()) {
      if (pluginKey.equals(activeRule.getPluginName())) {
        result.add(activeRule);
      }
    }
    return result;
  }

  public ActiveRule getActiveRule(String pluginKey, String ruleKey) {
    for (ActiveRule activeRule : getActiveRules()) {
      if (activeRule.getRuleKey().equals(ruleKey) && activeRule.getPluginName().equals(pluginKey)) {
        return activeRule;
      }
    }
    return null;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof RulesProfile)) {
      return false;
    }
    if (this == obj) {
      return true;
    }
    RulesProfile other = (RulesProfile) obj;
    return new EqualsBuilder().append(language, other.getLanguage()).append(name, other.getName()).isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37).append(language).append(name).toHashCode();
  }

  @Override
  public Object clone() {
    RulesProfile clone = new RulesProfile(getName(), getLanguage(), getDefaultProfile(), getProvided());
    if (CollectionUtils.isNotEmpty(getActiveRules())) {
      clone.setActiveRules(new ArrayList<ActiveRule>(CollectionUtils.collect(getActiveRules(), new Transformer() {
        public Object transform(Object input) {
          return ((ActiveRule) input).clone();
        }
      })));
    }
    if (CollectionUtils.isNotEmpty(getAlerts())) {
      clone.setAlerts(new ArrayList<Alert>(CollectionUtils.collect(getAlerts(), new Transformer() {
        public Object transform(Object input) {
          return ((Alert) input).clone();
        }
      })));
    }
    if (CollectionUtils.isNotEmpty(getProjects())) {
      clone.setProjects(new ArrayList<Resource>(CollectionUtils.collect(getProjects(), new Transformer() {
        public Object transform(Object input) {
          return ((Resource) input).clone();
        }
      })));
    }
    return clone;
  }

  @Override
  public String toString() {
    return new StringBuilder().append(name).append(", language=").append(language).toString();
  }
}
