/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.api;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.commons.io.IOUtils;
import org.sonar.squid.graph.Edge;
import org.sonar.squid.graph.EdgeUsage;
import org.sonar.squid.graph.Node;
import org.sonar.squid.measures.Measurable;
import org.sonar.squid.measures.Metric;

public abstract class SourceCode extends TreeNode<SourceCode> implements Node, Measurable, Comparable<SourceCode> {

  private final String        name;
  private Map<Metric, Double> measures     = new HashMap<Metric, Double>();
  private final String        key;
  private Map<Node, Edge>     dependedUpon = new HashMap<Node, Edge>();
  private Map<Node, Edge>     dependsUpon  = new HashMap<Node, Edge>();
  private int                 startAtLine  = -1;
  private int                 endAtLine    = -1;

  public SourceCode(String key) {
    this(key, null);
  }

  public SourceCode(String key, String name) {
    this.name = name;
    this.key = key;
    initializeMeasures();
  }

  public String getKey() {
    return key;
  }

  public int compareTo(SourceCode resource) {
    return key.compareTo(resource.getKey());
  }

  public String getName() {
    return name;
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof SourceCode)) {
      return false;
    }
    return getKey().equals(((SourceCode) obj).getKey());
  }

  @Override
  public int hashCode() {
    return key.hashCode();
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder(512);
    sb.append(this.getClass().getSimpleName()).append(" : ").append(getKey()).append(IOUtils.LINE_SEPARATOR);
    for (SourceCode child : getChildren()) {
      String childTree = child.toString();
      StringTokenizer tokenizer = new StringTokenizer(childTree, IOUtils.LINE_SEPARATOR);
      while (tokenizer.hasMoreTokens()) {
        sb.append("-").append(tokenizer.nextToken()).append(IOUtils.LINE_SEPARATOR);
      }
    }
    return sb.toString();
  }

  public final void computeMeasures() {
    for (SourceCode child : getChildren()) {
      child.computeMeasures();
    }
    for (Metric metric : Metric.values()) {
      if (!metric.aggregateIfThereIsAlreadyAValue() && getDouble(metric) != 0) {
        continue;
      }
      for (SourceCode child : getChildren()) {
        if (!metric.isCalculatedMetric() && metric.isThereAggregationFormula()) {
          add(metric, child);
        }
      }

    }
    updateMeasuresAfterConsolidation();
  }

  public void createEdgeWith(Node to, EdgeUsage link) {
    createEdgeWith(to, link, (Edge) null);
  }

  public void createEdgeWith(Node to, EdgeUsage link, Edge rootEdge) {
    if (equals(to)) {
      throw new IllegalStateException("You can't create an edge from one node to itself. SquidUnit : " + getKey());
    }
    if (getEdgeTo(to) != null) {
      throw new IllegalStateException("Those two nodes are already linked. From : " + getKey() + ", To : " + to.getKey());
    }
    Edge edge = new Edge(this, to, link, null);
    edge.addRootEdge(rootEdge);

    dependsUpon.put(to, edge);
    ((SourceCode) to).dependedUpon.put(this, edge);
  }

  public Edge getEdgeTo(Node to) {
    return dependsUpon.get(to);
  }

  public Edge getEdgeFrom(Node from) {
    return dependedUpon.get(from);
  }

  public EdgeUsage getUsageOf(Node to) {
    Edge relation = dependsUpon.get(to);
    if (relation == null) {
      return EdgeUsage.NO_LINK;
    }
    return relation.getUsage();
  }

  public EdgeUsage getUsageBy(Node from) {
    Edge edge = dependedUpon.get(from);
    if (edge == null) {
      return EdgeUsage.NO_LINK;
    }
    return edge.getUsage();
  }

  public Set<Node> dependedUpon() {
    return dependedUpon.keySet();
  }

  public Set<Node> dependsUpon() {
    return dependsUpon.keySet();
  }

  public boolean isType(Class<? extends SourceCode> resourceType) {
    return this.getClass() == resourceType;
  }

  public int getInt(Metric metric) {
    return getMeasure(metric).intValue();
  }

  public double getDouble(Metric metric) {
    return getMeasure(metric).doubleValue();
  }

  public void add(Metric metric, SourceCode child) {
    add(metric, child.getMeasure(metric));
  }

  public void add(Metric metric, double value) {
    setMeasure(metric, getMeasure(metric) + value);
  }

  private Double getMeasure(Metric metric) {
    if (metric.isCalculatedMetric()) {
      return metric.getCalculatedMetricFormula().calculate(this);
    }
    if (measures.get(metric) == null) {
      return (double) 0;
    }
    return measures.get(metric);
  }

  public void setMeasure(Metric metric, double measure) {
    if (metric.isCalculatedMetric()) {
      throw new IllegalStateException("It's not allowed to set the value of a calculated metric : " + metric.name());
    }
    measures.put(metric, measure);
  }

  public void setMeasure(Metric metric, int measure) {
    setMeasure(metric, (double) measure);
  }

  protected abstract void initializeMeasures();

  protected abstract void updateMeasuresAfterConsolidation();

  public void setStartAtLine(int startAtLine) {
    this.startAtLine = startAtLine;
    this.endAtLine = startAtLine;
  }

  public void setEndAtLine(int endAtLine) {
    this.endAtLine = endAtLine;
  }

  public int getStartAtLine() {
    return startAtLine;
  }

  public int getEndAtLine() {
    return endAtLine;
  }
}
