/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.text;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang.StringUtils;
import org.sonar.squid.api.AnalysisException;
import org.sonar.squid.api.CodeScanner;
import org.sonar.squid.api.CodeVisitor;
import org.sonar.squid.api.SquidConfiguration;
import org.sonar.squid.api.SourceFile;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.measures.Metric;

public abstract class AbstractTextScanner implements CodeScanner {

  private static final char  TAB_CHAR = '\u0009';
  private SquidConfiguration conf;

  public AbstractTextScanner(SquidConfiguration conf) {
    this.conf = conf;
  }

  public void scanCode(SourceCode project, Collection<File> filesToAnalyse) {
    for (File textFile : filesToAnalyse) {
      project.addChild(analyseFile(textFile));
    }
  }

  protected SourceCode analyseFile(File fileToAnalyse) {
    LineIterator lineIterator = null;
    try {
      SourceFile resource = new SourceFile(fileToAnalyse.getName());
      AnalysisState state = new AnalysisState();
      lineIterator = FileUtils.lineIterator(fileToAnalyse, conf.getCharset().name());
      while (lineIterator.hasNext()) {
        analyseLine(lineIterator.nextLine(), state);
      }
      resource.setMeasure(Metric.LINES, state.lines);
      resource.setMeasure(Metric.LINES_OF_CODE, state.loc);
      resource.setMeasure(Metric.BLANK_LINES, state.blankLines);
      resource.setMeasure(Metric.COMMENT_LINES, state.commentLines);
      return resource;
    } catch (FileNotFoundException e) {
      throw new AnalysisException("Unable to read file : " + fileToAnalyse.getName(), e);
    } catch (IOException e) {
      throw new AnalysisException("En error occured when analysing file : " + fileToAnalyse.getName(), e);
    } finally {
      LineIterator.closeQuietly(lineIterator);
    }
  }

  private void analyseLine(String line, AnalysisState state) {
    state.newLine();
    state.stopSingleLineComment();
    StringBuilder lineWithoutComment = new StringBuilder();
    if (StringUtils.isBlank(line)) {
      state.incrementBlankLines();
      return;
    }
    for (int i = 0; i < line.length(); i++) {
      if (!state.multiLinesCommentStarted && multiLinesCommentStart(line, i)) {
        state.startMultiLinesComment();
      } else if (state.multiLinesCommentStarted && multiLinesCommentStop(line, i)) {
        state.stopMultiLinesComment();
        i = i + 2;
      } else if (singleLinesCommentStart(line, i)) {
        state.startSingleLineComment();
        state.stopMultiLinesComment();
        break;
      } else if (!state.multiLinesCommentStarted && !state.singleLinesCommentStarted && !isBlankCharacter(line.charAt(i))) {
        lineWithoutComment.append(line.charAt(i));
      }
    }
    if (lineWithoutComment.toString().trim().length() > 0) {
      state.incrementLoc();
    }
  }

  private boolean isBlankCharacter(char character) {
    return Character.isWhitespace(character) || character == TAB_CHAR;
  }

  private boolean substringEquals(String line, int index, String substring) {
    if (line.length() < index + substring.length()) {
      return false;
    }
    return line.substring(index, index + substring.length()).equals(substring);
  }

  private boolean singleLinesCommentStart(String line, int i) {
    for (SingleLineCommentDelimiter singleLines : getSingleLineCommentDelimiter()) {
      String startWith = singleLines.getStartWith();
      if (substringEquals(line, i, startWith)) {
        return true;
      }
    }
    return false;
  }

  private boolean multiLinesCommentStart(String line, int i) {
    for (MultiLinesCommentDelimiters multiLines : getMultiLinesCommentDelimiters()) {
      String startWith = multiLines.getStartWith();
      if (substringEquals(line, i, startWith)) {
        return true;
      }
    }
    return false;
  }

  private boolean multiLinesCommentStop(String line, int i) {
    for (MultiLinesCommentDelimiters multiLines : getMultiLinesCommentDelimiters()) {
      String endWith = multiLines.getEndWith();
      if (substringEquals(line, i, endWith)) {
        return true;
      }
    }
    return false;
  }

  protected final char chartAt(String line, int index) {
    if (index < line.length()) {
      return line.charAt(index);
    }
    return ' ';
  }

  protected abstract List<MultiLinesCommentDelimiters> getMultiLinesCommentDelimiters();

  protected abstract List<SingleLineCommentDelimiter> getSingleLineCommentDelimiter();

  protected class AnalysisState {

    int     lines                     = 0;
    int     loc                       = 0;
    int     blankLines                = 0;
    int     commentLines              = 0;
    boolean multiLinesCommentStarted  = false;
    boolean singleLinesCommentStarted = false;

    public void newLine() {
      lines++;
      if (multiLinesCommentStarted) {
        commentLines++;
      }
    }

    public void incrementLoc() {
      loc++;
    }

    public void startSingleLineComment() {
      commentLines++;
      singleLinesCommentStarted = true;
    }

    public void startMultiLinesComment() {
      commentLines++;
      multiLinesCommentStarted = true;
    }

    public void stopMultiLinesComment() {
      multiLinesCommentStarted = false;
    }

    public void stopSingleLineComment() {
      singleLinesCommentStarted = false;
    }

    public void incrementBlankLines() {
      blankLines++;
    }
  }
  protected class MultiLinesCommentDelimiters {

    private final String startWith;
    private final String endWith;

    protected MultiLinesCommentDelimiters(String startWith, String endWith) {
      this.startWith = startWith;
      this.endWith = endWith;
    }

    protected String getStartWith() {
      return startWith;
    }

    protected String getEndWith() {
      return endWith;
    }
  }
  protected class SingleLineCommentDelimiter {

    private final String startWith;

    protected SingleLineCommentDelimiter(String startWith) {
      this.startWith = startWith;
    }

    protected String getStartWith() {
      return startWith;
    }
  }

  public void accept(CodeVisitor visitor) {
  }

  public Collection<Class<? extends CodeVisitor>> getVisitors() {
    return new ArrayList<Class<? extends CodeVisitor>>();
  }
}
